# fiberweave/snapping_config.py
"""
Snapping Configuration for FiberWeave
Automatically configures snapping when layers are loaded
"""

from qgis.core import (
    QgsProject, QgsSnappingConfig, QgsTolerance,
    QgsMessageLog, Qgis
)


class SnappingManager:
    """Manages snapping configuration for FiberWeave layers"""
    
    @staticmethod
    def configure_snapping(tolerance=10, enable_topological=True):
        """Configure snapping for all FiberWeave layers
        
        Args:
            tolerance (int): Snapping tolerance in pixels
            enable_topological (bool): Enable topological editing
        """
        try:
            project = QgsProject.instance()
            snapping_config = project.snappingConfig()
            
            # Enable snapping
            snapping_config.setEnabled(True)
            
            # Set snapping mode to all layers
            snapping_config.setMode(QgsSnappingConfig.AllLayers)
            
            # Set snapping type to vertex and segment
            snapping_config.setType(QgsSnappingConfig.VertexAndSegment)
            
            # Set tolerance
            snapping_config.setTolerance(tolerance)
            snapping_config.setUnits(QgsTolerance.Pixels)
            
            # Enable topological editing (shared nodes)
            if enable_topological:
                snapping_config.setIntersectionSnapping(True)
                project.setTopologicalEditing(True)
            
            # Apply configuration
            project.setSnappingConfig(snapping_config)
            
            QgsMessageLog.logMessage(
                f'✓ Snapping configured: tolerance={tolerance}px, topological={enable_topological}',
                'FiberWeave',
                Qgis.Success
            )
            
            return True
            
        except Exception as e:
            QgsMessageLog.logMessage(
                f'✗ Error configuring snapping: {str(e)}',
                'FiberWeave',
                Qgis.Critical
            )
            return False
    
    @staticmethod
    def configure_layer_snapping(layer_name, tolerance=10, snap_type=None):
        """Configure snapping for a specific layer
        
        Args:
            layer_name (str): Name of the layer
            tolerance (int): Snapping tolerance
            snap_type (QgsSnappingConfig.SnappingType): Type of snapping
        """
        try:
            project = QgsProject.instance()
            snapping_config = project.snappingConfig()
            
            # Find layer
            layers = project.mapLayersByName(layer_name)
            if not layers:
                return False
            
            layer = layers[0]
            
            # Configure individual layer settings
            layer_config = snapping_config.individualLayerSettings(layer)
            layer_config.setEnabled(True)
            layer_config.setTolerance(tolerance)
            layer_config.setUnits(QgsTolerance.Pixels)
            
            if snap_type:
                layer_config.setType(snap_type)
            else:
                layer_config.setType(QgsSnappingConfig.VertexAndSegment)
            
            snapping_config.setIndividualLayerSettings(layer, layer_config)
            project.setSnappingConfig(snapping_config)
            
            return True
            
        except Exception as e:
            QgsMessageLog.logMessage(
                f'✗ Error configuring layer snapping: {str(e)}',
                'FiberWeave',
                Qgis.Critical
            )
            return False
    
    @staticmethod
    def get_snapping_info():
        """Get current snapping configuration info"""
        project = QgsProject.instance()
        snapping_config = project.snappingConfig()
        
        info = {
            'enabled': snapping_config.enabled(),
            'mode': snapping_config.mode(),
            'type': snapping_config.type(),
            'tolerance': snapping_config.tolerance(),
            'units': snapping_config.units(),
            'topological': project.topologicalEditing(),
            'intersection': snapping_config.intersectionSnapping()
        }
        
        return info