# fiberweave/sample_data_generator.py
"""
Sample Network Data Generator for FiberWeave
Creates realistic test data for demonstration and testing
"""

import psycopg2
from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QPushButton, 
    QTextEdit, QLabel, QProgressBar, QMessageBox, QGroupBox,
    QSpinBox, QFormLayout
)
from qgis.PyQt.QtCore import Qt
from qgis.core import QgsMessageLog, Qgis


class SampleDataGenerator(QDialog):
    """Dialog for generating sample network data"""
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('FiberWeave Sample Data Generator')
        self.setMinimumWidth(600)
        self.setMinimumHeight(400)
        
        self.init_ui()
    
    def init_ui(self):
        """Initialize UI"""
        layout = QVBoxLayout()
        
        # Title
        title = QLabel('<h2>🎲 Generate Sample Network Data</h2>')
        title.setAlignment(Qt.AlignCenter)
        layout.addWidget(title)
        
        # Description
        desc = QLabel(
            '<p>This will create a realistic fiber network with:</p>'
            '<ul>'
            '<li>OLT (Central Office)</li>'
            '<li>Fiber cable routes</li>'
            '<li>Splitter nodes (1:8 and 1:32)</li>'
            '<li>ONUs (Customer premises)</li>'
            '<li>Splice points</li>'
            '<li>Complete network paths</li>'
            '</ul>'
            '<p><b>Warning:</b> This will add data to your database.</p>'
        )
        layout.addWidget(desc)
        
        # Configuration
        config_group = QGroupBox('Network Configuration')
        config_layout = QFormLayout()
        
        self.onu_count_spin = QSpinBox()
        self.onu_count_spin.setMinimum(10)
        self.onu_count_spin.setMaximum(500)
        self.onu_count_spin.setValue(50)
        
        config_layout.addRow('Number of ONUs:', self.onu_count_spin)
        config_group.setLayout(config_layout)
        layout.addWidget(config_group)
        
        # Progress bar
        self.progress = QProgressBar()
        self.progress.setVisible(False)
        layout.addWidget(self.progress)
        
        # Log output
        log_label = QLabel('<b>Generation Log:</b>')
        layout.addWidget(log_label)
        
        self.log_output = QTextEdit()
        self.log_output.setReadOnly(True)
        self.log_output.setMaximumHeight(150)
        layout.addWidget(self.log_output)
        
        # Buttons
        button_layout = QHBoxLayout()
        
        self.generate_btn = QPushButton('🎲 Generate Network')
        self.generate_btn.clicked.connect(self.generate_data)
        
        self.close_btn = QPushButton('✖ Close')
        self.close_btn.clicked.connect(self.reject)
        
        button_layout.addWidget(self.generate_btn)
        button_layout.addStretch()
        button_layout.addWidget(self.close_btn)
        
        layout.addLayout(button_layout)
        self.setLayout(layout)
    
    def log(self, message, level='info'):
        """Add message to log"""
        colors = {
            'info': 'black',
            'success': 'green',
            'warning': 'orange',
            'error': 'red'
        }
        color = colors.get(level, 'black')
        self.log_output.append(f'<span style="color: {color};">{message}</span>')
        QgsMessageLog.logMessage(message, 'FiberWeave Data Generator', Qgis.Info)
    
    def generate_data(self):
        """Generate sample network data"""
        
        if not self.plugin.db_connection:
            QMessageBox.warning(
                self,
                'No Connection',
                'Please connect to database first using 🔌 Connect DB'
            )
            return
        
        reply = QMessageBox.question(
            self,
            'Confirm Generation',
            'This will add sample data to your database.\n\n'
            'Continue?',
            QMessageBox.Yes | QMessageBox.No
        )
        
        if reply == QMessageBox.No:
            return
        
        self.progress.setVisible(True)
        self.progress.setMaximum(10)
        self.generate_btn.setEnabled(False)
        
        self.log('=' * 50, 'info')
        self.log('Starting sample data generation...', 'info')
        self.log('=' * 50, 'info')
        
        try:
            conn = self.plugin.db_connection
            cur = conn.cursor()
            
            onu_count = self.onu_count_spin.value()
            
            # Step 1: Create OLT
            self.progress.setValue(1)
            self.log('Step 1/10: Creating OLT equipment...', 'info')
            
            cur.execute("""
                INSERT INTO fttx.itu_olt_equipment (
                    olt_code, pon_standard, pon_class,
                    downstream_wavelength_nm, upstream_wavelength_nm,
                    downstream_rate_mbps, upstream_rate_mbps,
                    mean_launched_power_min_dbm, mean_launched_power_max_dbm,
                    receiver_sensitivity_dbm, max_logical_links, max_onus_supported,
                    equipment_status, manufacturer, model,
                    geom
                ) VALUES (
                    'OLT-CENTRAL-001', 'GPON', 'B+',
                    1490, 1310,
                    2488, 1244,
                    1.5, 5.0,
                    -28.0, 128, 512,
                    'active', 'Huawei', 'MA5800-X15',
                    ST_SetSRID(ST_MakePoint(28.0473, -26.2041), 4326)
                ) RETURNING olt_id;
            """)
            olt_id = cur.fetchone()[0]
            self.log(f'✓ OLT created (ID: {olt_id})', 'success')
            
            # Step 2: Create PON ports
            self.progress.setValue(2)
            self.log('Step 2/10: Creating PON ports...', 'info')
            
            pon_port_ids = []
            for i in range(4):  # 4 PON ports
                cur.execute("""
                    INSERT INTO fttx.itu_olt_pon_ports (
                        olt_id, port_number, port_label,
                        optical_budget_class, max_logical_reach_km,
                        max_onus, port_status
                    ) VALUES (
                        %s, %s, %s, 'B+', 20, 64, 'active'
                    ) RETURNING pon_port_id;
                """, (olt_id, i+1, f'PON-0/{i+1}'))
                pon_port_ids.append(cur.fetchone()[0])
            
            self.log(f'✓ Created {len(pon_port_ids)} PON ports', 'success')
            
            # Step 3: Create cable routes
            self.progress.setValue(3)
            self.log('Step 3/10: Creating cable routes...', 'info')
            
            # Main route from OLT
            cur.execute("""
                INSERT INTO fttx.itu_cable_routes (
                    route_code, route_name, infrastructure_type,
                    duct_specification, duct_count, route_length_m,
                    route_status,
                    geom
                ) VALUES (
                    'ROUTE-001', 'Main Distribution Route', 'underground',
                    '40mm HDPE', 4, 2500.0,
                    'active',
                    ST_SetSRID(ST_MakeLine(
                        ST_MakePoint(28.0473, -26.2041),
                        ST_MakePoint(28.0550, -26.2100)
                    ), 4326)
                ) RETURNING route_id;
            """)
            route_id = cur.fetchone()[0]
            self.log(f'✓ Cable route created (ID: {route_id})', 'success')
            
            # Step 4: Create fiber cables
            self.progress.setValue(4)
            self.log('Step 4/10: Creating fiber cables...', 'info')
            
            cur.execute("""
                INSERT INTO fttx.itu_fiber_cables (
                    cable_code, cable_category, cable_type, fiber_type,
                    fiber_count, cable_construction, sheath_material,
                    max_attenuation_db_per_km, installation_method,
                    operational_status, route_id, cable_length_m,
                    geom
                ) VALUES (
                    'FB-DIST-001', 'distribution', 'loose_tube', 'G.652D',
                    48, 'loose_tube', 'HDPE',
                    0.35, 'underground',
                    'active', %s, 2500.0,
                    ST_SetSRID(ST_MakeLine(
                        ST_MakePoint(28.0473, -26.2041),
                        ST_MakePoint(28.0550, -26.2100)
                    ), 4326)
                ) RETURNING cable_id;
            """, (route_id,))
            cable_id = cur.fetchone()[0]
            self.log(f'✓ Fiber cable created (ID: {cable_id})', 'success')
            
            # Step 5: Create fiber strands
            self.progress.setValue(5)
            self.log('Step 5/10: Creating fiber strands...', 'info')
            
            fiber_colors = ['blue', 'orange', 'green', 'brown', 'slate', 'white', 
                           'red', 'black', 'yellow', 'violet', 'rose', 'aqua']
            
            for i in range(48):
                cur.execute("""
                    INSERT INTO fttx.itu_fiber_strands (
                        cable_id, fiber_position, fiber_color_code,
                        attenuation_1310nm_db_per_km,
                        attenuation_1550nm_db_per_km,
                        strand_status
                    ) VALUES (
                        %s, %s, %s, 0.35, 0.25, 'available'
                    );
                """, (cable_id, i+1, fiber_colors[i % 12]))
            
            self.log(f'✓ Created 48 fiber strands', 'success')
            
            # Step 6: Create primary splitter node
            self.progress.setValue(6)
            self.log('Step 6/10: Creating splitter nodes...', 'info')
            
            cur.execute("""
                INSERT INTO fttx.itu_odn_nodes (
                    node_code, node_type, node_function,
                    splitter_type, split_ratio, split_configuration,
                    insertion_loss_typical_db, insertion_loss_max_db,
                    output_ports, installation_type, operational_status,
                    olt_id,
                    geom
                ) VALUES (
                    'SPL-PRIMARY-001', 'splitter', 'primary_split',
                    'PLC', '1:8', 'balanced',
                    10.5, 11.0,
                    8, 'cabinet', 'active',
                    %s,
                    ST_SetSRID(ST_MakePoint(28.0500, -26.2060), 4326)
                ) RETURNING node_id;
            """, (olt_id,))
            primary_splitter_id = cur.fetchone()[0]
            self.log(f'✓ Primary splitter created (1:8)', 'success')
            
            # Create secondary splitters
            secondary_splitters = []
            positions = [
                (28.0510, -26.2070),
                (28.0520, -26.2075),
                (28.0530, -26.2080),
                (28.0540, -26.2085)
            ]
            
            for idx, (lon, lat) in enumerate(positions):
                cur.execute("""
                    INSERT INTO fttx.itu_odn_nodes (
                        node_code, node_type, node_function,
                        splitter_type, split_ratio, split_configuration,
                        insertion_loss_typical_db, insertion_loss_max_db,
                        output_ports, installation_type, operational_status,
                        parent_node_id, olt_id,
                        geom
                    ) VALUES (
                        %s, 'splitter', 'secondary_split',
                        'PLC', '1:32', 'balanced',
                        17.5, 18.0,
                        32, 'street_cabinet', 'active',
                        %s, %s,
                        ST_SetSRID(ST_MakePoint(%s, %s), 4326)
                    ) RETURNING node_id;
                """, (f'SPL-SEC-{idx+1:03d}', primary_splitter_id, olt_id, lon, lat))
                secondary_splitters.append(cur.fetchone()[0])
            
            self.log(f'✓ Created {len(secondary_splitters)} secondary splitters (1:32)', 'success')
            
            # Step 7: Create splice points
            self.progress.setValue(7)
            self.log('Step 7/10: Creating splice points...', 'info')
            
            splice_ids = []
            # Splice at primary splitter
            cur.execute("""
                INSERT INTO fttx.itu_splice_points (
                    splice_code, node_id, splice_type, splice_method,
                    typical_splice_loss_db, closure_type,
                    geom
                ) VALUES (
                    'SPLICE-PRI-001', %s, 'distribution', 'fusion',
                    0.05, 'dome_closure',
                    ST_SetSRID(ST_MakePoint(28.0500, -26.2060), 4326)
                ) RETURNING splice_id;
            """, (primary_splitter_id,))
            splice_ids.append(cur.fetchone()[0])
            
            self.log(f'✓ Created splice points', 'success')
            
            # Step 8: Create ONUs
            self.progress.setValue(8)
            self.log(f'Step 8/10: Creating {onu_count} ONUs...', 'info')
            
            import random
            onu_ids = []
            
            # Distribute ONUs across secondary splitters
            onus_per_splitter = onu_count // len(secondary_splitters)
            
            for splitter_idx, splitter_id in enumerate(secondary_splitters):
                base_lon = 28.0510 + (splitter_idx * 0.001)
                base_lat = -26.2070 - (splitter_idx * 0.001)
                
                for onu_idx in range(onus_per_splitter):
                    # Randomize position around splitter
                    lon = base_lon + random.uniform(-0.002, 0.002)
                    lat = base_lat + random.uniform(-0.002, 0.002)
                    
                    cur.execute("""
                        INSERT INTO fttx.itu_onu_equipment (
                            onu_serial_number, pon_standard, onu_class,
                            upstream_wavelength_nm, downstream_wavelength_nm,
                            onu_tx_power_min_dbm, onu_tx_power_max_dbm,
                            onu_rx_sensitivity_dbm, ethernet_ports,
                            connected_olt_id, splitter_node_id,
                            onu_status, manufacturer, model,
                            geom
                        ) VALUES (
                            %s, 'GPON', 'B+',
                            1310, 1490,
                            0.5, 5.0,
                            -27.0, 4,
                            %s, %s,
                            'active', 'Huawei', 'HG8245H',
                            ST_SetSRID(ST_MakePoint(%s, %s), 4326)
                        ) RETURNING onu_id;
                    """, (
                        f'HWTC{splitter_idx:02d}{onu_idx:03d}ABCD',
                        olt_id,
                        splitter_id,
                        lon, lat
                    ))
                    onu_ids.append(cur.fetchone()[0])
            
            self.log(f'✓ Created {len(onu_ids)} ONUs', 'success')
            
            # Step 9: Create network paths
            self.progress.setValue(9)
            self.log('Step 9/10: Creating network paths...', 'info')
            
            # Create a few sample paths
            for i in range(min(10, len(onu_ids))):
                cur.execute("""
                    INSERT INTO fttx.itu_network_paths (
                        path_code, olt_port_id, onu_id,
                        total_path_length_km, total_fiber_loss_1550nm_db,
                        total_splice_loss_db, total_splitter_loss_db,
                        total_path_loss_db, path_status
                    ) VALUES (
                        %s, %s, %s,
                        2.5, 0.625, 0.15, 28.0,
                        28.775, 'active'
                    );
                """, (f'PATH-{i+1:04d}', pon_port_ids[0], onu_ids[i]))
            
            self.log(f'✓ Created 10 network paths', 'success')
            
            # Step 10: Commit
            self.progress.setValue(10)
            self.log('Step 10/10: Committing to database...', 'info')
            
            conn.commit()
            
            self.log('=' * 50, 'success')
            self.log('✓ Sample network generated successfully!', 'success')
            self.log('=' * 50, 'success')
            self.log('', 'info')
            self.log(f'Network Summary:', 'info')
            self.log(f'  - 1 OLT with 4 PON ports', 'info')
            self.log(f'  - 1 Primary splitter (1:8)', 'info')
            self.log(f'  - {len(secondary_splitters)} Secondary splitters (1:32)', 'info')
            self.log(f'  - {len(onu_ids)} ONUs (customers)', 'info')
            self.log(f'  - 1 Fiber cable (48 strands)', 'info')
            self.log(f'  - 10 Network paths', 'info')
            
            QMessageBox.information(
                self,
                'Success',
                f'✓ Sample network generated successfully!\n\n'
                f'Created:\n'
                f'  - 1 OLT with 4 PON ports\n'
                f'  - 5 Splitter nodes\n'
                f'  - {len(onu_ids)} ONUs\n'
                f'  - 1 Fiber cable with 48 strands\n\n'
                f'Now click 🗺️ Load Layers to see the network on the map!'
            )
            
        except Exception as e:
            conn.rollback()
            self.log('=' * 50, 'error')
            self.log(f'✗ Generation failed: {str(e)}', 'error')
            self.log('=' * 50, 'error')
            QMessageBox.critical(
                self,
                'Error',
                f'Generation failed:\n\n{str(e)}'
            )
        
        finally:
            self.progress.setVisible(False)
            self.generate_btn.setEnabled(True)