# itu_fttx_plugin/gui/utils_ui.py

from qgis.PyQt.QtCore import Qt, QSize
from qgis.PyQt.QtWidgets import (
    QDialog, QMessageBox, QVBoxLayout, QHBoxLayout, 
    QLabel, QPushButton, QLineEdit, QSpinBox, QComboBox, 
    QGroupBox, QFormLayout, QDialogButtonBox, QTextEdit, 
    QDoubleSpinBox, QCheckBox, QTabWidget, QWidget, QFrame,
    QGridLayout, QScrollArea, QGraphicsView, QGraphicsScene,
    QTableWidget, QTableWidgetItem, QHeaderView, QFileDialog
)
from qgis.PyQt.QtGui import QPen, QBrush, QColor, QFont, QPainter, QImage
from qgis.PyQt.QtPrintSupport import QPrinter
from qgis.PyQt.QtSvg import QSvgGenerator
import math


# ==========================================
# SPLICE DIAGRAM VIEWER
# ==========================================

class SpliceDiagramViewer(QDialog):
    """Splice Diagram Viewer - Shows closure splice diagrams"""
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Splice Diagram Viewer')
        self.setMinimumWidth(800)
        self.setMinimumHeight(600)
        
        self.init_ui()
        self.load_closures()
    
    def init_ui(self):
        """Initialize UI"""
        layout = QVBoxLayout()
        
        # Title
        title = QLabel('<h2>🔗 Splice Diagram Viewer</h2>')
        title.setAlignment(Qt.AlignCenter)
        layout.addWidget(title)
        
        # Closure selection
        selection_layout = QHBoxLayout()
        selection_layout.addWidget(QLabel('<b>Select Closure:</b>'))
        
        self.closure_combo = QComboBox()
        self.closure_combo.currentIndexChanged.connect(self.on_closure_selected)
        selection_layout.addWidget(self.closure_combo, 1)
        
        refresh_btn = QPushButton('🔄 Refresh')
        refresh_btn.clicked.connect(self.load_closures)
        selection_layout.addWidget(refresh_btn)
        
        layout.addLayout(selection_layout)
        
        # Diagram display area
        self.diagram_view = QTextEdit()
        self.diagram_view.setReadOnly(True)
        self.diagram_view.setFont(QFont('Courier New', 10))
        layout.addWidget(self.diagram_view)
        
        # Close button
        button_layout = QHBoxLayout()
        button_layout.addStretch()
        
        close_btn = QPushButton('Close')
        close_btn.clicked.connect(self.reject)
        button_layout.addWidget(close_btn)
        
        layout.addLayout(button_layout)
        
        self.setLayout(layout)
    
    def load_closures(self):
        """Load closures from database"""
        self.closure_combo.clear()
        
        if not self.plugin.db_connection:
            self.diagram_view.setPlainText(
                'Please connect to database first using 🔌 Connect DB button'
            )
            return
        
        try:
            cur = self.plugin.db_connection.cursor()
            
            cur.execute("""
                SELECT 
                    splice_id,
                    splice_code,
                    closure_type,
                    (SELECT COUNT(*) 
                     FROM fttx.itu_splice_connections sc 
                     WHERE sc.splice_id = sp.splice_id) as connection_count
                FROM fttx.itu_splice_points sp
                WHERE closure_type IS NOT NULL
                ORDER BY splice_code
            """)
            
            closures = cur.fetchall()
            
            if not closures:
                self.closure_combo.addItem('No closures found', None)
                self.diagram_view.setPlainText(
                    'No closures found in database.\n\n'
                    'Closures should have been created during sample data generation.\n'
                    'Try running the closure creation SQL script.'
                )
            else:
                for splice_id, code, closure_type, conn_count in closures:
                    display_text = f'{code} ({closure_type}, {conn_count} connections)'
                    self.closure_combo.addItem(display_text, splice_id)
            
            cur.close()
            
        except Exception as e:
            self.diagram_view.setPlainText(f'Error loading closures:\n\n{str(e)}')
    
    def on_closure_selected(self, index):
        """Handle closure selection"""
        if index < 0:
            return
        
        splice_id = self.closure_combo.itemData(index)
        if splice_id is None:
            return
        
        self.show_closure_diagram(splice_id)
    
    def show_closure_diagram(self, splice_id):
        """Display splice diagram for selected closure"""
        try:
            cur = self.plugin.db_connection.cursor()
            
            # Get closure details
            cur.execute("""
                SELECT 
                    splice_code,
                    closure_type,
                    splice_method,
                    sealing_method,
                    typical_splice_loss_db,
                    max_splice_loss_db,
                    otdr_tested,
                    test_result
                FROM fttx.itu_splice_points
                WHERE splice_id = %s
            """, (splice_id,))
            
            closure_data = cur.fetchone()
            if not closure_data:
                self.diagram_view.setPlainText('Closure not found')
                return
            
            (code, closure_type, method, sealing, typ_loss, max_loss, 
             tested, result) = closure_data
            
            # Get splice connections
            cur.execute("""
                SELECT 
                    sc.connection_id,
                    fs_in.fiber_position as input_pos,
                    fs_in.fiber_color_code as input_color,
                    fs_out.fiber_position as output_pos,
                    fs_out.fiber_color_code as output_color,
                    sc.insertion_loss_1550nm_db,
                    sc.return_loss_db,
                    sc.connection_quality
                FROM fttx.itu_splice_connections sc
                LEFT JOIN fttx.itu_fiber_strands fs_in 
                    ON sc.input_strand_id = fs_in.strand_id
                LEFT JOIN fttx.itu_fiber_strands fs_out 
                    ON sc.output_strand_id = fs_out.strand_id
                WHERE sc.splice_id = %s
                ORDER BY fs_in.fiber_position
            """, (splice_id,))
            
            connections = cur.fetchall()
            
            # Build diagram text
            diagram = []
            diagram.append('=' * 80)
            diagram.append(f'SPLICE CLOSURE DIAGRAM: {code}')
            diagram.append('=' * 80)
            diagram.append('')
            diagram.append(f'Closure Type:        {closure_type or "N/A"}')
            diagram.append(f'Splice Method:       {method or "N/A"}')
            diagram.append(f'Sealing Method:      {sealing or "N/A"}')
            diagram.append(f'Typical Loss:        {typ_loss} dB')
            diagram.append(f'Maximum Loss:        {max_loss} dB')
            diagram.append(f'OTDR Tested:         {"Yes" if tested else "No"}')
            diagram.append(f'Test Result:         {result or "N/A"}')
            diagram.append('')
            diagram.append('-' * 80)
            diagram.append('SPLICE CONNECTIONS')
            diagram.append('-' * 80)
            diagram.append('')
            
            if connections:
                diagram.append(f'Total Connections: {len(connections)}')
                diagram.append('')
                diagram.append('Input Fiber          →  Output Fiber         Loss    Quality')
                diagram.append('-' * 80)
                
                for conn in connections:
                    (conn_id, in_pos, in_color, out_pos, out_color, 
                     loss, ret_loss, quality) = conn
                    
                    input_str = f'Pos {in_pos:02d} ({in_color})' if in_pos else 'Unknown'
                    output_str = f'Pos {out_pos:02d} ({out_color})' if out_pos else 'Unknown'
                    loss_str = f'{loss:.3f} dB' if loss else 'N/A'
                    
                    line = f'{input_str:20s} →  {output_str:20s} {loss_str:8s} {quality or "N/A"}'
                    diagram.append(line)
                
                # Statistics
                diagram.append('')
                diagram.append('-' * 80)
                diagram.append('STATISTICS')
                diagram.append('-' * 80)
                
                losses = [c[5] for c in connections if c[5] is not None]
                if losses:
                    avg_loss = sum(losses) / len(losses)
                    min_loss = min(losses)
                    max_loss = max(losses)
                    
                    diagram.append(f'Average Loss: {avg_loss:.3f} dB')
                    diagram.append(f'Minimum Loss: {min_loss:.3f} dB')
                    diagram.append(f'Maximum Loss: {max_loss:.3f} dB')
                
            else:
                diagram.append('No splice connections defined for this closure.')
            
            diagram.append('')
            diagram.append('=' * 80)
            
            self.diagram_view.setPlainText('\n'.join(diagram))
            
            cur.close()
            
        except Exception as e:
            self.diagram_view.setPlainText(f'Error generating diagram:\n\n{str(e)}')


# ==========================================
# BUTTERFLY CLOSURE VIEWER
# ==========================================

class ButterflyClosureViewer(QDialog):
    """Visual representation of butterfly splice closure fiber arrangement"""
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('🦋 Butterfly Closure Viewer')
        self.setMinimumSize(900, 700)
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        
        header = QLabel('🦋 Butterfly Closure Fiber Arrangement')
        header.setStyleSheet('font-size: 14pt; font-weight: bold; background-color: #7B1FA2; color: white; padding: 10px;')
        layout.addWidget(header)
        
        info = QLabel('Butterfly closures split a single feeder cable into multiple distribution cables.')
        info.setWordWrap(True)
        layout.addWidget(info)
        
        # Configuration
        config_group = QGroupBox('Closure Configuration')
        config_layout = QFormLayout()
        
        self.input_fibers = QSpinBox()
        self.input_fibers.setRange(12, 288)
        self.input_fibers.setValue(24)
        self.input_fibers.valueChanged.connect(self.draw_butterfly)
        config_layout.addRow('Input Fiber Count:', self.input_fibers)
        
        self.output_cables = QSpinBox()
        self.output_cables.setRange(2, 12)
        self.output_cables.setValue(4)
        self.output_cables.valueChanged.connect(self.draw_butterfly)
        config_layout.addRow('Output Cable Count:', self.output_cables)
        
        self.closure_type = QComboBox()
        self.closure_type.addItems(['Inline', 'Butt', 'L-Type'])
        config_layout.addRow('Closure Type:', self.closure_type)
        
        config_group.setLayout(config_layout)
        layout.addWidget(config_group)
        
        # Graphics view
        self.butterfly_view = QGraphicsView()
        self.butterfly_scene = QGraphicsScene()
        self.butterfly_view.setScene(self.butterfly_scene)
        self.butterfly_view.setRenderHint(QPainter.Antialiasing)
        layout.addWidget(self.butterfly_view)
        
        # Fiber assignment table
        assignment_group = QGroupBox('Fiber Assignments')
        assignment_layout = QVBoxLayout()
        
        self.assignment_table = QTableWidget()
        self.assignment_table.setColumnCount(4)
        self.assignment_table.setHorizontalHeaderLabels([
            'Input Fiber', 'Output Cable', 'Output Fiber', 'Color'
        ])
        self.assignment_table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.assignment_table.setMaximumHeight(200)
        assignment_layout.addWidget(self.assignment_table)
        
        assignment_group.setLayout(assignment_layout)
        layout.addWidget(assignment_group)
        
        # Action buttons
        button_layout = QHBoxLayout()
        
        generate_btn = QPushButton('🔄 Generate Assignment')
        generate_btn.clicked.connect(self.generate_assignment)
        button_layout.addWidget(generate_btn)
        
        export_btn = QPushButton('💾 Export')
        export_btn.clicked.connect(self.export_butterfly)
        button_layout.addWidget(export_btn)
        
        button_layout.addStretch()
        
        close_btn = QPushButton('Close')
        close_btn.clicked.connect(self.reject)
        button_layout.addWidget(close_btn)
        
        layout.addLayout(button_layout)
        
        self.setLayout(layout)
        self.draw_butterfly()
    
    def draw_butterfly(self):
        """Draw the butterfly closure diagram"""
        self.butterfly_scene.clear()
        
        input_count = self.input_fibers.value()
        output_count = self.output_cables.value()
        fibers_per_output = input_count // output_count
        
        # Title
        title = self.butterfly_scene.addText(f'Butterfly Closure: {input_count}F → {output_count}x{fibers_per_output}F')
        font = QFont()
        font.setBold(True)
        font.setPointSize(12)
        title.setFont(font)
        title.setPos(200, 10)
        
        # Input cable (left)
        input_x = 50
        input_y = 100
        input_height = output_count * 60
        self.butterfly_scene.addRect(input_x, input_y, 100, input_height,
                                     QPen(Qt.black, 2), QBrush(QColor(100, 150, 250)))
        text = self.butterfly_scene.addText(f'{input_count}F\nInput')
        text.setPos(input_x + 20, input_y + input_height/2 - 20)
        
        # Closure (middle)
        closure_x = 250
        closure_y = input_y - 50
        closure_height = input_height + 100
        self.butterfly_scene.addEllipse(closure_x, closure_y, 200, closure_height,
                                       QPen(Qt.black, 3), QBrush(QColor(220, 220, 220)))
        text = self.butterfly_scene.addText('Butterfly\nClosure')
        font.setPointSize(10)
        text.setFont(font)
        text.setPos(closure_x + 60, closure_y + closure_height/2 - 20)
        
        # Output cables (right)
        output_x = 550
        colors = [QColor(255, 100, 100), QColor(100, 255, 100), 
                 QColor(100, 100, 255), QColor(255, 255, 100),
                 QColor(255, 100, 255), QColor(100, 255, 255)]
        
        for i in range(output_count):
            y_pos = input_y + (i * 60)
            color = colors[i % len(colors)]
            
            self.butterfly_scene.addRect(output_x, y_pos, 100, 50,
                                         QPen(Qt.black, 2), QBrush(color))
            text = self.butterfly_scene.addText(f'{fibers_per_output}F\nOut {i+1}')
            text.setPos(output_x + 15, y_pos + 10)
            
            # Draw connection line
            pen = QPen(color, 2)
            self.butterfly_scene.addLine(closure_x + 200, closure_y + 50 + (i * (closure_height-100)/output_count),
                                        output_x, y_pos + 25, pen)
        
        # Draw input connection
        self.butterfly_scene.addLine(input_x + 100, input_y + input_height/2,
                                     closure_x, closure_y + closure_height/2,
                                     QPen(Qt.black, 3))
    
    def generate_assignment(self):
        """Generate fiber assignment table"""
        self.assignment_table.setRowCount(0)
        
        input_count = self.input_fibers.value()
        output_count = self.output_cables.value()
        fibers_per_output = input_count // output_count
        
        fiber_colors = ['Blue', 'Orange', 'Green', 'Brown', 'Slate', 'White',
                       'Red', 'Black', 'Yellow', 'Violet', 'Rose', 'Aqua']
        
        for i in range(input_count):
            output_cable = (i // fibers_per_output) + 1
            output_fiber = (i % fibers_per_output) + 1
            color = fiber_colors[i % len(fiber_colors)]
            
            row = self.assignment_table.rowCount()
            self.assignment_table.insertRow(row)
            self.assignment_table.setItem(row, 0, QTableWidgetItem(f'F{i+1}'))
            self.assignment_table.setItem(row, 1, QTableWidgetItem(f'Cable {output_cable}'))
            self.assignment_table.setItem(row, 2, QTableWidgetItem(f'F{output_fiber}'))
            self.assignment_table.setItem(row, 3, QTableWidgetItem(color))
    
    def export_butterfly(self):
        """Export butterfly diagram to image file"""
        filepath, _ = QFileDialog.getSaveFileName(
            self, 'Export Butterfly Diagram', '',
            'PNG Images (*.png);;SVG Files (*.svg);;All Files (*)')
        if not filepath:
            return

        try:
            if filepath.lower().endswith('.svg'):
                generator = QSvgGenerator()
                generator.setFileName(filepath)
                generator.setSize(QSize(800, 600))
                generator.setViewBox(self.butterfly_scene.sceneRect())
                painter = QPainter(generator)
                self.butterfly_scene.render(painter)
                painter.end()
            else:
                rect = self.butterfly_scene.sceneRect()
                image = QImage(int(rect.width()), int(rect.height()), QImage.Format_ARGB32)
                image.fill(QColor(255, 255, 255))
                painter = QPainter(image)
                painter.setRenderHint(QPainter.Antialiasing)
                self.butterfly_scene.render(painter)
                painter.end()
                image.save(filepath)

            QMessageBox.information(self, 'Exported',
                f'Butterfly diagram exported to:\n{filepath}')
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Export failed:\n{str(e)}')


# ==========================================
# FIBER COLOR REFERENCE
# ==========================================

class FiberColorReference(QDialog):
    """Standard fiber optic color coding reference"""
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('🎨 Fiber Color Code Reference')
        self.setMinimumSize(800, 600)
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        
        header = QLabel('🎨 Fiber Optic Color Code Reference')
        header.setStyleSheet('font-size: 14pt; font-weight: bold; background-color: #00897B; color: white; padding: 10px;')
        layout.addWidget(header)
        
        info = QLabel('Standard fiber color codes according to TIA-598 and IEC standards.')
        info.setWordWrap(True)
        layout.addWidget(info)
        
        # Tabs for different standards
        tabs = QTabWidget()
        
        # TIA-598 Standard
        tabs.addTab(self.create_tia598_tab(), 'TIA-598 (US)')
        
        # IEC Standard
        tabs.addTab(self.create_iec_tab(), 'IEC (International)')
        
        # Tube Colors
        tabs.addTab(self.create_tube_colors_tab(), 'Buffer Tube Colors')
        
        layout.addWidget(tabs)
        
        # Close button
        button_box = QDialogButtonBox(QDialogButtonBox.Close)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
        self.setLayout(layout)
    
    def create_tia598_tab(self):
        """Create TIA-598 color reference tab"""
        widget = QWidget()
        layout = QVBoxLayout()
        
        layout.addWidget(QLabel('<b>TIA-598 Fiber Color Code (12-Fiber Base)</b>'))
        
        table = QTableWidget()
        table.setColumnCount(3)
        table.setHorizontalHeaderLabels(['Position', 'Color', 'Visual'])
        table.setRowCount(12)
        
        colors = [
            ('1', 'Blue', '#0000FF'),
            ('2', 'Orange', '#FF8800'),
            ('3', 'Green', '#00FF00'),
            ('4', 'Brown', '#8B4513'),
            ('5', 'Slate (Gray)', '#708090'),
            ('6', 'White', '#FFFFFF'),
            ('7', 'Red', '#FF0000'),
            ('8', 'Black', '#000000'),
            ('9', 'Yellow', '#FFFF00'),
            ('10', 'Violet', '#8B00FF'),
            ('11', 'Rose (Pink)', '#FF69B4'),
            ('12', 'Aqua (Turquoise)', '#00FFFF')
        ]
        
        for i, (pos, color_name, color_hex) in enumerate(colors):
            table.setItem(i, 0, QTableWidgetItem(pos))
            table.setItem(i, 1, QTableWidgetItem(color_name))
            
            color_item = QTableWidgetItem()
            color_item.setBackground(QColor(color_hex))
            if color_hex in ['#FFFFFF', '#FFFF00']:
                color_item.setForeground(QColor('#000000'))
            else:
                color_item.setForeground(QColor('#FFFFFF'))
            color_item.setText('████████')
            table.setItem(i, 2, color_item)
        
        table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        layout.addWidget(table)
        
        note = QLabel('<i>Note: For cables >12 fibers, colors repeat with different stripe patterns</i>')
        note.setWordWrap(True)
        layout.addWidget(note)
        
        widget.setLayout(layout)
        return widget
    
    def create_iec_tab(self):
        """Create IEC 60304 color reference tab"""
        widget = QWidget()
        layout = QVBoxLayout()

        layout.addWidget(QLabel('<b>IEC 60304 Fiber Color Code (12-Fiber Base)</b>'))

        table = QTableWidget()
        table.setColumnCount(3)
        table.setHorizontalHeaderLabels(['Position', 'Color', 'Visual'])
        table.setRowCount(12)

        # IEC 60304 standard colors (commonly used internationally)
        colors = [
            ('1', 'Red', '#FF0000'),
            ('2', 'Green', '#00CC00'),
            ('3', 'Blue', '#0000FF'),
            ('4', 'Yellow', '#FFFF00'),
            ('5', 'White', '#FFFFFF'),
            ('6', 'Grey', '#808080'),
            ('7', 'Brown', '#8B4513'),
            ('8', 'Violet', '#8B00FF'),
            ('9', 'Turquoise', '#40E0D0'),
            ('10', 'Black', '#000000'),
            ('11', 'Orange', '#FF8800'),
            ('12', 'Pink', '#FF69B4')
        ]

        for i, (pos, color_name, color_hex) in enumerate(colors):
            table.setItem(i, 0, QTableWidgetItem(pos))
            table.setItem(i, 1, QTableWidgetItem(color_name))

            color_item = QTableWidgetItem()
            color_item.setBackground(QColor(color_hex))
            if color_hex in ['#FFFFFF', '#FFFF00', '#40E0D0']:
                color_item.setForeground(QColor('#000000'))
            else:
                color_item.setForeground(QColor('#FFFFFF'))
            color_item.setText('\u2588' * 8)
            table.setItem(i, 2, color_item)

        table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        layout.addWidget(table)

        note = QLabel(
            '<i>Note: IEC 60304 is used primarily in Europe and internationally.\n'
            'Color order differs from TIA-598. Verify with local standards.</i>')
        note.setWordWrap(True)
        layout.addWidget(note)

        widget.setLayout(layout)
        return widget
    
    def create_tube_colors_tab(self):
        """Create buffer tube color reference"""
        widget = QWidget()
        layout = QVBoxLayout()
        
        layout.addWidget(QLabel('<b>Buffer Tube Color Code (12-Tube Standard)</b>'))
        
        table = QTableWidget()
        table.setColumnCount(3)
        table.setHorizontalHeaderLabels(['Tube', 'Color', 'Typical Fiber Count'])
        table.setRowCount(12)
        
        tube_colors = [
            ('1', 'Blue', '12'),
            ('2', 'Orange', '12'),
            ('3', 'Green', '12'),
            ('4', 'Brown', '12'),
            ('5', 'Slate', '12'),
            ('6', 'White', '12'),
            ('7', 'Red', '12'),
            ('8', 'Black', '12'),
            ('9', 'Yellow', '12'),
            ('10', 'Violet', '12'),
            ('11', 'Rose', '12'),
            ('12', 'Aqua', '12')
        ]
        
        for i, (tube, color, count) in enumerate(tube_colors):
            table.setItem(i, 0, QTableWidgetItem(tube))
            table.setItem(i, 1, QTableWidgetItem(color))
            table.setItem(i, 2, QTableWidgetItem(count))
        
        table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        layout.addWidget(table)
        
        widget.setLayout(layout)
        return widget
