# itu_fttx_plugin/gui/plant_manager.py

from qgis.PyQt.QtCore import Qt, QSize
from qgis.PyQt.QtWidgets import (
    QDialog, QMessageBox, QVBoxLayout, QHBoxLayout, 
    QLabel, QPushButton, QLineEdit, QSpinBox, QComboBox, 
    QGroupBox, QFormLayout, QDialogButtonBox, QTextEdit, 
    QDoubleSpinBox, QCheckBox, QTabWidget, QWidget, QFrame,
    QGridLayout, QProgressBar, QListWidget, QSplitter, 
    QTableWidget, QTableWidgetItem, QHeaderView, QListWidgetItem,
    QRadioButton, QButtonGroup, QScrollArea, QGraphicsView, 
    QGraphicsScene, QDateEdit
)
from qgis.PyQt.QtGui import QFont, QColor, QPen, QBrush, QPixmap, QPainter
from qgis.core import QgsGeometry, QgsPointXY
import math
from datetime import datetime, date
import psycopg2
from ..core.planners import PlanningEngine


# ==========================================
# PLANNING WIZARD
# ==========================================

class PlanningWizard(QDialog):
    """Network Planning Wizard for automated FTTX design"""
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('🧙 FTTX Planning Wizard')
        self.setMinimumSize(900, 700)
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        
        header = QLabel('🧙 FTTX Network Planning Wizard')
        header.setStyleSheet('font-size: 16pt; font-weight: bold; background-color: #5E35B1; color: white; padding: 15px;')
        layout.addWidget(header)
        
        info = QLabel('This wizard will guide you through planning a new FTTX network deployment.')
        info.setWordWrap(True)
        layout.addWidget(info)
        
        self.wizard_tabs = QTabWidget()
        self.wizard_tabs.addTab(self.create_project_info_tab(), '1. Project Info')
        self.wizard_tabs.addTab(self.create_coverage_tab(), '2. Coverage Area')
        self.wizard_tabs.addTab(self.create_design_tab(), '3. Network Design')
        self.wizard_tabs.addTab(self.create_review_tab(), '4. Review')
        
        layout.addWidget(self.wizard_tabs)
        
        nav_layout = QHBoxLayout()
        self.prev_btn = QPushButton('← Previous')
        self.prev_btn.clicked.connect(self.previous_step)
        nav_layout.addWidget(self.prev_btn)
        nav_layout.addStretch()
        self.next_btn = QPushButton('Next →')
        self.next_btn.clicked.connect(self.next_step)
        nav_layout.addWidget(self.next_btn)
        self.generate_btn = QPushButton('✨ Generate Plan')
        self.generate_btn.clicked.connect(self.generate_plan)
        self.generate_btn.setVisible(False)
        nav_layout.addWidget(self.generate_btn)
        cancel_btn = QPushButton('Cancel')
        cancel_btn.clicked.connect(self.reject)
        nav_layout.addWidget(cancel_btn)
        
        layout.addLayout(nav_layout)
        self.setLayout(layout)
        self.update_navigation()
    
    def create_project_info_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        form = QFormLayout()
        self.project_name = QLineEdit()
        self.project_name.setPlaceholderText('e.g., Downtown Fiber Rollout')
        form.addRow('Project Name:', self.project_name)
        self.project_code = QLineEdit()
        self.project_code.setPlaceholderText('e.g., PROJ-2024-001')
        form.addRow('Project Code:', self.project_code)
        self.client_name = QLineEdit()
        form.addRow('Client Name:', self.client_name)
        self.deployment_type = QComboBox()
        self.deployment_type.addItems(['Greenfield', 'Overlay', 'Upgrade'])
        form.addRow('Deployment Type:', self.deployment_type)
        layout.addLayout(form)
        layout.addStretch()
        widget.setLayout(layout)
        return widget
    
    def create_coverage_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        layout.addWidget(QLabel('Define the coverage area for this project:'))
        form = QFormLayout()
        self.coverage_type = QComboBox()
        self.coverage_type.addItems(['Select from Map', 'Manual Entry', 'Import Polygon'])
        form.addRow('Coverage Method:', self.coverage_type)
        self.estimated_homes = QSpinBox()
        self.estimated_homes.setRange(1, 100000)
        self.estimated_homes.setValue(500)
        form.addRow('Estimated Homes Passed:', self.estimated_homes)
        self.area_sqkm = QDoubleSpinBox()
        self.area_sqkm.setRange(0.1, 1000)
        self.area_sqkm.setValue(2.0)
        self.area_sqkm.setSuffix(' km²')
        form.addRow('Area Size:', self.area_sqkm)
        layout.addLayout(form)
        layout.addStretch()
        widget.setLayout(layout)
        return widget
    
    def create_design_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        layout.addWidget(QLabel('Configure network design parameters:'))
        form = QFormLayout()
        self.pon_standard = QComboBox()
        self.pon_standard.addItems(['GPON', 'XG-PON', 'XGS-PON', '10G-EPON'])
        form.addRow('PON Standard:', self.pon_standard)
        self.split_ratio = QComboBox()
        self.split_ratio.addItems(['1:32', '1:64', '1:128'])
        self.split_ratio.setCurrentText('1:32')
        form.addRow('Split Ratio:', self.split_ratio)
        self.max_olt_distance = QSpinBox()
        self.max_olt_distance.setRange(1, 60)
        self.max_olt_distance.setValue(20)
        self.max_olt_distance.setSuffix(' km')
        form.addRow('Max OLT Distance:', self.max_olt_distance)
        self.redundancy = QCheckBox('Include redundant paths')
        form.addRow('Redundancy:', self.redundancy)
        layout.addLayout(form)
        layout.addStretch()
        widget.setLayout(layout)
        return widget
    
    def create_review_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        layout.addWidget(QLabel('<b>Review your planning parameters:</b>'))
        self.review_text = QTextEdit()
        self.review_text.setReadOnly(True)
        layout.addWidget(self.review_text)
        widget.setLayout(layout)
        return widget
    
    def update_navigation(self):
        current_tab = self.wizard_tabs.currentIndex()
        self.prev_btn.setEnabled(current_tab > 0)
        self.next_btn.setVisible(current_tab < 3)
        self.generate_btn.setVisible(current_tab == 3)
        if current_tab == 3:
            self.update_review()
    
    def previous_step(self):
        current = self.wizard_tabs.currentIndex()
        if current > 0:
            self.wizard_tabs.setCurrentIndex(current - 1)
            self.update_navigation()
    
    def next_step(self):
        current = self.wizard_tabs.currentIndex()
        if current < 3:
            self.wizard_tabs.setCurrentIndex(current + 1)
            self.update_navigation()
    
    def update_review(self):
        review = f"""
<h2>Planning Summary</h2>
<h3>Project Information:</h3>
<ul>
<li><b>Project Name:</b> {self.project_name.text() or 'Not specified'}</li>
<li><b>Project Code:</b> {self.project_code.text() or 'Not specified'}</li>
<li><b>Client:</b> {self.client_name.text() or 'Not specified'}</li>
<li><b>Deployment Type:</b> {self.deployment_type.currentText()}</li>
</ul>
<h3>Coverage Area:</h3>
<ul>
<li><b>Estimated Homes:</b> {self.estimated_homes.value()}</li>
<li><b>Area Size:</b> {self.area_sqkm.value()} km²</li>
<li><b>Coverage Method:</b> {self.coverage_type.currentText()}</li>
</ul>
<h3>Network Design:</h3>
<ul>
<li><b>PON Standard:</b> {self.pon_standard.currentText()}</li>
<li><b>Split Ratio:</b> {self.split_ratio.currentText()}</li>
<li><b>Max Distance:</b> {self.max_olt_distance.value()} km</li>
<li><b>Redundancy:</b> {'Yes' if self.redundancy.isChecked() else 'No'}</li>
</ul>
"""
        self.review_text.setHtml(review)
    
    def generate_plan(self):
        if not self.project_name.text():
            QMessageBox.warning(self, 'Missing Info', 'Please enter a project name')
            return

        reply = QMessageBox.question(self, 'Generate Network',
            'Generate network plan with current parameters?', QMessageBox.Yes | QMessageBox.No)

        if reply == QMessageBox.Yes:
            # Use PlanningEngine for calculations
            split_text = self.split_ratio.currentText()
            onus_per_port = int(split_text.split(':')[1]) if ':' in split_text else 64

            params = {
                'customer_count': self.estimated_homes.value(),
                'onus_per_port': onus_per_port,
                'olt_ports_per_chassis': 16,
                'pon_standard': self.pon_standard.currentText(),
                'max_distance_km': self.max_olt_distance.value(),
                'redundancy': self.redundancy.isChecked(),
            }

            engine = PlanningEngine(self.plugin.db_connection)
            success, report = engine.execute_plan(params)

            if success:
                QMessageBox.information(self, 'Plan Generated',
                    f'Planning wizard completed!\n\n{report}')
                self.accept()
            else:
                QMessageBox.critical(self, 'Planning Failed', report)


# ==========================================
# INSIDE PLANT MANAGER
# ==========================================

class InsidePlantManager(QDialog):
    """Complete Inside Plant Management System"""
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Inside Plant Manager')
        self.setMinimumSize(1200, 850)
        self.init_ui()
        self.load_buildings()
    
    def init_ui(self):
        layout = QVBoxLayout()
        header = QLabel('🏢 Inside Plant Manager')
        header.setStyleSheet('font-size: 16pt; font-weight: bold; background-color: #00897B; color: white; padding: 15px;')
        layout.addWidget(header)
        
        main_splitter = QSplitter(Qt.Horizontal)
        left_panel = self.create_navigation_panel()
        main_splitter.addWidget(left_panel)
        
        self.content_tabs = QTabWidget()
        self.content_tabs.addTab(self.create_overview_tab(), '📊 Overview')
        self.content_tabs.addTab(self.create_buildings_tab(), '🏢 Buildings')
        self.content_tabs.addTab(self.create_rooms_tab(), '🚪 Rooms')
        self.content_tabs.addTab(self.create_racks_tab(), '📦 Racks')
        
        main_splitter.addWidget(self.content_tabs)
        main_splitter.setStretchFactor(0, 1)
        main_splitter.setStretchFactor(1, 3)
        layout.addWidget(main_splitter)
        
        button_layout = QHBoxLayout()
        refresh_btn = QPushButton('🔄 Refresh')
        refresh_btn.clicked.connect(self.refresh_all)
        button_layout.addWidget(refresh_btn)
        button_layout.addStretch()
        close_btn = QPushButton('Close')
        close_btn.clicked.connect(self.reject)
        button_layout.addWidget(close_btn)
        layout.addLayout(button_layout)
        
        self.setLayout(layout)
    
    def create_navigation_panel(self):
        widget = QWidget()
        layout = QVBoxLayout()
        layout.addWidget(QLabel('<b>Navigation:</b>'))
        self.nav_tree = QListWidget()
        self.nav_tree.itemClicked.connect(self.on_nav_item_clicked)
        layout.addWidget(self.nav_tree)
        self.quick_stats = QLabel()
        self.quick_stats.setStyleSheet('background-color: #f0f0f0; padding: 10px; border: 1px solid #ccc;')
        layout.addWidget(self.quick_stats)
        widget.setLayout(layout)
        return widget
    
    def create_overview_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        cards_layout = QGridLayout()
        cards_layout.addWidget(self.create_stat_card('🏢 Buildings', '0', '#2196F3'), 0, 0)
        cards_layout.addWidget(self.create_stat_card('🚪 Rooms', '0', '#4CAF50'), 0, 1)
        cards_layout.addWidget(self.create_stat_card('📦 Racks', '0', '#FF9800'), 0, 2)
        layout.addLayout(cards_layout)
        layout.addStretch()
        widget.setLayout(layout)
        return widget
    
    def create_stat_card(self, title, value, color):
        frame = QFrame()
        frame.setFrameStyle(QFrame.Box | QFrame.Raised)
        frame.setStyleSheet(f'background-color: {color}; color: white; border-radius: 5px;')
        layout = QVBoxLayout()
        title_label = QLabel(title)
        title_label.setStyleSheet('font-size: 12pt; font-weight: bold;')
        title_label.setAlignment(Qt.AlignCenter)
        layout.addWidget(title_label)
        value_label = QLabel(value)
        value_label.setStyleSheet('font-size: 24pt; font-weight: bold;')
        value_label.setAlignment(Qt.AlignCenter)
        value_label.setObjectName(f'stat_{title.split()[1].lower()}')
        layout.addWidget(value_label)
        frame.setLayout(layout)
        return frame
    
    def create_buildings_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        toolbar = QHBoxLayout()
        add_btn = QPushButton('➕ Add Building')
        add_btn.clicked.connect(self.add_building)
        toolbar.addWidget(add_btn)
        edit_btn = QPushButton('✏️ Edit')
        edit_btn.clicked.connect(self.edit_building)
        toolbar.addWidget(edit_btn)
        delete_btn = QPushButton('🗑️ Delete')
        delete_btn.clicked.connect(self.delete_building)
        toolbar.addWidget(delete_btn)
        toolbar.addStretch()
        layout.addLayout(toolbar)
        
        self.buildings_table = QTableWidget()
        self.buildings_table.setColumnCount(9)
        self.buildings_table.setHorizontalHeaderLabels([
            'ID', 'Code', 'Name', 'Street Address', 'City', 'Floors', 'Rooms', 'Type', 'Owner'
        ])
        self.buildings_table.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.buildings_table.setAlternatingRowColors(True)
        self.buildings_table.setSortingEnabled(True)
        self.buildings_table.setSelectionBehavior(QTableWidget.SelectRows)
        self.buildings_table.itemDoubleClicked.connect(self.edit_building)
        layout.addWidget(self.buildings_table)
        
        widget.setLayout(layout)
        return widget
    
    def create_rooms_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        toolbar = QHBoxLayout()
        add_btn = QPushButton('➕ Add Room')
        add_btn.clicked.connect(self.add_room)
        toolbar.addWidget(add_btn)
        edit_btn = QPushButton('✏️ Edit')
        edit_btn.clicked.connect(self.edit_room)
        toolbar.addWidget(edit_btn)
        delete_btn = QPushButton('🗑️ Delete')
        delete_btn.clicked.connect(self.delete_room)
        toolbar.addWidget(delete_btn)
        toolbar.addStretch()
        layout.addLayout(toolbar)
        
        self.rooms_table = QTableWidget()
        self.rooms_table.setColumnCount(9)
        self.rooms_table.setHorizontalHeaderLabels([
            'ID', 'Code', 'Name', 'Building', 'Floor', 'Type', 'Size (m²)', 'HVAC', 'UPS'
        ])
        self.rooms_table.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.rooms_table.setAlternatingRowColors(True)
        self.rooms_table.setSortingEnabled(True)
        self.rooms_table.setSelectionBehavior(QTableWidget.SelectRows)
        self.rooms_table.itemDoubleClicked.connect(self.edit_room)
        layout.addWidget(self.rooms_table)
        
        widget.setLayout(layout)
        return widget
    
    def create_racks_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        toolbar = QHBoxLayout()
        add_btn = QPushButton('➕ Add Rack')
        add_btn.clicked.connect(self.add_rack)
        toolbar.addWidget(add_btn)
        edit_btn = QPushButton('✏️ Edit')
        edit_btn.clicked.connect(self.edit_rack)
        toolbar.addWidget(edit_btn)
        delete_btn = QPushButton('🗑️ Delete')
        delete_btn.clicked.connect(self.delete_rack)
        toolbar.addWidget(delete_btn)
        toolbar.addStretch()
        layout.addLayout(toolbar)
        
        self.racks_table = QTableWidget()
        self.racks_table.setColumnCount(8)
        self.racks_table.setHorizontalHeaderLabels([
            'ID', 'Code', 'Name', 'Room', 'Height (RU)', 'Used RU', 'Available RU', 'Utilization %'
        ])
        self.racks_table.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.racks_table.setAlternatingRowColors(True)
        self.racks_table.setSortingEnabled(True)
        self.racks_table.setSelectionBehavior(QTableWidget.SelectRows)
        self.racks_table.itemDoubleClicked.connect(self.edit_rack)
        layout.addWidget(self.racks_table)
        
        widget.setLayout(layout)
        return widget
    
    # ==========================================
    # DATA LOADING
    # ==========================================
    
    def on_nav_item_clicked(self, item):
        text = item.text()
        if '🏢' in text:
            self.content_tabs.setCurrentIndex(1)
        elif '🚪' in text:
            self.content_tabs.setCurrentIndex(2)
        elif '📦' in text:
            self.content_tabs.setCurrentIndex(3)
    
    def load_buildings(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT building_id, building_code, building_name, 
                       street_address, city, floors_count, building_type, owner,
                       (SELECT COUNT(*) FROM fttx.rooms WHERE building_id = b.building_id) as room_count
                FROM fttx.buildings b
                ORDER BY building_code
            """)
            
            self.buildings_table.setRowCount(0)
            self.nav_tree.clear()
            
            for row_data in cursor.fetchall():
                row = self.buildings_table.rowCount()
                self.buildings_table.insertRow(row)
                for col, value in enumerate(row_data):
                    self.buildings_table.setItem(row, col, QTableWidgetItem(str(value or '')))
                nav_item = QListWidgetItem(f'🏢 {row_data[1]} - {row_data[2]}')
                nav_item.setData(Qt.UserRole, row_data[0])
                self.nav_tree.addItem(nav_item)
            
            cursor.close()
            self.load_rooms()  # Load rooms after buildings
            self.load_racks()  # Load racks after rooms
            self.update_quick_stats()
            
        except Exception as e:
            QMessageBox.warning(self, 'Warning', f'Could not load buildings:\n{str(e)}')
    
    def load_rooms(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT r.room_id, r.room_code, r.room_name, b.building_code, 
                       r.floor_number, r.room_type, r.room_size_sqm, r.has_hvac, r.has_ups
                FROM fttx.rooms r
                LEFT JOIN fttx.buildings b ON r.building_id = b.building_id
                ORDER BY b.building_code, r.room_code
            """)
            
            self.rooms_table.setRowCount(0)
            for row_data in cursor.fetchall():
                row = self.rooms_table.rowCount()
                self.rooms_table.insertRow(row)
                for col, value in enumerate(row_data):
                    if col in [7, 8]:  # Boolean columns
                        self.rooms_table.setItem(row, col, QTableWidgetItem('Yes' if value else 'No'))
                    else:
                        self.rooms_table.setItem(row, col, QTableWidgetItem(str(value or '')))
            
            cursor.close()
        except Exception as e:
            pass
    
    def load_racks(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT r.rack_id, r.rack_code, r.rack_name, rm.room_code, 
                       r.height_ru, r.used_ru, r.available_ru,
                       ROUND((r.used_ru::NUMERIC / NULLIF(r.height_ru, 0) * 100), 1)
                FROM fttx.racks r
                LEFT JOIN fttx.rooms rm ON r.room_id = rm.room_id
                ORDER BY rm.room_code, r.rack_code
            """)
            
            self.racks_table.setRowCount(0)
            for row_data in cursor.fetchall():
                row = self.racks_table.rowCount()
                self.racks_table.insertRow(row)
                for col, value in enumerate(row_data):
                    self.racks_table.setItem(row, col, QTableWidgetItem(str(value or '')))
            
            cursor.close()
        except Exception as e:
            pass
    
    def update_quick_stats(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT 
                    (SELECT COUNT(*) FROM fttx.buildings) as buildings,
                    (SELECT COUNT(*) FROM fttx.rooms) as rooms,
                    (SELECT COUNT(*) FROM fttx.racks) as racks
            """)
            
            stats = cursor.fetchone()
            if stats:
                buildings, rooms, racks = stats
                stats_text = f"""
<b>Quick Stats:</b><br>
Buildings: {buildings or 0}<br>
Rooms: {rooms or 0}<br>
Racks: {racks or 0}
"""
                self.quick_stats.setText(stats_text)
            cursor.close()
        except:
            pass
    
    def refresh_all(self):
        self.load_buildings()
        QMessageBox.information(self, 'Refreshed', 'Data refreshed successfully!')
    
    # ==========================================
    # CRUD OPERATIONS
    # ==========================================
    
    def add_building(self):
        dialog = AddBuildingDialog(self.plugin)
        if dialog.exec_():
            self.load_buildings()
    
    def edit_building(self):
        selected = self.buildings_table.selectedItems()
        if not selected:
            QMessageBox.warning(self, 'No Selection', 'Please select a building to edit')
            return
        building_id = int(self.buildings_table.item(self.buildings_table.currentRow(), 0).text())
        dialog = EditBuildingDialog(self.plugin, building_id)
        if dialog.exec_():
            self.load_buildings()
    
    def delete_building(self):
        selected = self.buildings_table.selectedItems()
        if not selected:
            QMessageBox.warning(self, 'No Selection', 'Please select a building to delete')
            return
        
        reply = QMessageBox.question(self, 'Confirm Delete', 
            'Delete selected building? This will also delete all associated rooms and racks.',
            QMessageBox.Yes | QMessageBox.No)
        if reply == QMessageBox.Yes:
            try:
                building_id = int(self.buildings_table.item(self.buildings_table.currentRow(), 0).text())
                cursor = self.plugin.db_connection.cursor()
                cursor.execute("DELETE FROM fttx.buildings WHERE building_id = %s", (building_id,))
                self.plugin.db_connection.commit()
                cursor.close()
                QMessageBox.information(self, 'Deleted', 'Building deleted successfully')
                self.load_buildings()
            except Exception as e:
                self.plugin.db_connection.rollback()
                QMessageBox.critical(self, 'Error', f'Failed to delete:\n{str(e)}')
    
    def add_room(self):
        dialog = AddRoomDialog(self.plugin)
        if dialog.exec_():
            self.load_buildings()
    
    def edit_room(self):
        selected = self.rooms_table.selectedItems()
        if not selected:
            QMessageBox.warning(self, 'No Selection', 'Please select a room to edit')
            return
        room_id = int(self.rooms_table.item(self.rooms_table.currentRow(), 0).text())
        dialog = EditRoomDialog(self.plugin, room_id)
        if dialog.exec_():
            self.load_buildings()
    
    def delete_room(self):
        selected = self.rooms_table.selectedItems()
        if not selected:
            QMessageBox.warning(self, 'No Selection', 'Please select a room to delete')
            return
        reply = QMessageBox.question(self, 'Confirm Delete', 'Delete selected room?', QMessageBox.Yes | QMessageBox.No)
        if reply == QMessageBox.Yes:
            try:
                room_id = int(self.rooms_table.item(self.rooms_table.currentRow(), 0).text())
                cursor = self.plugin.db_connection.cursor()
                cursor.execute("DELETE FROM fttx.rooms WHERE room_id = %s", (room_id,))
                self.plugin.db_connection.commit()
                cursor.close()
                QMessageBox.information(self, 'Deleted', 'Room deleted successfully')
                self.load_buildings()
            except Exception as e:
                self.plugin.db_connection.rollback()
                QMessageBox.critical(self, 'Error', f'Failed to delete:\n{str(e)}')
    
    def add_rack(self):
        dialog = AddRackDialog(self.plugin)
        if dialog.exec_():
            self.load_buildings()
    
    def edit_rack(self):
        selected = self.racks_table.selectedItems()
        if not selected:
            QMessageBox.warning(self, 'No Selection', 'Please select a rack to edit')
            return
        rack_id = int(self.racks_table.item(self.racks_table.currentRow(), 0).text())
        dialog = EditRackDialog(self.plugin, rack_id)
        if dialog.exec_():
            self.load_buildings()
    
    def delete_rack(self):
        selected = self.racks_table.selectedItems()
        if not selected:
            QMessageBox.warning(self, 'No Selection', 'Please select a rack to delete')
            return
        reply = QMessageBox.question(self, 'Confirm Delete', 'Delete selected rack?', QMessageBox.Yes | QMessageBox.No)
        if reply == QMessageBox.Yes:
            try:
                rack_id = int(self.racks_table.item(self.racks_table.currentRow(), 0).text())
                cursor = self.plugin.db_connection.cursor()
                cursor.execute("DELETE FROM fttx.racks WHERE rack_id = %s", (rack_id,))
                self.plugin.db_connection.commit()
                cursor.close()
                QMessageBox.information(self, 'Deleted', 'Rack deleted successfully')
                self.load_buildings()
            except Exception as e:
                self.plugin.db_connection.rollback()
                QMessageBox.critical(self, 'Error', f'Failed to delete:\n{str(e)}')


# ==========================================
# ADD/EDIT DIALOGS
# ==========================================

class AddBuildingDialog(QDialog):
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Add Building')
        self.setMinimumWidth(500)
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        form = QFormLayout()
        self.building_code = QLineEdit()
        form.addRow('Building Code*:', self.building_code)
        self.building_name = QLineEdit()
        form.addRow('Building Name:', self.building_name)
        self.street_address = QLineEdit()
        form.addRow('Street Address:', self.street_address)
        self.city = QLineEdit()
        form.addRow('City:', self.city)
        self.building_type = QComboBox()
        self.building_type.addItems(['Central Office', 'POP', 'Data Center', 'Co-location', 'Other'])
        form.addRow('Building Type:', self.building_type)
        self.floors_count = QSpinBox()
        self.floors_count.setRange(1, 100)
        self.floors_count.setValue(1)
        form.addRow('Number of Floors:', self.floors_count)
        self.owner = QLineEdit()
        form.addRow('Owner:', self.owner)
        layout.addLayout(form)
        buttons = QDialogButtonBox(QDialogButtonBox.Save | QDialogButtonBox.Cancel)
        buttons.accepted.connect(self.save)
        buttons.rejected.connect(self.reject)
        layout.addWidget(buttons)
        self.setLayout(layout)
    
    def save(self):
        if not self.building_code.text():
            QMessageBox.warning(self, 'Required', 'Building Code is required')
            return
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                INSERT INTO fttx.buildings 
                (building_code, building_name, street_address, city, building_type, floors_count, owner)
                VALUES (%s, %s, %s, %s, %s, %s, %s)
            """, (self.building_code.text(), self.building_name.text(), self.street_address.text(),
                  self.city.text(), self.building_type.currentText(), self.floors_count.value(), self.owner.text()))
            self.plugin.db_connection.commit()
            cursor.close()
            QMessageBox.information(self, 'Success', 'Building added successfully!')
            self.accept()
        except Exception as e:
            self.plugin.db_connection.rollback()
            QMessageBox.critical(self, 'Error', f'Failed to save:\n{str(e)}')


class EditBuildingDialog(QDialog):
    def __init__(self, plugin, building_id):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.building_id = building_id
        self.setWindowTitle('Edit Building')
        self.setMinimumWidth(500)
        self.init_ui()
        self.load_data()
    
    def init_ui(self):
        layout = QVBoxLayout()
        form = QFormLayout()
        self.building_code = QLineEdit()
        form.addRow('Building Code*:', self.building_code)
        self.building_name = QLineEdit()
        form.addRow('Building Name:', self.building_name)
        self.street_address = QLineEdit()
        form.addRow('Street Address:', self.street_address)
        self.city = QLineEdit()
        form.addRow('City:', self.city)
        self.building_type = QComboBox()
        self.building_type.addItems(['Central Office', 'POP', 'Data Center', 'Co-location', 'Other'])
        form.addRow('Building Type:', self.building_type)
        self.floors_count = QSpinBox()
        self.floors_count.setRange(1, 100)
        form.addRow('Number of Floors:', self.floors_count)
        self.owner = QLineEdit()
        form.addRow('Owner:', self.owner)
        layout.addLayout(form)
        buttons = QDialogButtonBox(QDialogButtonBox.Save | QDialogButtonBox.Cancel)
        buttons.accepted.connect(self.save)
        buttons.rejected.connect(self.reject)
        layout.addWidget(buttons)
        self.setLayout(layout)
    
    def load_data(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT building_code, building_name, street_address, city, building_type, floors_count, owner
                FROM fttx.buildings WHERE building_id = %s
            """, (self.building_id,))
            data = cursor.fetchone()
            if data:
                self.building_code.setText(data[0] or '')
                self.building_name.setText(data[1] or '')
                self.street_address.setText(data[2] or '')
                self.city.setText(data[3] or '')
                self.building_type.setCurrentText(data[4] or 'Other')
                self.floors_count.setValue(data[5] or 1)
                self.owner.setText(data[6] or '')
            cursor.close()
        except Exception as e:
            QMessageBox.warning(self, 'Warning', f'Could not load data:\n{str(e)}')
    
    def save(self):
        if not self.building_code.text():
            QMessageBox.warning(self, 'Required', 'Building Code is required')
            return
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                UPDATE fttx.buildings SET building_code=%s, building_name=%s, street_address=%s,
                       city=%s, building_type=%s, floors_count=%s, owner=%s
                WHERE building_id = %s
            """, (self.building_code.text(), self.building_name.text(), self.street_address.text(),
                  self.city.text(), self.building_type.currentText(), self.floors_count.value(),
                  self.owner.text(), self.building_id))
            self.plugin.db_connection.commit()
            cursor.close()
            QMessageBox.information(self, 'Success', 'Building updated successfully!')
            self.accept()
        except Exception as e:
            self.plugin.db_connection.rollback()
            QMessageBox.critical(self, 'Error', f'Failed to save:\n{str(e)}')


class AddRoomDialog(QDialog):
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Add Telecom Room')
        self.setMinimumWidth(500)
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        form = QFormLayout()
        self.room_code = QLineEdit()
        form.addRow('Room Code*:', self.room_code)
        self.room_name = QLineEdit()
        form.addRow('Room Name:', self.room_name)
        self.building = QComboBox()
        self.load_buildings()
        form.addRow('Building*:', self.building)
        self.floor_number = QSpinBox()
        self.floor_number.setRange(0, 100)
        form.addRow('Floor Number:', self.floor_number)
        self.room_type = QComboBox()
        self.room_type.addItems(['MDF', 'IDF', 'Equipment Room', 'Server Room', 'Telecom Closet'])
        form.addRow('Room Type:', self.room_type)
        self.room_size = QDoubleSpinBox()
        self.room_size.setRange(0, 10000)
        self.room_size.setSuffix(' m²')
        form.addRow('Room Size:', self.room_size)
        self.has_hvac = QCheckBox()
        form.addRow('Has HVAC:', self.has_hvac)
        self.has_ups = QCheckBox()
        form.addRow('Has UPS:', self.has_ups)
        layout.addLayout(form)
        buttons = QDialogButtonBox(QDialogButtonBox.Save | QDialogButtonBox.Cancel)
        buttons.accepted.connect(self.save)
        buttons.rejected.connect(self.reject)
        layout.addWidget(buttons)
        self.setLayout(layout)
    
    def load_buildings(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("SELECT building_id, building_code, building_name FROM fttx.buildings ORDER BY building_code")
            for building_id, code, name in cursor.fetchall():
                display = f'{code} - {name}' if name else code
                self.building.addItem(display, building_id)
            cursor.close()
        except Exception as e:
            QMessageBox.warning(self, 'Warning', f'Could not load buildings:\n{str(e)}')
    
    def save(self):
        if not self.room_code.text():
            QMessageBox.warning(self, 'Required', 'Room Code is required')
            return
        if self.building.currentData() is None:
            QMessageBox.warning(self, 'Required', 'Please select a building')
            return
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                INSERT INTO fttx.rooms 
                (room_code, room_name, building_id, floor_number, room_type, room_size_sqm, has_hvac, has_ups)
                VALUES (%s, %s, %s, %s, %s, %s, %s, %s)
            """, (self.room_code.text(), self.room_name.text(), self.building.currentData(),
                  self.floor_number.value(), self.room_type.currentText(), self.room_size.value(),
                  self.has_hvac.isChecked(), self.has_ups.isChecked()))
            self.plugin.db_connection.commit()
            cursor.close()
            QMessageBox.information(self, 'Success', 'Room added successfully!')
            self.accept()
        except Exception as e:
            self.plugin.db_connection.rollback()
            QMessageBox.critical(self, 'Error', f'Failed to save:\n{str(e)}')


class EditRoomDialog(QDialog):
    def __init__(self, plugin, room_id):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.room_id = room_id
        self.setWindowTitle('Edit Room')
        self.setMinimumWidth(500)
        self.init_ui()
        self.load_data()
    
    def init_ui(self):
        layout = QVBoxLayout()
        form = QFormLayout()
        self.room_code = QLineEdit()
        form.addRow('Room Code*:', self.room_code)
        self.room_name = QLineEdit()
        form.addRow('Room Name:', self.room_name)
        self.building = QComboBox()
        self.load_buildings()
        form.addRow('Building*:', self.building)
        self.floor_number = QSpinBox()
        self.floor_number.setRange(0, 100)
        form.addRow('Floor Number:', self.floor_number)
        self.room_type = QComboBox()
        self.room_type.addItems(['MDF', 'IDF', 'Equipment Room', 'Server Room', 'Telecom Closet'])
        form.addRow('Room Type:', self.room_type)
        self.room_size = QDoubleSpinBox()
        self.room_size.setRange(0, 10000)
        self.room_size.setSuffix(' m²')
        form.addRow('Room Size:', self.room_size)
        self.has_hvac = QCheckBox()
        form.addRow('Has HVAC:', self.has_hvac)
        self.has_ups = QCheckBox()
        form.addRow('Has UPS:', self.has_ups)
        layout.addLayout(form)
        buttons = QDialogButtonBox(QDialogButtonBox.Save | QDialogButtonBox.Cancel)
        buttons.accepted.connect(self.save)
        buttons.rejected.connect(self.reject)
        layout.addWidget(buttons)
        self.setLayout(layout)
    
    def load_buildings(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("SELECT building_id, building_code, building_name FROM fttx.buildings ORDER BY building_code")
            for building_id, code, name in cursor.fetchall():
                display = f'{code} - {name}' if name else code
                self.building.addItem(display, building_id)
            cursor.close()
        except Exception as e:
            pass
    
    def load_data(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT room_code, room_name, building_id, floor_number, room_type, room_size_sqm, has_hvac, has_ups
                FROM fttx.rooms WHERE room_id = %s
            """, (self.room_id,))
            data = cursor.fetchone()
            if data:
                self.room_code.setText(data[0] or '')
                self.room_name.setText(data[1] or '')
                # Set building combo
                for i in range(self.building.count()):
                    if self.building.itemData(i) == data[2]:
                        self.building.setCurrentIndex(i)
                        break
                self.floor_number.setValue(data[3] or 0)
                self.room_type.setCurrentText(data[4] or 'Equipment Room')
                self.room_size.setValue(data[5] or 0)
                self.has_hvac.setChecked(data[6] or False)
                self.has_ups.setChecked(data[7] or False)
            cursor.close()
        except Exception as e:
            QMessageBox.warning(self, 'Warning', f'Could not load data:\n{str(e)}')
    
    def save(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                UPDATE fttx.rooms SET room_code=%s, room_name=%s, building_id=%s, floor_number=%s,
                       room_type=%s, room_size_sqm=%s, has_hvac=%s, has_ups=%s
                WHERE room_id = %s
            """, (self.room_code.text(), self.room_name.text(), self.building.currentData(),
                  self.floor_number.value(), self.room_type.currentText(), self.room_size.value(),
                  self.has_hvac.isChecked(), self.has_ups.isChecked(), self.room_id))
            self.plugin.db_connection.commit()
            cursor.close()
            QMessageBox.information(self, 'Success', 'Room updated successfully!')
            self.accept()
        except Exception as e:
            self.plugin.db_connection.rollback()
            QMessageBox.critical(self, 'Error', f'Failed to save:\n{str(e)}')


class AddRackDialog(QDialog):
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Add Equipment Rack')
        self.setMinimumWidth(500)
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        form = QFormLayout()
        self.rack_code = QLineEdit()
        form.addRow('Rack Code*:', self.rack_code)
        self.rack_name = QLineEdit()
        form.addRow('Rack Name:', self.rack_name)
        self.room = QComboBox()
        self.load_rooms()
        form.addRow('Room*:', self.room)
        self.height_ru = QSpinBox()
        self.height_ru.setRange(1, 52)
        self.height_ru.setValue(42)
        self.height_ru.setSuffix(' RU')
        form.addRow('Height:', self.height_ru)
        self.rack_type = QComboBox()
        self.rack_type.addItems(['Open Frame', 'Enclosed', 'Wall Mount', '2-Post', '4-Post'])
        form.addRow('Rack Type:', self.rack_type)
        layout.addLayout(form)
        buttons = QDialogButtonBox(QDialogButtonBox.Save | QDialogButtonBox.Cancel)
        buttons.accepted.connect(self.save)
        buttons.rejected.connect(self.reject)
        layout.addWidget(buttons)
        self.setLayout(layout)
    
    def load_rooms(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("SELECT room_id, room_code, room_name FROM fttx.rooms ORDER BY room_code")
            for room_id, code, name in cursor.fetchall():
                display = f'{code} - {name}' if name else code
                self.room.addItem(display, room_id)
            cursor.close()
        except Exception as e:
            QMessageBox.warning(self, 'Warning', f'Could not load rooms:\n{str(e)}')
    
    def save(self):
        if not self.rack_code.text():
            QMessageBox.warning(self, 'Required', 'Rack Code is required')
            return
        if self.room.currentData() is None:
            QMessageBox.warning(self, 'Required', 'Please select a room')
            return
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                INSERT INTO fttx.racks 
                (rack_code, rack_name, room_id, height_ru, rack_type, available_ru)
                VALUES (%s, %s, %s, %s, %s, %s)
            """, (self.rack_code.text(), self.rack_name.text(), self.room.currentData(),
                  self.height_ru.value(), self.rack_type.currentText(), self.height_ru.value()))
            self.plugin.db_connection.commit()
            cursor.close()
            QMessageBox.information(self, 'Success', 'Rack added successfully!')
            self.accept()
        except Exception as e:
            self.plugin.db_connection.rollback()
            QMessageBox.critical(self, 'Error', f'Failed to save:\n{str(e)}')


class EditRackDialog(QDialog):
    def __init__(self, plugin, rack_id):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.rack_id = rack_id
        self.setWindowTitle('Edit Rack')
        self.setMinimumWidth(500)
        self.init_ui()
        self.load_data()
    
    def init_ui(self):
        layout = QVBoxLayout()
        form = QFormLayout()
        self.rack_code = QLineEdit()
        form.addRow('Rack Code*:', self.rack_code)
        self.rack_name = QLineEdit()
        form.addRow('Rack Name:', self.rack_name)
        self.room = QComboBox()
        self.load_rooms()
        form.addRow('Room*:', self.room)
        self.height_ru = QSpinBox()
        self.height_ru.setRange(1, 52)
        self.height_ru.setSuffix(' RU')
        form.addRow('Height:', self.height_ru)
        self.rack_type = QComboBox()
        self.rack_type.addItems(['Open Frame', 'Enclosed', 'Wall Mount', '2-Post', '4-Post'])
        form.addRow('Rack Type:', self.rack_type)
        layout.addLayout(form)
        buttons = QDialogButtonBox(QDialogButtonBox.Save | QDialogButtonBox.Cancel)
        buttons.accepted.connect(self.save)
        buttons.rejected.connect(self.reject)
        layout.addWidget(buttons)
        self.setLayout(layout)
    
    def load_rooms(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("SELECT room_id, room_code, room_name FROM fttx.rooms ORDER BY room_code")
            for room_id, code, name in cursor.fetchall():
                display = f'{code} - {name}' if name else code
                self.room.addItem(display, room_id)
            cursor.close()
        except Exception as e:
            pass
    
    def load_data(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT rack_code, rack_name, room_id, height_ru, rack_type
                FROM fttx.racks WHERE rack_id = %s
            """, (self.rack_id,))
            data = cursor.fetchone()
            if data:
                self.rack_code.setText(data[0] or '')
                self.rack_name.setText(data[1] or '')
                for i in range(self.room.count()):
                    if self.room.itemData(i) == data[2]:
                        self.room.setCurrentIndex(i)
                        break
                self.height_ru.setValue(data[3] or 42)
                self.rack_type.setCurrentText(data[4] or 'Enclosed')
            cursor.close()
        except Exception as e:
            QMessageBox.warning(self, 'Warning', f'Could not load data:\n{str(e)}')
    
    def save(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                UPDATE fttx.racks SET rack_code=%s, rack_name=%s, room_id=%s, height_ru=%s, rack_type=%s
                WHERE rack_id = %s
            """, (self.rack_code.text(), self.rack_name.text(), self.room.currentData(),
                  self.height_ru.value(), self.rack_type.currentText(), self.rack_id))
            self.plugin.db_connection.commit()
            cursor.close()
            QMessageBox.information(self, 'Success', 'Rack updated successfully!')
            self.accept()
        except Exception as e:
            self.plugin.db_connection.rollback()
            QMessageBox.critical(self, 'Error', f'Failed to save:\n{str(e)}')