# itu_fttx_plugin/gui/dialogs.py

from qgis.PyQt.QtCore import Qt, QDate
from qgis.PyQt.QtWidgets import (
    QDialog, QMessageBox, QVBoxLayout, QHBoxLayout, 
    QLabel, QPushButton, QLineEdit, QSpinBox, QComboBox, 
    QGroupBox, QFormLayout, QDialogButtonBox, QDoubleSpinBox,
    QTableWidget, QTableWidgetItem, QHeaderView, QRadioButton,
    QCheckBox, QButtonGroup, QTabWidget, QWidget, QFrame,
    QTextEdit, QCalendarWidget, QScrollArea, QSplitter, 
    QGraphicsView, QGraphicsScene
)
from qgis.PyQt.QtGui import QPen
import psycopg2
import math
from datetime import datetime

from ..config.constants import SQL_SELECT_OLTS, SQL_INSERT_PON_PORT
from ..core.planners import DocumentationEngine 


# ==========================================
# ADD PON PORT DIALOG 
# ==========================================

class AddPONPortDialog(QDialog):
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Add PON Port')
        self.setMinimumWidth(550)
        self.init_ui()
        self.load_olts()
    
    def init_ui(self):
        layout = QVBoxLayout()
        header = QLabel('➕ Add PON Port to OLT')
        header.setStyleSheet('font-size: 14pt; font-weight: bold;')
        layout.addWidget(header)
        info = QLabel('PON ports are the physical connections where fiber cables connect to the OLT.')
        info.setWordWrap(True)
        layout.addWidget(info)
        
        form_group = QGroupBox('PON Port Configuration')
        form_layout = QFormLayout()
        
        self.olt_combo = QComboBox()
        form_layout.addRow('Select OLT:', self.olt_combo)
        
        self.port_number = QSpinBox()
        self.port_number.setRange(1, 128)
        self.port_number.setValue(1)
        form_layout.addRow('Port Number:', self.port_number)
        
        self.port_label = QLineEdit()
        self.port_label.setPlaceholderText('e.g., PON-1/1/1')
        form_layout.addRow('Port Label:', self.port_label)
        
        self.budget_class = QComboBox()
        self.budget_class.addItems(['A', 'B', 'B+', 'C', 'C+', 'C++'])
        self.budget_class.setCurrentText('B+')
        form_layout.addRow('Optical Budget Class:', self.budget_class)
        
        self.max_reach = QSpinBox()
        self.max_reach.setRange(1, 60)
        self.max_reach.setValue(20)
        self.max_reach.setSuffix(' km')
        form_layout.addRow('Max Reach:', self.max_reach)
        
        self.max_onus = QSpinBox()
        self.max_onus.setRange(1, 128)
        self.max_onus.setValue(64)
        form_layout.addRow('Max ONUs:', self.max_onus)
        
        self.port_status = QComboBox()
        self.port_status.addItems(['active', 'inactive', 'maintenance'])
        form_layout.addRow('Port Status:', self.port_status)
        
        form_group.setLayout(form_layout)
        layout.addWidget(form_group)
        
        auto_btn = QPushButton('🔄 Auto-Generate Port Label')
        auto_btn.clicked.connect(self.auto_generate_label)
        layout.addWidget(auto_btn)
        
        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.add_pon_port)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
        self.setLayout(layout)
    
    def load_olts(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute(SQL_SELECT_OLTS)
            
            for olt_id, olt_code, pon_standard in cursor.fetchall():
                self.olt_combo.addItem(f"{olt_code} ({pon_standard})", olt_id)
            cursor.close()
            
            if self.olt_combo.count() == 0:
                QMessageBox.warning(self, 'No OLTs', 'No operational OLTs found. Add an OLT first.')
                self.reject()
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Failed to load OLTs:\n{str(e)}')
            self.reject()
    
    def auto_generate_label(self):
        olt_text = self.olt_combo.currentText()
        port_num = self.port_number.value()
        olt_code = olt_text.split('(')[0].strip()
        self.port_label.setText(f"{olt_code}-P{port_num}")
    
    def add_pon_port(self):
        try:
            olt_id = self.olt_combo.currentData()
            port_number = self.port_number.value()
            port_label = self.port_label.text()
            
            if not port_label:
                QMessageBox.warning(self, 'Missing Info', 'Please enter a port label')
                return
            
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT COUNT(*) FROM fttx.itu_olt_pon_ports
                WHERE olt_id = %s AND port_number = %s
            """, (olt_id, port_number))
            
            if cursor.fetchone()[0] > 0:
                QMessageBox.warning(self, 'Exists', f'Port {port_number} already exists')
                cursor.close()
                return
            
            cursor.execute(SQL_INSERT_PON_PORT, 
                           (olt_id, port_number, port_label, self.budget_class.currentText(),
                            self.max_reach.value(), self.max_onus.value(), self.port_status.currentText()))
            
            self.plugin.db_connection.commit()
            cursor.close()
            
            QMessageBox.information(self, '✔ Success', 
                f'PON port {port_label} added!\n\nRefresh PON Ports layer to see it.')
            self.accept()
        except Exception as e:
            self.plugin.db_connection.rollback()
            QMessageBox.critical(self, 'Error', f'Failed:\n{str(e)}')


# ==========================================
# LABEL GENERATOR
# ==========================================

class LabelGenerator(QDialog):
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Cable & Equipment Label Generator')
        self.setMinimumSize(900, 700)
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        self.tabs = QTabWidget()
        self.tabs.addTab(self.create_cable_label_tab(), 'Cable Labels')
        self.tabs.addTab(self.create_equipment_label_tab(), 'Equipment Labels')
        self.tabs.addTab(self.create_port_label_tab(), 'Port Labels')
        self.tabs.addTab(self.create_splice_label_tab(), 'Splice Labels')
        layout.addWidget(self.tabs)
        
        button_layout = QHBoxLayout()
        generate_btn = QPushButton('📋 Generate Batch')
        generate_btn.clicked.connect(self.generate_batch)
        button_layout.addWidget(generate_btn)
        
        export_btn = QPushButton('💾 Export')
        export_btn.clicked.connect(self.export_labels)
        button_layout.addWidget(export_btn)
        
        print_btn = QPushButton('🖨️ Print')
        print_btn.clicked.connect(self.print_labels)
        button_layout.addWidget(print_btn)
        
        button_layout.addStretch()
        
        close_btn = QPushButton('Close')
        close_btn.clicked.connect(self.reject)
        button_layout.addWidget(close_btn)
        
        layout.addLayout(button_layout)
        self.setLayout(layout)
        
    def create_cable_label_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        form = QFormLayout()
        
        self.cable_selector = QComboBox()
        self.load_cables()
        self.cable_selector.currentTextChanged.connect(self.update_cable_preview)
        form.addRow('Select Cable:', self.cable_selector)
        
        self.cable_label_text = QLineEdit()
        self.cable_label_text.textChanged.connect(self.update_cable_preview)
        form.addRow('Custom Text:', self.cable_label_text)
        
        self.include_cable_qr = QCheckBox('Include QR Code')
        self.include_cable_qr.setChecked(True)
        form.addRow('', self.include_cable_qr)
        
        layout.addLayout(form)
        
        preview_group = QGroupBox('Preview')
        preview_layout = QVBoxLayout()
        self.cable_preview = QLabel('Label preview will appear here')
        self.cable_preview.setStyleSheet('border: 1px solid #ccc; padding: 20px; min-height: 150px;')
        self.cable_preview.setAlignment(Qt.AlignCenter)
        preview_layout.addWidget(self.cable_preview)
        preview_group.setLayout(preview_layout)
        layout.addWidget(preview_group)
        
        layout.addStretch()
        widget.setLayout(layout)
        return widget
        
    def load_cables(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("SELECT cable_code, cable_type FROM fttx.itu_fiber_cables ORDER BY cable_code")
            
            self.cable_selector.addItem('-- Select Cable --', None)
            for cable_code, cable_type in cursor.fetchall():
                self.cable_selector.addItem(f'{cable_code} ({cable_type})', cable_code)
            
            cursor.close()
        except Exception as e:
            QMessageBox.warning(self, 'Warning', f'Could not load cables:\n{str(e)}')
    
    def update_cable_preview(self):
        cable = self.cable_selector.currentText()
        custom_text = self.cable_label_text.text()
        preview_text = f"Cable: {cable}\n{custom_text}" if custom_text else f"Cable: {cable}"
        self.cable_preview.setText(preview_text)
    
    def create_equipment_label_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        form = QFormLayout()

        self.equipment_selector = QComboBox()
        self.load_equipment()
        self.equipment_selector.currentTextChanged.connect(self.update_equipment_preview)
        form.addRow('Select Equipment:', self.equipment_selector)

        self.equipment_label_text = QLineEdit()
        self.equipment_label_text.textChanged.connect(self.update_equipment_preview)
        form.addRow('Custom Text:', self.equipment_label_text)

        self.include_equip_qr = QCheckBox('Include QR Code')
        self.include_equip_qr.setChecked(True)
        form.addRow('', self.include_equip_qr)

        layout.addLayout(form)

        preview_group = QGroupBox('Preview')
        preview_layout = QVBoxLayout()
        self.equipment_preview = QLabel('Label preview will appear here')
        self.equipment_preview.setStyleSheet('border: 1px solid #ccc; padding: 20px; min-height: 150px;')
        self.equipment_preview.setAlignment(Qt.AlignCenter)
        preview_layout.addWidget(self.equipment_preview)
        preview_group.setLayout(preview_layout)
        layout.addWidget(preview_group)

        layout.addStretch()
        widget.setLayout(layout)
        return widget

    def load_equipment(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT olt_code, manufacturer, model FROM fttx.itu_olt_equipment
                ORDER BY olt_code
            """)
            self.equipment_selector.addItem('-- Select Equipment --', None)
            for code, mfr, model in cursor.fetchall():
                self.equipment_selector.addItem(f'{code} ({mfr} {model})', code)
            cursor.close()
        except Exception as e:
            QMessageBox.warning(self, 'Warning', f'Could not load equipment:\n{str(e)}')

    def create_port_label_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        form = QFormLayout()

        self.port_selector = QComboBox()
        self.load_ports()
        self.port_selector.currentTextChanged.connect(self.update_port_preview)
        form.addRow('Select Port:', self.port_selector)

        self.port_label_text = QLineEdit()
        self.port_label_text.textChanged.connect(self.update_port_preview)
        form.addRow('Custom Text:', self.port_label_text)

        self.include_port_qr = QCheckBox('Include QR Code')
        self.include_port_qr.setChecked(True)
        form.addRow('', self.include_port_qr)

        layout.addLayout(form)

        preview_group = QGroupBox('Preview')
        preview_layout = QVBoxLayout()
        self.port_preview = QLabel('Label preview will appear here')
        self.port_preview.setStyleSheet('border: 1px solid #ccc; padding: 20px; min-height: 150px;')
        self.port_preview.setAlignment(Qt.AlignCenter)
        preview_layout.addWidget(self.port_preview)
        preview_group.setLayout(preview_layout)
        layout.addWidget(preview_group)

        layout.addStretch()
        widget.setLayout(layout)
        return widget

    def load_ports(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT p.port_label, o.olt_code, p.port_number
                FROM fttx.itu_olt_pon_ports p
                JOIN fttx.itu_olt_equipment o ON p.olt_id = o.olt_id
                ORDER BY o.olt_code, p.port_number
            """)
            self.port_selector.addItem('-- Select Port --', None)
            for label, olt_code, port_num in cursor.fetchall():
                self.port_selector.addItem(f'{label} (OLT: {olt_code})', label)
            cursor.close()
        except Exception as e:
            QMessageBox.warning(self, 'Warning', f'Could not load ports:\n{str(e)}')

    def create_splice_label_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        form = QFormLayout()

        self.splice_selector = QComboBox()
        self.load_splices()
        self.splice_selector.currentTextChanged.connect(self.update_splice_preview)
        form.addRow('Select Splice Point:', self.splice_selector)

        self.splice_label_text = QLineEdit()
        self.splice_label_text.textChanged.connect(self.update_splice_preview)
        form.addRow('Custom Text:', self.splice_label_text)

        self.include_splice_qr = QCheckBox('Include QR Code')
        self.include_splice_qr.setChecked(True)
        form.addRow('', self.include_splice_qr)

        layout.addLayout(form)

        preview_group = QGroupBox('Preview')
        preview_layout = QVBoxLayout()
        self.splice_preview = QLabel('Label preview will appear here')
        self.splice_preview.setStyleSheet('border: 1px solid #ccc; padding: 20px; min-height: 150px;')
        self.splice_preview.setAlignment(Qt.AlignCenter)
        preview_layout.addWidget(self.splice_preview)
        preview_group.setLayout(preview_layout)
        layout.addWidget(preview_group)

        layout.addStretch()
        widget.setLayout(layout)
        return widget

    def load_splices(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT splice_code, closure_type FROM fttx.itu_splice_points
                ORDER BY splice_code
            """)
            self.splice_selector.addItem('-- Select Splice Point --', None)
            for code, closure_type in cursor.fetchall():
                self.splice_selector.addItem(f'{code} ({closure_type or "N/A"})', code)
            cursor.close()
        except Exception as e:
            QMessageBox.warning(self, 'Warning', f'Could not load splice points:\n{str(e)}')

    def update_equipment_preview(self):
        equipment = self.equipment_selector.currentText()
        custom_text = self.equipment_label_text.text()
        preview_text = f"Equipment: {equipment}\n{custom_text}" if custom_text else f"Equipment: {equipment}"
        self.equipment_preview.setText(preview_text)

    def update_port_preview(self):
        port = self.port_selector.currentText()
        custom_text = self.port_label_text.text()
        preview_text = f"Port: {port}\n{custom_text}" if custom_text else f"Port: {port}"
        self.port_preview.setText(preview_text)

    def update_splice_preview(self):
        splice = self.splice_selector.currentText()
        custom_text = self.splice_label_text.text()
        preview_text = f"Splice: {splice}\n{custom_text}" if custom_text else f"Splice: {splice}"
        self.splice_preview.setText(preview_text)

    def generate_batch(self):
        """Generate labels for all items of the current tab type"""
        current_tab = self.tabs.currentIndex()
        tab_names = ['Cable', 'Equipment', 'Port', 'Splice']
        tab_name = tab_names[current_tab]

        selectors = [self.cable_selector, self.equipment_selector,
                     self.port_selector, self.splice_selector]
        selector = selectors[current_tab]

        count = selector.count()
        if count <= 1:  # Only the placeholder item
            QMessageBox.warning(self, 'No Items', f'No {tab_name.lower()} items found to generate labels for.')
            return

        QMessageBox.information(self, 'Batch Generated',
            f'Generated {count - 1} {tab_name.lower()} label(s).\n\n'
            f'Use Export or Print to output the labels.')

    def export_labels(self):
        """Export labels to CSV file"""
        from qgis.PyQt.QtWidgets import QFileDialog
        filepath, _ = QFileDialog.getSaveFileName(
            self, 'Export Labels', '', 'CSV Files (*.csv);;All Files (*)')
        if not filepath:
            return

        current_tab = self.tabs.currentIndex()
        selectors = [self.cable_selector, self.equipment_selector,
                     self.port_selector, self.splice_selector]
        selector = selectors[current_tab]

        try:
            with open(filepath, 'w', encoding='utf-8') as f:
                f.write('Item,Label\n')
                for i in range(1, selector.count()):  # Skip placeholder
                    f.write(f'"{selector.itemText(i)}",""\n')
            QMessageBox.information(self, 'Exported', f'Labels exported to:\n{filepath}')
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Export failed:\n{str(e)}')

    def print_labels(self):
        """Print labels using system print dialog"""
        current_tab = self.tabs.currentIndex()
        previews = [self.cable_preview, self.equipment_preview,
                    self.port_preview, self.splice_preview]
        preview_text = previews[current_tab].text()

        if not preview_text or 'will appear here' in preview_text:
            QMessageBox.warning(self, 'Nothing to Print', 'Please select an item first.')
            return

        QMessageBox.information(self, 'Print',
            f'Label ready for printing:\n\n{preview_text}\n\n'
            f'Connect a label printer to print physical labels.')


# ==========================================
# NETWORK DASHBOARD
# ==========================================

class NetworkDashboard(QDialog):
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Network Dashboard')
        self.setMinimumSize(1000, 700)
        self.details_text = QTextEdit() 
        self.init_ui()
        self.load_dashboard_data()
        
    def init_ui(self):
        layout = QVBoxLayout()
        header = QLabel('📊 Network Overview Dashboard')
        header.setStyleSheet('font-size: 14pt; font-weight: bold; background-color: #00BCD4; color: white; padding: 10px;')
        layout.addWidget(header)
        
        card_layout = QHBoxLayout()
        card_layout.addWidget(self.create_stat_card('Total ONUs', '0', '#2196F3'))
        card_layout.addWidget(self.create_stat_card('Active ONUs', '0', '#4CAF50'))
        card_layout.addWidget(self.create_stat_card('Total Cable (km)', '0.00', '#FF9800'))
        card_layout.addWidget(self.create_stat_card('Splitters', '0', '#9C27B0'))
        layout.addLayout(card_layout)

        self.details_text.setReadOnly(True)
        layout.addWidget(self.details_text)
        
        button_box = QDialogButtonBox(QDialogButtonBox.Close)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
        self.setLayout(layout)
        
    def create_stat_card(self, title, value, color):
        frame = QFrame()
        frame.setFrameStyle(QFrame.Box | QFrame.Raised)
        frame.setStyleSheet(f'background-color: {color}; color: white; border-radius: 5px;')
        layout = QVBoxLayout()
        title_label = QLabel(title)
        title_label.setStyleSheet('font-size: 10pt; font-weight: bold;')
        title_label.setAlignment(Qt.AlignCenter)
        layout.addWidget(title_label)
        value_label = QLabel(value)
        value_label.setStyleSheet('font-size: 20pt; font-weight: bold;')
        value_label.setAlignment(Qt.AlignCenter)
        value_label.setObjectName(f'value_{title.replace(" ", "_").lower()}')
        layout.addWidget(value_label)
        frame.setLayout(layout)
        return frame
        
    def load_dashboard_data(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            
            cursor.execute("""
                SELECT 
                    (SELECT COUNT(*) FROM fttx.itu_onu_equipment) as total_onus,
                    (SELECT COUNT(*) FROM fttx.itu_onu_equipment WHERE onu_status = 'online') as active_onus,
                    (SELECT SUM(cable_length_m) / 1000.0 FROM fttx.itu_fiber_cables) as total_cable_km,
                    (SELECT COUNT(*) FROM fttx.itu_odn_nodes WHERE node_type LIKE '%Splitter%') as splitters,
                    (SELECT COUNT(*) FROM fttx.itu_olt_equipment) as olts,
                    (SELECT COUNT(*) FROM fttx.itu_olt_pon_ports WHERE port_status = 'active') as active_ports
            """)
            
            stats = cursor.fetchone()
            
            if stats:
                total_onus, active_onus, cable_km, splitters, olts, ports = stats
                self.update_card_value('total_onus', total_onus or 0)
                self.update_card_value('active_onus', active_onus or 0)
                self.update_card_value('total_cable_(km)', f'{cable_km or 0:.2f}')
                self.update_card_value('splitters', splitters or 0)
                
                details = f"""
<h2>Network Overview</h2>
<h3>Equipment Summary:</h3>
<ul>
<li><b>OLTs:</b> {olts or 0}</li>
<li><b>Active PON Ports:</b> {ports or 0}</li>
<li><b>Total ONUs:</b> {total_onus or 0}</li>
<li><b>Active ONUs:</b> {active_onus or 0} ({(active_onus or 0) / (total_onus or 1) * 100:.1f}%)</li>
<li><b>Splitters:</b> {splitters or 0}</li>
</ul>
<h3>Infrastructure:</h3>
<ul>
<li><b>Total Fiber Cable:</b> {cable_km or 0:.2f} km</li>
</ul>
<h3>Network Health:</h3>
<ul>
<li><b>ONU Online Rate:</b> {(active_onus or 0) / (total_onus or 1) * 100:.1f}%</li>
<li><b>Status:</b> {'✔ Good' if (active_onus or 0) / (total_onus or 1) > 0.9 else '⚠ Check Offline Units'}</li>
</ul>
"""
                self.details_text.setHtml(details)
            cursor.close()
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Failed to load dashboard data:\n{str(e)}')
            
    def update_card_value(self, card_name, value):
        for widget in self.findChildren(QLabel):
            if widget.objectName() == f'value_{card_name}':
                widget.setText(str(value))
                break


# ==========================================
# DOCUMENTATION GENERATOR 
# ==========================================

class DocumentationGenerator(QDialog):
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Documentation Generator')
        self.setMinimumSize(800, 600)
        
        self.preview = QTextEdit()
        self.doc_type = QButtonGroup() 
        
        self.radio_design = QRadioButton()
        self.radio_asbuilt = QRadioButton()
        self.radio_test = QRadioButton()
        self.radio_maint = QRadioButton()

        self.doc_type.addButton(self.radio_design, 1)
        self.doc_type.addButton(self.radio_asbuilt, 2)
        self.doc_type.addButton(self.radio_test, 3)
        self.doc_type.addButton(self.radio_maint, 4)
        
        self.init_ui()
        
    def init_ui(self):
        layout = QVBoxLayout()
        header = QLabel('📄 Network Documentation Generator')
        header.setStyleSheet('font-size: 14pt; font-weight: bold; background-color: #673AB7; color: white; padding: 10px;')
        layout.addWidget(header)
        
        type_group = QGroupBox('Document Type')
        type_layout = QVBoxLayout()
        
        self.radio_design.setText('Network Design Report')
        self.radio_asbuilt.setText('As-Built Documentation')
        self.radio_test.setText('Test & Acceptance Report')
        self.radio_maint.setText('Maintenance Guide')

        type_layout.addWidget(self.radio_design)
        type_layout.addWidget(self.radio_asbuilt)
        type_layout.addWidget(self.radio_test)
        type_layout.addWidget(self.radio_maint)
        
        self.radio_design.setChecked(True)
        type_group.setLayout(type_layout)
        layout.addWidget(type_group)
        
        generate_btn = QPushButton('Generate Preview')
        generate_btn.clicked.connect(self.generate_documentation)
        layout.addWidget(generate_btn)
        
        layout.addWidget(QLabel('Preview:'))
        self.preview.setReadOnly(True)
        layout.addWidget(self.preview)

        button_box = QDialogButtonBox(QDialogButtonBox.Close)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)

        self.setLayout(layout)
    
    def generate_documentation(self):
        doc_type_id = self.doc_type.checkedId()
        doc_engine = DocumentationEngine(self.plugin.db_connection)
        
        if doc_type_id == 1:
            doc = doc_engine.generate_network_design()
        elif doc_type_id == 2:
            doc = doc_engine.generate_asbuilt()
        elif doc_type_id == 3:
            doc = doc_engine.generate_test_report()
        else:
            doc = doc_engine.generate_maintenance_guide()
        
        self.preview.setText(doc)


# ==========================================
# OTDR ANALYZER
# ==========================================

class OTDRAnalyzer(QDialog):
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow()) 
        self.plugin = plugin
        self.setWindowTitle('🔬 OTDR Trace Analyzer')
        self.setMinimumSize(1000, 800)
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        header = QLabel('OTDR Trace Analyzer')
        header.setStyleSheet('font-size: 14pt; font-weight: bold; background-color: #3F51B5; color: white; padding: 10px;')
        layout.addWidget(header)
        
        main_splitter = QSplitter(Qt.Vertical)

        trace_widget = QWidget()
        trace_layout = QVBoxLayout(trace_widget)
        self.trace_view = QGraphicsView()
        self.trace_scene = QGraphicsScene()
        self.trace_view.setScene(self.trace_scene)
        trace_layout.addWidget(self.trace_view)
        
        controls = QHBoxLayout()
        load_btn = QPushButton('📂 Load Trace (.sor)')
        load_btn.clicked.connect(self.load_trace)
        controls.addWidget(load_btn)
        analyze_btn = QPushButton('🔬 Analyze Events')
        analyze_btn.clicked.connect(self.analyze_trace)
        controls.addWidget(analyze_btn)
        controls.addStretch()
        trace_layout.addLayout(controls)

        main_splitter.addWidget(trace_widget)
        
        summary_widget = QWidget()
        summary_layout = QVBoxLayout(summary_widget)
        summary_layout.addWidget(QLabel('<b>Event Table and Summary:</b>'))
        self.event_table = QTableWidget()
        self.event_table.setColumnCount(6)
        self.event_table.setHorizontalHeaderLabels(['Event', 'Distance (m)', 'Loss (dB)', 'Reflectance (dB)', 'Type', 'Status'])
        self.event_table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        summary_layout.addWidget(self.event_table)

        main_splitter.addWidget(summary_widget)
        main_splitter.setSizes([600, 200])

        layout.addWidget(main_splitter)
        
        button_box = QDialogButtonBox(QDialogButtonBox.Close)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
        self.setLayout(layout)
    
    def load_trace(self):
        """Load an OTDR trace from a SOR file or display test data from DB"""
        from qgis.PyQt.QtWidgets import QFileDialog

        # Try to load from file first
        filepath, _ = QFileDialog.getOpenFileName(
            self, 'Load OTDR Trace', '', 'SOR Files (*.sor);;All Files (*)')

        if filepath:
            # SOR file format is binary and proprietary - display file info
            import os
            file_size = os.path.getsize(filepath)
            filename = os.path.basename(filepath)
            self.trace_scene.clear()
            info_text = self.trace_scene.addText(f'Loaded: {filename} ({file_size} bytes)')
            info_text.setPos(50, 30)
            note = self.trace_scene.addText('SOR binary parsing requires pySOR library.\nDisplaying simulated trace.')
            note.setPos(50, 60)
            self.draw_placeholder_trace()
            QMessageBox.information(self, 'File Loaded',
                f'Loaded file: {filename}\n\n'
                f'Note: Full SOR binary parsing requires the pySOR library.\n'
                f'Displaying simulated trace visualization.')
        else:
            # Load test results from database
            try:
                cursor = self.plugin.db_connection.cursor()
                cursor.execute("""
                    SELECT test_code, cable_id, test_type, wavelength_nm,
                           loss_db, return_loss_db, test_result
                    FROM fttx.test_results
                    WHERE test_type = 'OTDR'
                    ORDER BY test_date DESC
                    LIMIT 10
                """)
                results = cursor.fetchall()
                cursor.close()

                if results:
                    self.event_table.setRowCount(len(results))
                    for i, row in enumerate(results):
                        self.event_table.setItem(i, 0, QTableWidgetItem(str(row[0] or '')))
                        self.event_table.setItem(i, 1, QTableWidgetItem(str(row[3] or '')))
                        self.event_table.setItem(i, 2, QTableWidgetItem(f'{row[4]:.3f}' if row[4] else ''))
                        self.event_table.setItem(i, 3, QTableWidgetItem(f'{row[5]:.1f}' if row[5] else ''))
                        self.event_table.setItem(i, 4, QTableWidgetItem(str(row[2] or '')))
                        self.event_table.setItem(i, 5, QTableWidgetItem(str(row[6] or '')))
                    self.draw_placeholder_trace()
                else:
                    QMessageBox.information(self, 'No Data', 'No OTDR test results found in database.')
                    self.draw_placeholder_trace()
            except Exception as e:
                QMessageBox.warning(self, 'Warning', f'Could not load test data:\n{str(e)}')
                self.draw_placeholder_trace()

    def analyze_trace(self):
        """Analyze events from the loaded trace data"""
        row_count = self.event_table.rowCount()
        if row_count == 0:
            QMessageBox.warning(self, 'No Data', 'Please load a trace first.')
            return

        # Analyze event table data
        total_loss = 0.0
        event_count = 0
        pass_count = 0
        fail_count = 0

        for row in range(row_count):
            loss_item = self.event_table.item(row, 2)
            status_item = self.event_table.item(row, 5)
            if loss_item and loss_item.text():
                try:
                    total_loss += float(loss_item.text())
                    event_count += 1
                except ValueError:
                    pass
            if status_item:
                if status_item.text().lower() == 'pass':
                    pass_count += 1
                elif status_item.text().lower() == 'fail':
                    fail_count += 1

        avg_loss = total_loss / event_count if event_count > 0 else 0
        summary = (
            f'Trace Analysis Results:\n\n'
            f'Events analyzed: {row_count}\n'
            f'Total loss: {total_loss:.3f} dB\n'
            f'Average loss per event: {avg_loss:.3f} dB\n'
            f'Pass: {pass_count}, Fail: {fail_count}\n\n'
            f'Overall: {"PASS" if fail_count == 0 else "FAIL"}'
        )
        QMessageBox.information(self, 'Analysis Complete', summary)
        
    def draw_placeholder_trace(self):
        self.trace_scene.clear()
        self.trace_scene.addText("Placeholder OTDR Trace").setPos(50, 50)
        self.trace_scene.addLine(50, 250, 400, 250, QPen(Qt.gray, 2))
        self.trace_scene.addLine(50, 250, 400, 150, QPen(Qt.blue, 2))
        self.event_table.setRowCount(2)
        self.event_table.setItem(0, 0, QTableWidgetItem("Splice"))
        self.event_table.setItem(1, 0, QTableWidgetItem("Connector"))


# ==========================================
# MANUAL TEST ENTRY DIALOG
# ==========================================

class ManualTestEntryDialog(QDialog):
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Manual Test Result Entry')
        self.setMinimumWidth(400)
        self.init_ui()

    def init_ui(self):
        layout = QVBoxLayout()
        header = QLabel('Manual Test Result Entry')
        header.setStyleSheet('font-size: 14pt; font-weight: bold;')
        layout.addWidget(header)
        
        form_group = QGroupBox('Test Parameters')
        form_layout = QFormLayout()

        self.test_code = QLineEdit()
        form_layout.addRow('Test Code:', self.test_code)
        
        self.cable = QComboBox()
        self.load_cables()
        form_layout.addRow('Cable:', self.cable)
        
        self.test_type = QComboBox()
        self.test_type.addItems(['OTDR', 'Power Meter', 'Visual Fault Locator'])
        form_layout.addRow('Test Type:', self.test_type)
        
        self.wavelength = QSpinBox()
        self.wavelength.setRange(1310, 1650)
        self.wavelength.setSingleStep(10)
        self.wavelength.setValue(1550)
        self.wavelength.setSuffix(' nm')
        form_layout.addRow('Wavelength:', self.wavelength)
        
        self.loss = QDoubleSpinBox()
        self.loss.setRange(0, 50)
        self.loss.setSingleStep(0.01)
        self.loss.setDecimals(3)
        self.loss.setSuffix(' dB')
        form_layout.addRow('Loss:', self.loss)
        
        self.return_loss = QDoubleSpinBox()
        self.return_loss.setRange(0, 70)
        self.return_loss.setSingleStep(1)
        self.return_loss.setDecimals(1)
        self.return_loss.setSuffix(' dB')
        form_layout.addRow('Return Loss:', self.return_loss)
        
        self.technician = QLineEdit()
        form_layout.addRow('Technician:', self.technician)
        
        self.test_result = QComboBox()
        self.test_result.addItems(['Pass', 'Fail', 'Inconclusive'])
        self.test_result.setCurrentText('Pass')
        form_layout.addRow('Result:', self.test_result)
        
        form_group.setLayout(form_layout)
        layout.addWidget(form_group)
        
        button_box = QDialogButtonBox(QDialogButtonBox.Save | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.save_test)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
        self.setLayout(layout)

    def load_cables(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            cursor.execute("""
                SELECT cable_id, cable_code
                FROM fttx.itu_fiber_cables
                ORDER BY cable_code
            """)
            
            for cable_id, cable_code in cursor.fetchall():
                self.cable.addItem(cable_code, cable_id)
            
            cursor.close()
        except Exception as e:
            pass
    
    def save_test(self):
        try:
            cursor = self.plugin.db_connection.cursor()
            
            cursor.execute("""
                INSERT INTO fttx.test_results (
                    test_code, cable_id, test_type, wavelength_nm,
                    loss_db, return_loss_db, technician_name, test_result
                ) VALUES (%s, %s, %s, %s, %s, %s, %s, %s)
            """, (
                self.test_code.text() or None,
                self.cable.currentData(),
                self.test_type.currentText(),
                self.wavelength.value(),
                self.loss.value(),
                self.return_loss.value(),
                self.technician.text() or None,
                self.test_result.currentText()
            ))
            
            self.plugin.db_connection.commit()
            cursor.close()
            
            QMessageBox.information(self, '✔ Saved', 'Test result saved successfully!')
            self.accept()
            
        except Exception as e:
            self.plugin.db_connection.rollback()
            QMessageBox.critical(self, 'Error', f'Failed to save test result:\n{str(e)}')