# itu_fttx_plugin/gui/data_import.py

from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import (
    QDialog, QMessageBox, QVBoxLayout, QHBoxLayout, 
    QLabel, QPushButton, QLineEdit, QComboBox, 
    QGroupBox, QFormLayout, QDialogButtonBox, QTextEdit, 
    QCheckBox, QTabWidget, QWidget, QFileDialog,
    QTableWidget, QTableWidgetItem, QHeaderView, QProgressBar,
    QSpinBox
)
from qgis.core import (
    QgsVectorLayer, QgsProject, QgsFeature, QgsGeometry,
    QgsPointXY, QgsCoordinateReferenceSystem, QgsCoordinateTransform,
    QgsWkbTypes
)
import os
import csv
import json
import psycopg2
from datetime import datetime


class DataImportManager(QDialog):
    """Import network data from various sources into FTTX database"""
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('📥 Network Data Import Manager')
        self.setMinimumSize(1000, 700)
        self.imported_data = {}
        self.field_mappings = {}
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        
        header = QLabel('📥 Network Data Import Manager')
        header.setStyleSheet('font-size: 14pt; font-weight: bold; background-color: #009688; color: white; padding: 10px;')
        layout.addWidget(header)
        
        info = QLabel('Import network data from CSV, Excel, Shapefile, KML, or existing QGIS layers.')
        info.setWordWrap(True)
        layout.addWidget(info)
        
        # Main tabs
        self.tabs = QTabWidget()
        
        # Tab 1: Source Selection
        self.tabs.addTab(self.create_source_tab(), '1️⃣ Select Source')
        
        # Tab 2: Field Mapping
        self.tabs.addTab(self.create_mapping_tab(), '2️⃣ Map Fields')
        
        # Tab 3: Validation & Preview
        self.tabs.addTab(self.create_validation_tab(), '3️⃣ Validate')
        
        # Tab 4: Import
        self.tabs.addTab(self.create_import_tab(), '4️⃣ Import')
        
        layout.addWidget(self.tabs)
        
        # Navigation buttons
        nav_layout = QHBoxLayout()
        
        self.prev_btn = QPushButton('← Previous')
        self.prev_btn.clicked.connect(self.previous_tab)
        nav_layout.addWidget(self.prev_btn)
        
        nav_layout.addStretch()
        
        self.next_btn = QPushButton('Next →')
        self.next_btn.clicked.connect(self.next_tab)
        nav_layout.addWidget(self.next_btn)
        
        self.import_btn = QPushButton('✨ Import Data')
        self.import_btn.clicked.connect(self.execute_import)
        self.import_btn.setVisible(False)
        nav_layout.addWidget(self.import_btn)
        
        close_btn = QPushButton('Close')
        close_btn.clicked.connect(self.reject)
        nav_layout.addWidget(close_btn)
        
        layout.addLayout(nav_layout)
        
        self.setLayout(layout)
        self.update_navigation()
    
    # ==========================================
    # TAB 1: SOURCE SELECTION
    # ==========================================
    
    def create_source_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        
        # Source type selection
        source_group = QGroupBox('Data Source Type')
        source_layout = QFormLayout()
        
        self.source_type = QComboBox()
        self.source_type.addItems([
            'CSV File',
            'Excel File (.xlsx, .xls)',
            'Shapefile',
            'KML/KMZ',
            'GeoJSON',
            'QGIS Layer (Current Project)',
            'PostGIS Database (External)',
            'Oracle Spatial',
            'AutoCAD DXF'
        ])
        self.source_type.currentTextChanged.connect(self.on_source_type_changed)
        source_layout.addRow('Source Type:', self.source_type)
        
        source_group.setLayout(source_layout)
        layout.addWidget(source_group)
        
        # File selection
        file_group = QGroupBox('File/Source Selection')
        file_layout = QVBoxLayout()
        
        file_select_layout = QHBoxLayout()
        self.file_path = QLineEdit()
        self.file_path.setPlaceholderText('Select file or source...')
        file_select_layout.addWidget(self.file_path)
        
        self.browse_btn = QPushButton('📂 Browse')
        self.browse_btn.clicked.connect(self.browse_file)
        file_select_layout.addWidget(self.browse_btn)
        
        file_layout.addLayout(file_select_layout)
        
        # Layer selection (for QGIS layers)
        self.layer_selector = QComboBox()
        self.layer_selector.setVisible(False)
        file_layout.addWidget(QLabel('Select Layer:'))
        file_layout.addWidget(self.layer_selector)
        
        # Load button
        self.load_btn = QPushButton('🔄 Load Source Data')
        self.load_btn.clicked.connect(self.load_source_data)
        file_layout.addWidget(self.load_btn)
        
        file_group.setLayout(file_layout)
        layout.addWidget(file_group)
        
        # Target table selection
        target_group = QGroupBox('Target FTTX Table')
        target_layout = QFormLayout()
        
        self.target_table = QComboBox()
        self.target_table.addItems([
            'itu_fiber_cables - Fiber Cables',
            'itu_odn_nodes - ODN Nodes (Poles, Manholes, Splitters)',
            'itu_onu_equipment - ONUs/ONTs',
            'itu_olt_equipment - OLTs',
            'itu_cable_routes - Cable Routes',
            'buildings - Buildings',
            'rooms - Telecom Rooms',
            'racks - Equipment Racks',
            'rack_equipment - Inside Plant Equipment'
        ])
        self.target_table.currentTextChanged.connect(self.on_target_changed)
        target_layout.addRow('Import To:', self.target_table)
        
        target_group.setLayout(target_layout)
        layout.addWidget(target_group)
        
        # Source info
        self.source_info = QTextEdit()
        self.source_info.setReadOnly(True)
        self.source_info.setMaximumHeight(150)
        layout.addWidget(QLabel('Source Information:'))
        layout.addWidget(self.source_info)
        
        layout.addStretch()
        widget.setLayout(layout)
        return widget
    
    # ==========================================
    # TAB 2: FIELD MAPPING
    # ==========================================
    
    def create_mapping_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        
        layout.addWidget(QLabel('<b>Map Source Fields to FTTX Database Fields:</b>'))
        
        info = QLabel('Match your source data fields with the corresponding FTTX database fields.')
        info.setWordWrap(True)
        layout.addWidget(info)
        
        # Mapping table
        self.mapping_table = QTableWidget()
        self.mapping_table.setColumnCount(4)
        self.mapping_table.setHorizontalHeaderLabels([
            'FTTX Field', 'Required?', 'Source Field', 'Sample Value'
        ])
        self.mapping_table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        layout.addWidget(self.mapping_table)
        
        # Auto-map button
        auto_map_layout = QHBoxLayout()
        auto_map_btn = QPushButton('🤖 Auto-Map Fields')
        auto_map_btn.setToolTip('Automatically match fields with similar names')
        auto_map_btn.clicked.connect(self.auto_map_fields)
        auto_map_layout.addWidget(auto_map_btn)
        
        clear_map_btn = QPushButton('🗑️ Clear Mappings')
        clear_map_btn.clicked.connect(self.clear_mappings)
        auto_map_layout.addWidget(clear_map_btn)
        
        auto_map_layout.addStretch()
        layout.addLayout(auto_map_layout)
        
        # Transform options
        transform_group = QGroupBox('Data Transformation Options')
        transform_layout = QFormLayout()
        
        self.skip_existing = QCheckBox('Skip records that already exist (based on code/ID)')
        self.skip_existing.setChecked(True)
        transform_layout.addRow('', self.skip_existing)
        
        self.update_existing = QCheckBox('Update existing records if found')
        transform_layout.addRow('', self.update_existing)
        
        self.validate_geometry = QCheckBox('Validate geometry before import')
        self.validate_geometry.setChecked(True)
        transform_layout.addRow('', self.validate_geometry)
        
        self.transform_crs = QCheckBox('Transform coordinates to EPSG:4326 (WGS84)')
        self.transform_crs.setChecked(True)
        transform_layout.addRow('', self.transform_crs)
        
        transform_group.setLayout(transform_layout)
        layout.addWidget(transform_group)
        
        widget.setLayout(layout)
        return widget
    
    # ==========================================
    # TAB 3: VALIDATION
    # ==========================================
    
    def create_validation_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        
        layout.addWidget(QLabel('<b>Validation Results:</b>'))
        
        # Validation summary
        summary_group = QGroupBox('Summary')
        summary_layout = QFormLayout()
        
        self.total_records_label = QLabel('0')
        summary_layout.addRow('Total Records:', self.total_records_label)
        
        self.valid_records_label = QLabel('0')
        summary_layout.addRow('Valid Records:', self.valid_records_label)
        
        self.invalid_records_label = QLabel('0')
        summary_layout.addRow('Invalid Records:', self.invalid_records_label)
        
        self.duplicate_records_label = QLabel('0')
        summary_layout.addRow('Duplicates:', self.duplicate_records_label)
        
        summary_group.setLayout(summary_layout)
        layout.addWidget(summary_group)
        
        # Validate button
        validate_btn = QPushButton('🔍 Validate Data')
        validate_btn.clicked.connect(self.validate_data)
        layout.addWidget(validate_btn)
        
        # Validation details
        layout.addWidget(QLabel('Validation Details:'))
        self.validation_results = QTextEdit()
        self.validation_results.setReadOnly(True)
        layout.addWidget(self.validation_results)
        
        # Preview sample records
        layout.addWidget(QLabel('Preview (First 10 records):'))
        self.preview_table = QTableWidget()
        self.preview_table.setMaximumHeight(200)
        layout.addWidget(self.preview_table)
        
        widget.setLayout(layout)
        return widget
    
    # ==========================================
    # TAB 4: IMPORT
    # ==========================================
    
    def create_import_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        
        layout.addWidget(QLabel('<b>Import Progress:</b>'))
        
        # Import options
        options_group = QGroupBox('Import Options')
        options_layout = QFormLayout()
        
        self.batch_size = QSpinBox()
        self.batch_size.setRange(10, 10000)
        self.batch_size.setValue(100)
        self.batch_size.setSuffix(' records')
        options_layout.addRow('Batch Size:', self.batch_size)
        
        self.create_backup = QCheckBox('Create backup before import')
        self.create_backup.setChecked(True)
        options_layout.addRow('', self.create_backup)
        
        self.rollback_on_error = QCheckBox('Rollback all if any error occurs')
        self.rollback_on_error.setChecked(True)
        options_layout.addRow('', self.rollback_on_error)
        
        options_group.setLayout(options_layout)
        layout.addWidget(options_group)
        
        # Progress
        self.progress_bar = QProgressBar()
        self.progress_bar.setRange(0, 100)
        layout.addWidget(self.progress_bar)
        
        self.progress_label = QLabel('Ready to import...')
        layout.addWidget(self.progress_label)
        
        # Import log
        layout.addWidget(QLabel('Import Log:'))
        self.import_log = QTextEdit()
        self.import_log.setReadOnly(True)
        layout.addWidget(self.import_log)
        
        # Export log button
        export_log_btn = QPushButton('💾 Export Log')
        export_log_btn.clicked.connect(self.export_log)
        layout.addWidget(export_log_btn)
        
        widget.setLayout(layout)
        return widget
    
    # ==========================================
    # HELPER METHODS
    # ==========================================
    
    def on_source_type_changed(self, source_type):
        """Handle source type changes"""
        is_qgis_layer = 'QGIS Layer' in source_type
        
        self.file_path.setVisible(not is_qgis_layer)
        self.browse_btn.setVisible(not is_qgis_layer)
        self.layer_selector.setVisible(is_qgis_layer)
        
        if is_qgis_layer:
            self.load_qgis_layers()
    
    def load_qgis_layers(self):
        """Load available QGIS layers"""
        self.layer_selector.clear()
        layers = QgsProject.instance().mapLayers().values()
        
        for layer in layers:
            if layer.type() == QgsVectorLayer.VectorLayer:
                self.layer_selector.addItem(layer.name(), layer.id())
    
    def browse_file(self):
        """Open file browser"""
        source_type = self.source_type.currentText()
        
        filters = {
            'CSV File': 'CSV Files (*.csv);;All Files (*.*)',
            'Excel File': 'Excel Files (*.xlsx *.xls);;All Files (*.*)',
            'Shapefile': 'Shapefiles (*.shp);;All Files (*.*)',
            'KML/KMZ': 'KML Files (*.kml *.kmz);;All Files (*.*)',
            'GeoJSON': 'GeoJSON Files (*.geojson *.json);;All Files (*.*)',
            'AutoCAD DXF': 'DXF Files (*.dxf);;All Files (*.*)'
        }
        
        file_filter = filters.get(source_type, 'All Files (*.*)')
        
        file_path, _ = QFileDialog.getOpenFileName(
            self, 'Select File', '', file_filter
        )
        
        if file_path:
            self.file_path.setText(file_path)
    
    def load_source_data(self):
        """Load data from selected source"""
        source_type = self.source_type.currentText()
        
        try:
            if 'QGIS Layer' in source_type:
                self.load_from_qgis_layer()
            elif 'CSV' in source_type:
                self.load_from_csv()
            elif 'Excel' in source_type:
                self.load_from_excel()
            elif 'Shapefile' in source_type:
                self.load_from_shapefile()
            elif 'GeoJSON' in source_type:
                self.load_from_geojson()
            else:
                QMessageBox.information(self, 'Not Implemented', 
                    f'{source_type} import is not yet implemented.\n\n' +
                    'Currently supported: CSV, Excel, Shapefile, GeoJSON, QGIS Layer')
                return
            
            self.update_source_info()
            QMessageBox.information(self, 'Success', 
                f'Loaded {len(self.imported_data.get("features", []))} records from source')
            
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Failed to load source:\n{str(e)}')
    
    def load_from_csv(self):
        """Load data from CSV file"""
        file_path = self.file_path.text()
        if not file_path or not os.path.exists(file_path):
            raise Exception('Invalid file path')
        
        features = []
        with open(file_path, 'r', encoding='utf-8-sig') as f:
            reader = csv.DictReader(f)
            self.imported_data['fields'] = reader.fieldnames
            
            for row in reader:
                features.append(row)
        
        self.imported_data['features'] = features
        self.imported_data['has_geometry'] = any(
            field.lower() in ['lat', 'latitude', 'y', 'lon', 'longitude', 'x', 'wkt', 'geometry']
            for field in self.imported_data['fields']
        )
    
    def load_from_excel(self):
        """Load data from Excel file"""
        try:
            import openpyxl
        except ImportError:
            raise Exception('openpyxl library not installed. Cannot read Excel files.')
        
        file_path = self.file_path.text()
        workbook = openpyxl.load_workbook(file_path)
        sheet = workbook.active
        
        # Get headers from first row
        headers = [cell.value for cell in sheet[1]]
        self.imported_data['fields'] = headers
        
        # Get data rows
        features = []
        for row in sheet.iter_rows(min_row=2, values_only=True):
            feature = dict(zip(headers, row))
            features.append(feature)
        
        self.imported_data['features'] = features
        self.imported_data['has_geometry'] = any(
            str(field).lower() in ['lat', 'latitude', 'y', 'lon', 'longitude', 'x', 'wkt', 'geometry']
            for field in headers if field
        )
    
    def load_from_shapefile(self):
        """Load data from Shapefile"""
        file_path = self.file_path.text()
        layer = QgsVectorLayer(file_path, 'temp', 'ogr')
        
        if not layer.isValid():
            raise Exception('Invalid shapefile')
        
        self.load_from_vector_layer(layer)
    
    def load_from_geojson(self):
        """Load data from GeoJSON"""
        file_path = self.file_path.text()
        layer = QgsVectorLayer(file_path, 'temp', 'ogr')
        
        if not layer.isValid():
            raise Exception('Invalid GeoJSON file')
        
        self.load_from_vector_layer(layer)
    
    def load_from_qgis_layer(self):
        """Load data from existing QGIS layer"""
        layer_id = self.layer_selector.currentData()
        layer = QgsProject.instance().mapLayer(layer_id)
        
        if not layer:
            raise Exception('Layer not found')
        
        self.load_from_vector_layer(layer)
    
    def load_from_vector_layer(self, layer):
        """Common method to load from any vector layer"""
        fields = [field.name() for field in layer.fields()]
        self.imported_data['fields'] = fields
        self.imported_data['has_geometry'] = layer.geometryType() != QgsWkbTypes.NullGeometry
        self.imported_data['geometry_type'] = layer.geometryTypeName()
        self.imported_data['crs'] = layer.crs().authid()
        
        features = []
        for feature in layer.getFeatures():
            feature_dict = {field: feature[field] for field in fields}
            
            if feature.hasGeometry():
                geom = feature.geometry()
                feature_dict['_geometry'] = geom
                feature_dict['_geom_wkt'] = geom.asWkt()
            
            features.append(feature_dict)
        
        self.imported_data['features'] = features
    
    def update_source_info(self):
        """Update source information display"""
        info = f"""
<b>Source Loaded Successfully</b><br><br>
<b>Records:</b> {len(self.imported_data.get('features', []))}<br>
<b>Fields:</b> {len(self.imported_data.get('fields', []))}<br>
<b>Has Geometry:</b> {'Yes' if self.imported_data.get('has_geometry') else 'No'}<br>
"""
        
        if self.imported_data.get('geometry_type'):
            info += f"<b>Geometry Type:</b> {self.imported_data['geometry_type']}<br>"
        
        if self.imported_data.get('crs'):
            info += f"<b>CRS:</b> {self.imported_data['crs']}<br>"
        
        info += f"<br><b>Fields:</b> {', '.join(self.imported_data.get('fields', []))}"
        
        self.source_info.setHtml(info)
        
        # Populate field mapping table
        self.populate_mapping_table()
    
    def on_target_changed(self):
        """Handle target table changes"""
        if self.imported_data.get('fields'):
            self.populate_mapping_table()
    
    def populate_mapping_table(self):
        """Populate field mapping table based on target"""
        target = self.target_table.currentText().split(' - ')[0]
        
        # Define required fields for each target table
        field_definitions = {
            'itu_fiber_cables': [
                ('cable_code', True), ('cable_type', True), ('fiber_type', False),
                ('fiber_count', False), ('cable_length_m', False), ('geom', True)
            ],
            'itu_odn_nodes': [
                ('node_code', True), ('node_type', True), ('split_ratio', False),
                ('output_ports', False), ('geom', True)
            ],
            'itu_onu_equipment': [
                ('onu_serial_number', True), ('pon_standard', False), 
                ('installation_address', False), ('geom', True)
            ],
            'itu_olt_equipment': [
                ('olt_code', True), ('pon_standard', True), ('manufacturer', False),
                ('model', False), ('geom', True)
            ],
            'buildings': [
                ('building_code', True), ('building_name', False), ('street_address', False),
                ('city', False), ('geom', True)
            ],
            'racks': [
                ('rack_code', True), ('rack_name', False), ('height_ru', True),
                ('room_id', True)
            ]
        }
        
        fields = field_definitions.get(target, [])
        source_fields = self.imported_data.get('fields', [])
        
        self.mapping_table.setRowCount(len(fields))
        
        for i, (field_name, required) in enumerate(fields):
            # FTTX Field
            self.mapping_table.setItem(i, 0, QTableWidgetItem(field_name))
            
            # Required?
            req_item = QTableWidgetItem('Yes' if required else 'No')
            if required:
                req_item.setBackground(Qt.yellow)
            self.mapping_table.setItem(i, 1, req_item)
            
            # Source Field (dropdown)
            combo = QComboBox()
            combo.addItem('-- Not Mapped --', None)
            for source_field in source_fields:
                combo.addItem(source_field, source_field)
            
            self.mapping_table.setCellWidget(i, 2, combo)
            
            # Sample Value (empty for now)
            self.mapping_table.setItem(i, 3, QTableWidgetItem(''))
    
    def auto_map_fields(self):
        """Automatically map fields based on name similarity"""
        if not self.imported_data.get('fields'):
            QMessageBox.warning(self, 'No Data', 'Load source data first')
            return
        
        source_fields = self.imported_data['fields']
        mapped_count = 0
        
        for row in range(self.mapping_table.rowCount()):
            fttx_field = self.mapping_table.item(row, 0).text().lower()
            combo = self.mapping_table.cellWidget(row, 2)
            
            # Try exact match first
            for source_field in source_fields:
                if source_field.lower() == fttx_field:
                    combo.setCurrentText(source_field)
                    mapped_count += 1
                    break
            
            # Try partial match
            if combo.currentIndex() == 0:  # Still not mapped
                for source_field in source_fields:
                    if fttx_field in source_field.lower() or source_field.lower() in fttx_field:
                        combo.setCurrentText(source_field)
                        mapped_count += 1
                        break
        
        QMessageBox.information(self, 'Auto-Map Complete', 
            f'Automatically mapped {mapped_count} fields')
        
        # Update sample values
        self.update_sample_values()
    
    def update_sample_values(self):
        """Update sample values in mapping table"""
        if not self.imported_data.get('features'):
            return
        
        first_record = self.imported_data['features'][0]
        
        for row in range(self.mapping_table.rowCount()):
            combo = self.mapping_table.cellWidget(row, 2)
            source_field = combo.currentData()
            
            if source_field and source_field in first_record:
                sample_value = str(first_record[source_field])[:50]
                self.mapping_table.setItem(row, 3, QTableWidgetItem(sample_value))
    
    def clear_mappings(self):
        """Clear all field mappings"""
        for row in range(self.mapping_table.rowCount()):
            combo = self.mapping_table.cellWidget(row, 2)
            combo.setCurrentIndex(0)
            self.mapping_table.setItem(row, 3, QTableWidgetItem(''))
    
    def validate_data(self):
        """Validate imported data before import"""
        if not self.imported_data.get('features'):
            QMessageBox.warning(self, 'No Data', 'Load source data first')
            return
        
        # Collect field mappings
        self.field_mappings = {}
        for row in range(self.mapping_table.rowCount()):
            fttx_field = self.mapping_table.item(row, 0).text()
            combo = self.mapping_table.cellWidget(row, 2)
            source_field = combo.currentData()
            
            if source_field:
                self.field_mappings[fttx_field] = source_field
        
        # Validate
        total = len(self.imported_data['features'])
        valid = 0
        invalid = 0
        duplicates = 0
        errors = []
        
        seen_codes = set()
        
        for i, feature in enumerate(self.imported_data['features']):
            is_valid = True
            
            # Check required fields
            for row in range(self.mapping_table.rowCount()):
                required = self.mapping_table.item(row, 1).text() == 'Yes'
                fttx_field = self.mapping_table.item(row, 0).text()
                
                if required and fttx_field not in self.field_mappings:
                    errors.append(f'Row {i+1}: Missing required field {fttx_field}')
                    is_valid = False
                    break
                
                if required and fttx_field in self.field_mappings:
                    source_field = self.field_mappings[fttx_field]
                    if not feature.get(source_field):
                        errors.append(f'Row {i+1}: Empty required field {fttx_field}')
                        is_valid = False
                        break
            
            # Check for duplicates (based on code field)
            code_field = self.field_mappings.get('cable_code') or self.field_mappings.get('node_code') or self.field_mappings.get('onu_serial_number')
            if code_field and code_field in feature:
                code = feature[code_field]
                if code in seen_codes:
                    duplicates += 1
                    errors.append(f'Row {i+1}: Duplicate code {code}')
                    is_valid = False
                else:
                    seen_codes.add(code)
            
            if is_valid:
                valid += 1
            else:
                invalid += 1
        
        # Update labels
        self.total_records_label.setText(str(total))
        self.valid_records_label.setText(str(valid))
        self.invalid_records_label.setText(str(invalid))
        self.duplicate_records_label.setText(str(duplicates))
        
        # Show errors
        if errors:
            error_text = '\n'.join(errors[:50])  # Show first 50 errors
            if len(errors) > 50:
                error_text += f'\n\n... and {len(errors) - 50} more errors'
            self.validation_results.setPlainText(error_text)
        else:
            self.validation_results.setHtml('<span style="color: green;"><b>✓ All records valid!</b></span>')
        
        # Update preview table
        self.update_preview()
    
    def update_preview(self):
        """Update preview table with sample records"""
        if not self.imported_data.get('features'):
            return
        
        preview_count = min(10, len(self.imported_data['features']))
        preview_features = self.imported_data['features'][:preview_count]
        
        if not preview_features:
            return
        
        # Get mapped fields
        mapped_fields = list(self.field_mappings.values())
        
        self.preview_table.setColumnCount(len(mapped_fields))
        self.preview_table.setHorizontalHeaderLabels(mapped_fields)
        self.preview_table.setRowCount(preview_count)
        
        for row, feature in enumerate(preview_features):
            for col, field in enumerate(mapped_fields):
                value = str(feature.get(field, ''))[:100]
                self.preview_table.setItem(row, col, QTableWidgetItem(value))
        
        self.preview_table.horizontalHeader().setSectionResizeMode(QHeaderView.ResizeToContents)
    
    def execute_import(self):
        """Execute the data import"""
        if not self.imported_data.get('features'):
            QMessageBox.warning(self, 'No Data', 'Load and validate data first')
            return
        
        # Confirm import
        reply = QMessageBox.question(
            self, 'Confirm Import',
            f'Import {len(self.imported_data["features"])} records into {self.target_table.currentText()}?',
            QMessageBox.Yes | QMessageBox.No
        )
        
        if reply != QMessageBox.Yes:
            return
        
        # Create backup if requested
        if self.create_backup.isChecked():
            self.log_message('Creating database backup...')
            # Backup logic would go here
        
        # Start import
        self.log_message('Starting import...')
        self.progress_bar.setValue(0)
        
        try:
            target_table = self.target_table.currentText().split(' - ')[0]
            
            if target_table == 'itu_fiber_cables':
                self.import_fiber_cables()
            elif target_table == 'itu_odn_nodes':
                self.import_odn_nodes()
            elif target_table == 'itu_onu_equipment':
                self.import_onus()
            elif target_table == 'itu_olt_equipment':
                self.import_olts()
            elif target_table == 'buildings':
                self.import_buildings()
            elif target_table == 'racks':
                self.import_racks()
            else:
                raise Exception(f'Import for {target_table} not implemented yet')
            
            self.progress_bar.setValue(100)
            self.log_message('✓ Import completed successfully!')
            
            QMessageBox.information(self, 'Success', 
                'Data imported successfully!\n\nRefresh your QGIS layers to see the new data.')
            
        except Exception as e:
            self.log_message(f'✗ Import failed: {str(e)}')
            QMessageBox.critical(self, 'Import Failed', f'Error during import:\n{str(e)}')
            
            if self.rollback_on_error.isChecked():
                self.log_message('Rolling back changes...')
                self.plugin.db_connection.rollback()
    
    def import_fiber_cables(self):
        """Import fiber cables"""
        cursor = self.plugin.db_connection.cursor()
        batch_size = self.batch_size.value()
        total = len(self.imported_data['features'])
        imported = 0
        skipped = 0
        
        for i, feature in enumerate(self.imported_data['features']):
            try:
                # Extract mapped fields
                cable_code = feature.get(self.field_mappings.get('cable_code'))
                cable_type = feature.get(self.field_mappings.get('cable_type'))
                fiber_type = feature.get(self.field_mappings.get('fiber_type', ''), 'G.652.D')
                fiber_count = int(feature.get(self.field_mappings.get('fiber_count', ''), 0) or 0)
                cable_length = float(feature.get(self.field_mappings.get('cable_length_m', ''), 0) or 0)
                
                # Skip if exists
                if self.skip_existing.isChecked():
                    cursor.execute("SELECT COUNT(*) FROM fttx.itu_fiber_cables WHERE cable_code = %s", (cable_code,))
                    if cursor.fetchone()[0] > 0:
                        skipped += 1
                        continue
                
                # Get geometry
                geom_wkt = self.extract_geometry(feature)
                
                # Insert
                cursor.execute("""
                    INSERT INTO fttx.itu_fiber_cables 
                    (cable_code, cable_type, fiber_type, fiber_count, cable_length_m, geom)
                    VALUES (%s, %s, %s, %s, %s, ST_GeomFromText(%s, 4326))
                """, (cable_code, cable_type, fiber_type, fiber_count, cable_length, geom_wkt))
                
                imported += 1
                
                # Commit in batches
                if imported % batch_size == 0:
                    self.plugin.db_connection.commit()
                    progress = int((i + 1) / total * 100)
                    self.progress_bar.setValue(progress)
                    self.progress_label.setText(f'Importing... {imported} of {total}')
                
            except Exception as e:
                self.log_message(f'Error on row {i+1}: {str(e)}')
                if self.rollback_on_error.isChecked():
                    raise
        
        # Final commit
        self.plugin.db_connection.commit()
        self.log_message(f'Imported {imported} cables, skipped {skipped}')
    
    def import_odn_nodes(self):
        """Import ODN nodes (poles, splitters, etc.)"""
        cursor = self.plugin.db_connection.cursor()
        batch_size = self.batch_size.value()
        total = len(self.imported_data['features'])
        imported = 0
        skipped = 0
        
        for i, feature in enumerate(self.imported_data['features']):
            try:
                node_code = feature.get(self.field_mappings.get('node_code'))
                node_type = feature.get(self.field_mappings.get('node_type'))
                split_ratio = feature.get(self.field_mappings.get('split_ratio'))
                output_ports = int(feature.get(self.field_mappings.get('output_ports', ''), 0) or 0)
                
                if self.skip_existing.isChecked():
                    cursor.execute("SELECT COUNT(*) FROM fttx.itu_odn_nodes WHERE node_code = %s", (node_code,))
                    if cursor.fetchone()[0] > 0:
                        skipped += 1
                        continue
                
                geom_wkt = self.extract_geometry(feature)
                
                cursor.execute("""
                    INSERT INTO fttx.itu_odn_nodes 
                    (node_code, node_type, split_ratio, output_ports, geom)
                    VALUES (%s, %s, %s, %s, ST_GeomFromText(%s, 4326))
                """, (node_code, node_type, split_ratio, output_ports, geom_wkt))
                
                imported += 1
                
                if imported % batch_size == 0:
                    self.plugin.db_connection.commit()
                    progress = int((i + 1) / total * 100)
                    self.progress_bar.setValue(progress)
                    self.progress_label.setText(f'Importing... {imported} of {total}')
                
            except Exception as e:
                self.log_message(f'Error on row {i+1}: {str(e)}')
                if self.rollback_on_error.isChecked():
                    raise
        
        self.plugin.db_connection.commit()
        self.log_message(f'Imported {imported} nodes, skipped {skipped}')
    
    def import_onus(self):
        """Import ONUs"""
        cursor = self.plugin.db_connection.cursor()
        batch_size = self.batch_size.value()
        total = len(self.imported_data['features'])
        imported = 0
        skipped = 0
        
        for i, feature in enumerate(self.imported_data['features']):
            try:
                serial_number = feature.get(self.field_mappings.get('onu_serial_number'))
                pon_standard = feature.get(self.field_mappings.get('pon_standard', ''), 'GPON')
                address = feature.get(self.field_mappings.get('installation_address', ''))
                
                if self.skip_existing.isChecked():
                    cursor.execute("SELECT COUNT(*) FROM fttx.itu_onu_equipment WHERE onu_serial_number = %s", 
                                 (serial_number,))
                    if cursor.fetchone()[0] > 0:
                        skipped += 1
                        continue
                
                geom_wkt = self.extract_geometry(feature)
                
                cursor.execute("""
                    INSERT INTO fttx.itu_onu_equipment 
                    (onu_serial_number, pon_standard, installation_address, geom)
                    VALUES (%s, %s, %s, ST_GeomFromText(%s, 4326))
                """, (serial_number, pon_standard, address, geom_wkt))
                
                imported += 1
                
                if imported % batch_size == 0:
                    self.plugin.db_connection.commit()
                    progress = int((i + 1) / total * 100)
                    self.progress_bar.setValue(progress)
                    self.progress_label.setText(f'Importing... {imported} of {total}')
                
            except Exception as e:
                self.log_message(f'Error on row {i+1}: {str(e)}')
                if self.rollback_on_error.isChecked():
                    raise
        
        self.plugin.db_connection.commit()
        self.log_message(f'Imported {imported} ONUs, skipped {skipped}')
    
    def import_olts(self):
        """Import OLTs"""
        cursor = self.plugin.db_connection.cursor()
        batch_size = self.batch_size.value()
        total = len(self.imported_data['features'])
        imported = 0
        skipped = 0
        
        for i, feature in enumerate(self.imported_data['features']):
            try:
                olt_code = feature.get(self.field_mappings.get('olt_code'))
                pon_standard = feature.get(self.field_mappings.get('pon_standard'), 'GPON')
                manufacturer = feature.get(self.field_mappings.get('manufacturer', ''))
                model = feature.get(self.field_mappings.get('model', ''))
                
                if self.skip_existing.isChecked():
                    cursor.execute("SELECT COUNT(*) FROM fttx.itu_olt_equipment WHERE olt_code = %s", (olt_code,))
                    if cursor.fetchone()[0] > 0:
                        skipped += 1
                        continue
                
                geom_wkt = self.extract_geometry(feature)
                
                cursor.execute("""
                    INSERT INTO fttx.itu_olt_equipment 
                    (olt_code, pon_standard, manufacturer, model, geom)
                    VALUES (%s, %s, %s, %s, ST_GeomFromText(%s, 4326))
                """, (olt_code, pon_standard, manufacturer, model, geom_wkt))
                
                imported += 1
                
                if imported % batch_size == 0:
                    self.plugin.db_connection.commit()
                    progress = int((i + 1) / total * 100)
                    self.progress_bar.setValue(progress)
                    self.progress_label.setText(f'Importing... {imported} of {total}')
                
            except Exception as e:
                self.log_message(f'Error on row {i+1}: {str(e)}')
                if self.rollback_on_error.isChecked():
                    raise
        
        self.plugin.db_connection.commit()
        self.log_message(f'Imported {imported} OLTs, skipped {skipped}')
    
    def import_buildings(self):
        """Import buildings"""
        cursor = self.plugin.db_connection.cursor()
        batch_size = self.batch_size.value()
        total = len(self.imported_data['features'])
        imported = 0
        skipped = 0
        
        for i, feature in enumerate(self.imported_data['features']):
            try:
                building_code = feature.get(self.field_mappings.get('building_code'))
                building_name = feature.get(self.field_mappings.get('building_name', ''))
                address = feature.get(self.field_mappings.get('street_address', ''))
                city = feature.get(self.field_mappings.get('city', ''))
                
                if self.skip_existing.isChecked():
                    cursor.execute("SELECT COUNT(*) FROM fttx.buildings WHERE building_code = %s", 
                                 (building_code,))
                    if cursor.fetchone()[0] > 0:
                        skipped += 1
                        continue
                
                geom_wkt = self.extract_geometry(feature)
                
                cursor.execute("""
                    INSERT INTO fttx.buildings 
                    (building_code, building_name, street_address, city, geom)
                    VALUES (%s, %s, %s, %s, ST_GeomFromText(%s, 4326))
                """, (building_code, building_name, address, city, geom_wkt))
                
                imported += 1
                
                if imported % batch_size == 0:
                    self.plugin.db_connection.commit()
                    progress = int((i + 1) / total * 100)
                    self.progress_bar.setValue(progress)
                    self.progress_label.setText(f'Importing... {imported} of {total}')
                
            except Exception as e:
                self.log_message(f'Error on row {i+1}: {str(e)}')
                if self.rollback_on_error.isChecked():
                    raise
        
        self.plugin.db_connection.commit()
        self.log_message(f'Imported {imported} buildings, skipped {skipped}')
    
    def import_racks(self):
        """Import equipment racks"""
        cursor = self.plugin.db_connection.cursor()
        batch_size = self.batch_size.value()
        total = len(self.imported_data['features'])
        imported = 0
        skipped = 0
        
        for i, feature in enumerate(self.imported_data['features']):
            try:
                rack_code = feature.get(self.field_mappings.get('rack_code'))
                rack_name = feature.get(self.field_mappings.get('rack_name', ''))
                height_ru = int(feature.get(self.field_mappings.get('height_ru', ''), 42) or 42)
                room_id = int(feature.get(self.field_mappings.get('room_id', ''), 0) or 0)
                
                if self.skip_existing.isChecked():
                    cursor.execute("SELECT COUNT(*) FROM fttx.racks WHERE rack_code = %s", (rack_code,))
                    if cursor.fetchone()[0] > 0:
                        skipped += 1
                        continue
                
                cursor.execute("""
                    INSERT INTO fttx.racks 
                    (rack_code, rack_name, height_ru, room_id, available_ru)
                    VALUES (%s, %s, %s, %s, %s)
                """, (rack_code, rack_name, height_ru, room_id, height_ru))
                
                imported += 1
                
                if imported % batch_size == 0:
                    self.plugin.db_connection.commit()
                    progress = int((i + 1) / total * 100)
                    self.progress_bar.setValue(progress)
                    self.progress_label.setText(f'Importing... {imported} of {total}')
                
            except Exception as e:
                self.log_message(f'Error on row {i+1}: {str(e)}')
                if self.rollback_on_error.isChecked():
                    raise
        
        self.plugin.db_connection.commit()
        self.log_message(f'Imported {imported} racks, skipped {skipped}')
    
    def extract_geometry(self, feature):
        """Extract geometry from feature in WKT format"""
        # Check if feature has _geometry from QGIS layer
        if '_geom_wkt' in feature:
            return feature['_geom_wkt']
        
        # Check for WKT field
        wkt_field = self.field_mappings.get('geom')
        if wkt_field and wkt_field in feature:
            return feature[wkt_field]
        
        # Try to build point from lat/lon
        lat_field = None
        lon_field = None
        
        for field, source in self.field_mappings.items():
            if 'lat' in source.lower() or source.lower() == 'y':
                lat_field = source
            if 'lon' in source.lower() or source.lower() == 'x':
                lon_field = source
        
        if lat_field and lon_field:
            try:
                lat = float(feature.get(lat_field, 0))
                lon = float(feature.get(lon_field, 0))
                return f'POINT({lon} {lat})'
            except:
                pass
        
        # Default fallback
        return 'POINT(0 0)'
    
    def log_message(self, message):
        """Add message to import log"""
        timestamp = datetime.now().strftime('%H:%M:%S')
        self.import_log.append(f'[{timestamp}] {message}')
    
    def export_log(self):
        """Export import log to file"""
        file_path, _ = QFileDialog.getSaveFileName(
            self, 'Export Log', '', 'Text Files (*.txt);;All Files (*.*)'
        )
        
        if file_path:
            with open(file_path, 'w') as f:
                f.write(self.import_log.toPlainText())
            
            QMessageBox.information(self, 'Exported', f'Log exported to:\n{file_path}')
    
    # ==========================================
    # NAVIGATION
    # ==========================================
    
    def update_navigation(self):
        """Update navigation button states"""
        current_tab = self.tabs.currentIndex()
        self.prev_btn.setEnabled(current_tab > 0)
        self.next_btn.setVisible(current_tab < 3)
        self.import_btn.setVisible(current_tab == 3)
    
    def previous_tab(self):
        """Go to previous tab"""
        current = self.tabs.currentIndex()
        if current > 0:
            self.tabs.setCurrentIndex(current - 1)
            self.update_navigation()
    
    def next_tab(self):
        """Go to next tab"""
        current = self.tabs.currentIndex()
        if current < 3:
            self.tabs.setCurrentIndex(current + 1)
            self.update_navigation()
            
            # Auto-actions on tab change
            if current == 0:  # Moving to mapping tab
                if self.imported_data.get('features'):
                    self.auto_map_fields()
            elif current == 1:  # Moving to validation tab
                self.validate_data()