# itu_fttx_plugin/gui/calculators_ui.py

from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import (
    QDialog, QMessageBox, QVBoxLayout, QHBoxLayout, 
    QLabel, QPushButton, QLineEdit, QSpinBox, QComboBox, 
    QGroupBox, QFormLayout, QDialogButtonBox, QTextEdit, 
    QDoubleSpinBox, QCheckBox, QScrollArea, QWidget,
    QProgressBar, QFileDialog
)
from ..core.calculators import PowerBudgetEngine, CableTensionEngine, AerialSlackEngine
from ..config.constants import DEFAULT_SPLITTER_LOSS
import math

# ==========================================
# POWER BUDGET CALCULATOR
# ==========================================

class PowerBudgetCalculator(QDialog):
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Power Budget Calculator - ITU-T G.984.2')
        self.setMinimumSize(750, 650)
        self.engine = PowerBudgetEngine()
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        
        # Header
        header = QLabel('⚡ Optical Power Budget Calculator')
        header.setStyleSheet('font-size: 14pt; font-weight: bold; background-color: #FF5722; color: white; padding: 10px;')
        layout.addWidget(header)
        
        # Transmitter Parameters
        tx_group = QGroupBox('Transmitter Parameters')
        tx_layout = QFormLayout()
        
        self.olt_tx_power = QDoubleSpinBox()
        self.olt_tx_power.setRange(-10, 10)
        self.olt_tx_power.setValue(3.0)
        self.olt_tx_power.setSuffix(' dBm')
        self.olt_tx_power.setDecimals(2)
        tx_layout.addRow('OLT TX Power:', self.olt_tx_power)
        
        self.onu_tx_power = QDoubleSpinBox()
        self.onu_tx_power.setRange(-10, 10)
        self.onu_tx_power.setValue(2.0)
        self.onu_tx_power.setSuffix(' dBm')
        self.onu_tx_power.setDecimals(2)
        tx_layout.addRow('ONU TX Power:', self.onu_tx_power)
        
        tx_group.setLayout(tx_layout)
        layout.addWidget(tx_group)
        
        # Receiver Parameters
        rx_group = QGroupBox('Receiver Parameters')
        rx_layout = QFormLayout()
        
        self.olt_rx_sensitivity = QDoubleSpinBox()
        self.olt_rx_sensitivity.setRange(-40, 0)
        self.olt_rx_sensitivity.setValue(-28.0)
        self.olt_rx_sensitivity.setSuffix(' dBm')
        self.olt_rx_sensitivity.setDecimals(2)
        rx_layout.addRow('OLT RX Sensitivity:', self.olt_rx_sensitivity)
        
        self.onu_rx_sensitivity = QDoubleSpinBox()
        self.onu_rx_sensitivity.setRange(-40, 0)
        self.onu_rx_sensitivity.setValue(-27.0)
        self.onu_rx_sensitivity.setSuffix(' dBm')
        self.onu_rx_sensitivity.setDecimals(2)
        rx_layout.addRow('ONU RX Sensitivity:', self.onu_rx_sensitivity)
        
        rx_group.setLayout(rx_layout)
        layout.addWidget(rx_group)
        
        # Loss Parameters
        loss_group = QGroupBox('Loss Parameters')
        loss_layout = QFormLayout()
        
        self.fiber_length = QDoubleSpinBox()
        self.fiber_length.setRange(0.1, 100)
        self.fiber_length.setValue(10.0)
        self.fiber_length.setSuffix(' km')
        self.fiber_length.setDecimals(2)
        loss_layout.addRow('Fiber Length:', self.fiber_length)
        
        self.fiber_attenuation = QDoubleSpinBox()
        self.fiber_attenuation.setRange(0.1, 1.0)
        self.fiber_attenuation.setValue(0.35)
        self.fiber_attenuation.setSuffix(' dB/km')
        self.fiber_attenuation.setDecimals(3)
        loss_layout.addRow('Fiber Attenuation:', self.fiber_attenuation)
        
        self.splitter_ratio = QComboBox()
        self.splitter_ratio.addItems(['1:2', '1:4', '1:8', '1:16', '1:32', '1:64'])
        self.splitter_ratio.setCurrentText('1:32')
        loss_layout.addRow('Splitter Ratio:', self.splitter_ratio)
        
        self.connector_count = QSpinBox()
        self.connector_count.setRange(0, 20)
        self.connector_count.setValue(4)
        loss_layout.addRow('Connector Count:', self.connector_count)
        
        self.connector_loss = QDoubleSpinBox()
        self.connector_loss.setRange(0.1, 2.0)
        self.connector_loss.setValue(0.5)
        self.connector_loss.setSuffix(' dB')
        self.connector_loss.setDecimals(2)
        loss_layout.addRow('Connector Loss:', self.connector_loss)
        
        self.splice_count = QSpinBox()
        self.splice_count.setRange(0, 50)
        self.splice_count.setValue(2)
        loss_layout.addRow('Splice Count:', self.splice_count)
        
        self.splice_loss = QDoubleSpinBox()
        self.splice_loss.setRange(0.01, 1.0)
        self.splice_loss.setValue(0.1)
        self.splice_loss.setSuffix(' dB')
        self.splice_loss.setDecimals(2)
        loss_layout.addRow('Splice Loss:', self.splice_loss)
        
        loss_group.setLayout(loss_layout)
        layout.addWidget(loss_group)
        
        # Calculate Button
        calc_btn = QPushButton('🧮 Calculate Power Budget')
        calc_btn.clicked.connect(self.calculate_budget)
        calc_btn.setStyleSheet('font-weight: bold; padding: 10px;')
        layout.addWidget(calc_btn)
        
        # Results
        self.results_text = QTextEdit()
        self.results_text.setReadOnly(True)
        self.results_text.setMaximumHeight(200)
        layout.addWidget(QLabel('Results:'))
        layout.addWidget(self.results_text)
        
        # Close button
        button_box = QDialogButtonBox(QDialogButtonBox.Close)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
        self.setLayout(layout)
        
    def calculate_budget(self):
        try:
            params = {
                'olt_tx_power': self.olt_tx_power.value(),
                'onu_rx_sensitivity': self.onu_rx_sensitivity.value(),
                'onu_tx_power': self.onu_tx_power.value(),
                'olt_rx_sensitivity': self.olt_rx_sensitivity.value(),
                'fiber_length': self.fiber_length.value(),
                'fiber_attenuation': self.fiber_attenuation.value(),
                'splitter_ratio': self.splitter_ratio.currentText(),
                'connector_count': self.connector_count.value(),
                'connector_loss': self.connector_loss.value(),
                'splice_count': self.splice_count.value(),
                'splice_loss': self.splice_loss.value()
            }
            
            results = self.engine.calculate(params)
            
            # Calculate budgets (not returned by engine)
            downstream_budget = params['olt_tx_power'] - params['onu_rx_sensitivity']
            upstream_budget = params['onu_tx_power'] - params['olt_rx_sensitivity']
            
            # Format results using actual keys from engine
            output = f"""
<h2>Power Budget Analysis</h2>

<h3>Downstream (OLT → ONU):</h3>
<ul>
<li><b>Total Loss:</b> {results['total_loss']:.2f} dB</li>
<li><b>Power Budget:</b> {downstream_budget:.2f} dB</li>
<li><b>Received Power:</b> {results['onu_rx_power']:.2f} dBm</li>
<li><b>Margin:</b> {results['downstream_margin']:.2f} dB</li>
<li><b>Status:</b> <span style="color: {'green' if results['downstream_margin'] >= 3 else 'red'}">
    {'✓ PASS' if results['downstream_margin'] >= 3 else '✗ FAIL'}</span></li>
</ul>

<h3>Upstream (ONU → OLT):</h3>
<ul>
<li><b>Total Loss:</b> {results['total_loss']:.2f} dB</li>
<li><b>Power Budget:</b> {upstream_budget:.2f} dB</li>
<li><b>Received Power:</b> {results['olt_rx_power']:.2f} dBm</li>
<li><b>Margin:</b> {results['upstream_margin']:.2f} dB</li>
<li><b>Status:</b> <span style="color: {'green' if results['upstream_margin'] >= 3 else 'red'}">
    {'✓ PASS' if results['upstream_margin'] >= 3 else '✗ FAIL'}</span></li>
</ul>

<h3>Loss Breakdown:</h3>
<ul>
<li><b>Fiber Loss:</b> {results['fiber_loss']:.2f} dB</li>
<li><b>Splitter Loss:</b> {results['splitter_loss']:.2f} dB</li>
<li><b>Connector Loss:</b> {results['connector_loss_total']:.2f} dB</li>
<li><b>Splice Loss:</b> {results['splice_loss_total']:.2f} dB</li>
<li><b>Total Loss:</b> {results['total_loss']:.2f} dB</li>
</ul>

<h3>Compliance:</h3>
<ul>
<li><b>Minimum Margin Required:</b> {results['min_margin']:.1f} dB</li>
<li><b>ITU-T G.984.2 Compliant:</b> <span style="color: {'green' if results['is_compliant'] else 'red'}">
    {'✓ YES' if results['is_compliant'] else '✗ NO'}</span></li>
</ul>
"""
            
            self.results_text.setHtml(output)
            
            # Show warning if margins are low
            if results['downstream_margin'] < 3 or results['upstream_margin'] < 3:
                QMessageBox.warning(self, 'Low Margin', 
                    'Power budget margin is below 3 dB recommendation!')
            
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Calculation error:\n{str(e)}')


# ==========================================
# CABLE TENSION CALCULATOR
# ==========================================

class CableTensionCalculator(QDialog):
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Cable Pull Tension Calculator')
        self.setMinimumSize(650, 600)
        self.engine = CableTensionEngine()
        self.init_ui()
        self.update_defaults()
    
    def init_ui(self):
        layout = QVBoxLayout()
        
        # Header
        header = QLabel('🔧 Cable Pull Tension Calculator')
        header.setStyleSheet('font-size: 14pt; font-weight: bold; background-color: #607D8B; color: white; padding: 10px;')
        layout.addWidget(header)
        
        # Cable Parameters
        cable_group = QGroupBox('Cable Parameters')
        cable_layout = QFormLayout()
        
        self.cable_weight = QDoubleSpinBox()
        self.cable_weight.setRange(10, 500)
        self.cable_weight.setValue(80)
        self.cable_weight.setSuffix(' kg/km')
        self.cable_weight.setDecimals(1)
        cable_layout.addRow('Cable Weight:', self.cable_weight)
        
        self.max_tension_rated = QSpinBox()
        self.max_tension_rated.setRange(100, 5000)
        self.max_tension_rated.setValue(600)
        self.max_tension_rated.setSuffix(' N')
        cable_layout.addRow('Max Rated Tension:', self.max_tension_rated)
        
        cable_group.setLayout(cable_layout)
        layout.addWidget(cable_group)
        
        # Pull Parameters
        pull_group = QGroupBox('Pull Parameters')
        pull_layout = QFormLayout()
        
        self.pull_length = QDoubleSpinBox()
        self.pull_length.setRange(10, 2000)
        self.pull_length.setValue(500)
        self.pull_length.setSuffix(' m')
        self.pull_length.setDecimals(1)
        pull_layout.addRow('Pull Length:', self.pull_length)
        
        self.coefficient_friction = QDoubleSpinBox()
        self.coefficient_friction.setRange(0.1, 1.0)
        self.coefficient_friction.setValue(0.5)
        self.coefficient_friction.setDecimals(2)
        pull_layout.addRow('Coefficient of Friction:', self.coefficient_friction)
        
        self.bend_count = QSpinBox()
        self.bend_count.setRange(0, 20)
        self.bend_count.setValue(3)
        pull_layout.addRow('Number of Bends:', self.bend_count)
        
        self.bend_angle = QDoubleSpinBox()
        self.bend_angle.setRange(0, 180)
        self.bend_angle.setValue(90)
        self.bend_angle.setSuffix(' °')
        self.bend_angle.setDecimals(1)
        pull_layout.addRow('Average Bend Angle:', self.bend_angle)
        
        self.elevation_change = QDoubleSpinBox()
        self.elevation_change.setRange(-100, 100)
        self.elevation_change.setValue(0)
        self.elevation_change.setSuffix(' m')
        self.elevation_change.setDecimals(1)
        pull_layout.addRow('Elevation Change:', self.elevation_change)
        
        pull_group.setLayout(pull_layout)
        layout.addWidget(pull_group)
        
        # Calculate Button
        calc_btn = QPushButton('🧮 Calculate Tension')
        calc_btn.clicked.connect(self.calculate_tension)
        calc_btn.setStyleSheet('font-weight: bold; padding: 10px;')
        layout.addWidget(calc_btn)
        
        # Results
        self.results_text = QTextEdit()
        self.results_text.setReadOnly(True)
        self.results_text.setMaximumHeight(180)
        layout.addWidget(QLabel('Results:'))
        layout.addWidget(self.results_text)
        
        # Close button
        button_box = QDialogButtonBox(QDialogButtonBox.Close)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
        self.setLayout(layout)
    
    def update_defaults(self):
        """Update default values based on selections"""
        pass  # Can be enhanced to load presets
        
    def calculate_tension(self):
        try:
            params = {
                'cable_weight': self.cable_weight.value(),
                'max_tension_rated': self.max_tension_rated.value(),
                'pull_length': self.pull_length.value(),
                'coefficient_friction': self.coefficient_friction.value(),
                'bend_count': self.bend_count.value(),
                'bend_angle': self.bend_angle.value(),
                'elevation_change': self.elevation_change.value()
            }
            
            results = self.engine.calculate(params)
            
            # Calculate safety factor and determine if safe
            safety_factor = results['max_allowed'] / results['total_tension'] if results['total_tension'] > 0 else 999
            is_safe = results['total_tension'] <= results['max_allowed']
            
            status_color = 'green' if is_safe else 'red'
            status_text = '✓ SAFE' if is_safe else '✗ UNSAFE'
            
            output = f"""
<h2>Pull Tension Analysis</h2>

<h3>Results:</h3>
<ul>
<li><b>Calculated Tension:</b> {results['total_tension']:.1f} N</li>
<li><b>Max Rated Tension:</b> {results['max_rated']:.1f} N</li>
<li><b>Max Allowed (with safety):</b> {results['max_allowed']:.1f} N</li>
<li><b>Safety Factor:</b> {safety_factor:.2f}</li>
<li><b>Utilization:</b> {results['utilization']:.1f}%</li>
<li><b>Status:</b> <span style="color: {status_color}; font-weight: bold;">{status_text}</span></li>
</ul>

<h3>Component Breakdown:</h3>
<ul>
<li><b>Straight Pull Tension:</b> {results['straight_tension']:.1f} N</li>
<li><b>Bend Penalty:</b> {results['bend_tension']:.1f} N</li>
<li><b>Elevation Component:</b> {results['elevation_tension']:.1f} N</li>
<li><b>Total Tension:</b> {results['total_tension']:.1f} N</li>
</ul>

<h3>Recommendations:</h3>
<ul>
<li>{'⚠ Reduce pull length or add intermediate pull points' if not is_safe else '✓ Pull parameters are acceptable'}</li>
<li>{'⚠ Use cable lubricant to reduce friction' if results['straight_tension'] > 200 else '✓ Friction is within acceptable range'}</li>
<li>{'⚠ Consider reducing number of bends' if results['bend_tension'] > 100 else '✓ Bend configuration is acceptable'}</li>
</ul>
"""
            
            self.results_text.setHtml(output)
            
            if not is_safe:
                QMessageBox.warning(self, 'Unsafe Pull', 
                    f'Calculated tension ({results["total_tension"]:.1f} N) exceeds safe limit ({results["max_allowed"]:.1f} N)!\n\n' +
                    'Reduce pull length, add intermediate pull points, or use lubricant.')
            
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Calculation error:\n{str(e)}')


# ==========================================
# AERIAL SLACK CALCULATOR
# ==========================================

class AerialSlackCalculator(QDialog):
    
    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Aerial Cable Slack/Sag Calculator')
        self.setMinimumSize(650, 550)
        self.engine = AerialSlackEngine()
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        
        # Header
        header = QLabel('📏 Aerial Cable Slack/Sag Calculator')
        header.setStyleSheet('font-size: 14pt; font-weight: bold; background-color: #00796B; color: white; padding: 10px;')
        layout.addWidget(header)
        
        # Span Parameters
        span_group = QGroupBox('Span Parameters')
        span_layout = QFormLayout()
        
        self.span_length = QDoubleSpinBox()
        self.span_length.setRange(10, 200)
        self.span_length.setValue(50)
        self.span_length.setSuffix(' m')
        self.span_length.setDecimals(1)
        span_layout.addRow('Span Length:', self.span_length)
        
        self.pole_height_diff = QDoubleSpinBox()
        self.pole_height_diff.setRange(-10, 10)
        self.pole_height_diff.setValue(0)
        self.pole_height_diff.setSuffix(' m')
        self.pole_height_diff.setDecimals(2)
        span_layout.addRow('Pole Height Difference:', self.pole_height_diff)
        
        span_group.setLayout(span_layout)
        layout.addWidget(span_group)
        
        # Cable Parameters
        cable_group = QGroupBox('Cable & Loading Parameters')
        cable_layout = QFormLayout()
        
        self.cable_weight_aerial = QDoubleSpinBox()
        self.cable_weight_aerial.setRange(10, 200)
        self.cable_weight_aerial.setValue(60)
        self.cable_weight_aerial.setSuffix(' kg/km')
        self.cable_weight_aerial.setDecimals(1)
        cable_layout.addRow('Cable Weight:', self.cable_weight_aerial)
        
        self.ice_loading = QDoubleSpinBox()
        self.ice_loading.setRange(0, 50)
        self.ice_loading.setValue(0)
        self.ice_loading.setSuffix(' kg/km')
        self.ice_loading.setDecimals(1)
        cable_layout.addRow('Ice Loading:', self.ice_loading)
        
        self.wind_loading = QDoubleSpinBox()
        self.wind_loading.setRange(0, 100)
        self.wind_loading.setValue(0)
        self.wind_loading.setSuffix(' Pa')
        self.wind_loading.setDecimals(1)
        cable_layout.addRow('Wind Loading:', self.wind_loading)
        
        cable_group.setLayout(cable_layout)
        layout.addWidget(cable_group)
        
        # Temperature Parameters
        temp_group = QGroupBox('Temperature Parameters')
        temp_layout = QFormLayout()
        
        self.install_temp = QDoubleSpinBox()
        self.install_temp.setRange(-40, 60)
        self.install_temp.setValue(20)
        self.install_temp.setSuffix(' °C')
        self.install_temp.setDecimals(1)
        temp_layout.addRow('Installation Temp:', self.install_temp)
        
        self.min_temp = QDoubleSpinBox()
        self.min_temp.setRange(-40, 60)
        self.min_temp.setValue(-10)
        self.min_temp.setSuffix(' °C')
        self.min_temp.setDecimals(1)
        temp_layout.addRow('Minimum Temp:', self.min_temp)
        
        self.max_temp = QDoubleSpinBox()
        self.max_temp.setRange(-40, 60)
        self.max_temp.setValue(40)
        self.max_temp.setSuffix(' °C')
        self.max_temp.setDecimals(1)
        temp_layout.addRow('Maximum Temp:', self.max_temp)
        
        temp_group.setLayout(temp_layout)
        layout.addWidget(temp_group)
        
        # Calculate Button
        calc_btn = QPushButton('🧮 Calculate Slack & Sag')
        calc_btn.clicked.connect(self.calculate_slack)
        calc_btn.setStyleSheet('font-weight: bold; padding: 10px;')
        layout.addWidget(calc_btn)
        
        # Results
        self.results_text = QTextEdit()
        self.results_text.setReadOnly(True)
        self.results_text.setMaximumHeight(200)
        layout.addWidget(QLabel('Results:'))
        layout.addWidget(self.results_text)
        
        # Close button
        button_box = QDialogButtonBox(QDialogButtonBox.Close)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
        self.setLayout(layout)
        
    def calculate_slack(self):
        try:
            params = {
                'span_length': self.span_length.value(),
                'pole_height_diff': self.pole_height_diff.value(),
                'cable_weight_aerial': self.cable_weight_aerial.value(),
                'ice_loading': self.ice_loading.value(),
                'wind_loading': self.wind_loading.value(),
                'install_temp': self.install_temp.value(),
                'min_temp': self.min_temp.value(),
                'max_temp': self.max_temp.value()
            }
            
            results = self.engine.calculate(params)
            
            # Assume minimum clearance requirement
            min_clearance = 5.5  # meters (typical for overhead cables)
            clearance_ok = results['sag'] < min_clearance
            
            # Format results
            output = f"""
<h2>Aerial Cable Analysis</h2>

<h3>Sag Results:</h3>
<ul>
<li><b>Midspan Sag:</b> {results['sag']:.3f} m</li>
<li><b>Sag Ratio:</b> {results['sag_ratio']:.4f}</li>
<li><b>Horizontal Tension:</b> {results['horizontal_tension']:.1f} N</li>
</ul>

<h3>Cable Length:</h3>
<ul>
<li><b>Straight Line Distance:</b> {self.span_length.value():.2f} m</li>
<li><b>Actual Cable Length:</b> {results['cable_length']:.3f} m</li>
<li><b>Geometric Slack:</b> {results['geometric_slack']:.3f} m ({results['slack_percent']:.2f}%)</li>
</ul>

<h3>Temperature Effects:</h3>
<ul>
<li><b>Temperature Range:</b> {results['temp_range']:.1f}°C</li>
<li><b>Thermal Expansion:</b> {results['temp_expansion']:.3f} m</li>
<li><b>Recommended Total Slack:</b> {results['recommended_slack']:.3f} m ({results['recommended_slack_percent']:.2f}%)</li>
<li><b>Installed Cable Length:</b> {results['installed_length']:.3f} m</li>
</ul>

<h3>Clearances:</h3>
<ul>
<li><b>Minimum Ground Clearance Required:</b> {min_clearance:.1f} m</li>
<li><b>Actual Clearance (estimated):</b> {min_clearance - results['sag']:.2f} m</li>
<li><b>Clearance OK:</b> {'✓ Yes' if clearance_ok else '✗ No - Increase tension or reduce span'}</li>
</ul>

<h3>Installation Guide:</h3>
<ul>
<li><b>Order Cable Length:</b> {results['installed_length']:.2f} m for {self.span_length.value():.1f} m span</li>
<li><b>Install Temperature:</b> {self.install_temp.value():.1f}°C</li>
<li><b>Notes:</b> Include {results['recommended_slack_percent']:.1f}% slack for thermal expansion and sag</li>
</ul>
"""
            
            self.results_text.setHtml(output)
            
            if not clearance_ok:
                QMessageBox.warning(self, 'Clearance Warning', 
                    f'Ground clearance may be insufficient!\n\n' +
                    f'Sag: {results["sag"]:.2f} m exceeds safe limit.\n' +
                    'Consider: reducing span length, increasing tension, or raising attachment points.')
            
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Calculation error:\n{str(e)}')