# itu_fttx_plugin/gui/bom_generator.py

# Standard Python Imports
import os
import math
from datetime import datetime

# PyQt/QGIS Imports
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import (
    QDialog, QMessageBox, QVBoxLayout, QHBoxLayout, 
    QLabel, QPushButton, QLineEdit, QSpinBox, QComboBox, 
    QGroupBox, QFormLayout, QDialogButtonBox, QTextEdit, 
    QDoubleSpinBox, QCheckBox, QTabWidget, QWidget, QFrame,
    QTableWidget, QTableWidgetItem, QHeaderView, QRadioButton,
    QButtonGroup, QFileDialog
)

# Relative Imports for Internal Modules
from ..core.planners import BOMEngine # Core Logic
from ..config.constants import APP_NAME # General Constants

class BOMGenerator(QDialog):
    """Bill of Materials (BOM) Generator with Cost Estimation"""
    
    def __init__(self, plugin):
        # Fix: Pass the QGIS main window as parent
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Bill of Materials (BOM) Generator')
        self.setMinimumSize(1100, 800)
        self.bom_data = []
        self.init_ui()
    
    def init_ui(self):
        """Builds the complete user interface for the BOM Generator."""
        layout = QVBoxLayout()
        
        # Header
        header = QLabel('📋 Bill of Materials Generator')
        header.setStyleSheet(
            'font-size: 16pt; font-weight: bold; '
            'background-color: #1976D2; color: white; padding: 15px;'
        )
        layout.addWidget(header)
        
        # Control Panel (restored logic)
        controls_group = QGroupBox('BOM Configuration')
        controls_layout = QHBoxLayout()
        
        scope_layout = QVBoxLayout()
        scope_layout.addWidget(QLabel('<b>Scope:</b>'))
        self.scope_group = QButtonGroup()
        self.scope_entire = QRadioButton('Entire Network')
        self.scope_entire.setChecked(True)
        self.scope_group.addButton(self.scope_entire)
        scope_layout.addWidget(self.scope_entire)
        self.scope_project = QRadioButton('Specific Project')
        self.scope_group.addButton(self.scope_project)
        scope_layout.addWidget(self.scope_project)
        self.scope_area = QRadioButton('Geographic Area')
        self.scope_group.addButton(self.scope_area)
        scope_layout.addWidget(self.scope_area)
        controls_layout.addLayout(scope_layout)
        
        filter_layout = QFormLayout()
        self.project_filter = QComboBox()
        self.project_filter.addItems(['All Projects', 'Project A', 'Project B', 'New Deployment'])
        self.project_filter.setEnabled(False)
        filter_layout.addRow('Project:', self.project_filter)
        self.status_filter = QComboBox()
        self.status_filter.addItems(['All Status', 'Planned', 'Under Construction', 'Operational'])
        filter_layout.addRow('Status:', self.status_filter)
        self.include_labor = QCheckBox('Include Labor Costs')
        self.include_labor.setChecked(True)
        filter_layout.addRow('', self.include_labor)
        self.include_contingency = QCheckBox('Add Contingency (10%)')
        self.include_contingency.setChecked(True)
        filter_layout.addRow('', self.include_contingency)
        controls_layout.addLayout(filter_layout)
        
        pricing_layout = QFormLayout()
        self.currency = QComboBox()
        self.currency.addItems(['USD $', 'EUR €', 'GBP £', 'ZAR R'])
        pricing_layout.addRow('Currency:', self.currency)
        self.markup_percent = QDoubleSpinBox()
        self.markup_percent.setRange(0, 100)
        self.markup_percent.setValue(15)
        self.markup_percent.setSuffix(' %')
        pricing_layout.addRow('Markup:', self.markup_percent)
        self.discount_percent = QDoubleSpinBox()
        self.discount_percent.setRange(0, 50)
        self.discount_percent.setValue(0)
        self.discount_percent.setSuffix(' %')
        pricing_layout.addRow('Discount:', self.discount_percent)
        controls_layout.addLayout(pricing_layout)
        
        controls_group.setLayout(controls_layout)
        layout.addWidget(controls_group)
        
        self.scope_project.toggled.connect(lambda: self.project_filter.setEnabled(self.scope_project.isChecked()))
        
        # Generate button
        generate_btn = QPushButton('📊 Generate BOM')
        generate_btn.clicked.connect(self.generate_bom)
        generate_btn.setStyleSheet(
            'background-color: #4CAF50; color: white; '
            'font-weight: bold; padding: 10px; font-size: 12pt;'
        )
        layout.addWidget(generate_btn)
        
        # Tabs for different views (instantiating necessary attributes)
        self.tabs = QTabWidget()
        self.summary_text = QTextEdit()
        self.bom_table = QTableWidget()
        self.category_table = QTableWidget()
        self.analysis_text = QTextEdit()

        self.tabs.addTab(self.create_summary_tab(), '📊 Summary')
        self.tabs.addTab(self.create_detailed_tab(), '📋 Detailed BOM')
        self.tabs.addTab(self.create_category_tab(), '📂 By Category')
        self.tabs.addTab(self.create_analysis_tab(), '💰 Cost Analysis')
        
        layout.addWidget(self.tabs)
        
        # Export buttons
        export_layout = QHBoxLayout()
        export_excel_btn = QPushButton('📊 Export to Excel')
        export_excel_btn.clicked.connect(self.export_excel)
        export_layout.addWidget(export_excel_btn)
        export_csv_btn = QPushButton('📄 Export to CSV')
        export_csv_btn.clicked.connect(self.export_csv)
        export_layout.addWidget(export_csv_btn)
        export_pdf_btn = QPushButton('📑 Export to PDF')
        export_pdf_btn.clicked.connect(self.export_pdf)
        export_layout.addWidget(export_pdf_btn)
        print_btn = QPushButton('🖨️ Print')
        print_btn.clicked.connect(self.print_bom)
        export_layout.addWidget(print_btn)
        export_layout.addStretch()
        layout.addLayout(export_layout)
        
        # Close button
        button_box = QDialogButtonBox(QDialogButtonBox.Close)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
        self.setLayout(layout)

    def create_summary_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        self.summary_text.setReadOnly(True)
        self.summary_text.setStyleSheet('background-color: #f5f5f5; font-family: Courier;')
        layout.addWidget(self.summary_text)
        widget.setLayout(layout)
        return widget
    
    def create_detailed_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        search_layout = QHBoxLayout()
        search_layout.addWidget(QLabel('Search:'))
        self.search_box = QLineEdit()
        self.search_box.setPlaceholderText('Filter items...')
        self.search_box.textChanged.connect(self.filter_bom_table)
        search_layout.addWidget(self.search_box)
        search_layout.addStretch()
        layout.addLayout(search_layout)
        self.bom_table.setColumnCount(10)
        self.bom_table.setHorizontalHeaderLabels(['Item #', 'Category', 'Description', 'Part Number','Manufacturer', 'Quantity', 'Unit', 'Unit Price','Total Price', 'Lead Time'])
        self.bom_table.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.bom_table.setAlternatingRowColors(True)
        self.bom_table.setSortingEnabled(True)
        layout.addWidget(self.bom_table)
        widget.setLayout(layout)
        return widget
    
    def create_category_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        self.category_table.setColumnCount(5)
        self.category_table.setHorizontalHeaderLabels(['Category', 'Item Count', 'Total Quantity', 'Total Cost', 'Percentage'])
        self.category_table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.category_table.setAlternatingRowColors(True)
        layout.addWidget(self.category_table)
        widget.setLayout(layout)
        return widget
    
    def create_analysis_tab(self):
        widget = QWidget()
        layout = QVBoxLayout()
        self.analysis_text.setReadOnly(True)
        self.analysis_text.setStyleSheet('background-color: #f5f5f5; font-family: Courier;')
        layout.addWidget(self.analysis_text)
        widget.setLayout(layout)
        return widget

    def generate_bom(self):
        """Generates BOM by calling the core engine and populating UI."""
        try:
            # Instantiate the decoupled engine
            bom_engine = BOMEngine(self.plugin.db_connection)
            
            # Gather config params from UI
            config_params = {
                'include_labor': self.include_labor.isChecked(),
                'status_filter': self.status_filter.currentText(),
                'currency': self.currency.currentText(),
                'markup': 1 + (self.markup_percent.value() / 100),
                'discount': 1 - (self.discount_percent.value() / 100),
                'include_contingency': self.include_contingency.isChecked()
            }
            
            # Delegate to the core engine
            self.bom_data = bom_engine.generate_bom(config_params)
            
            # Populate UI elements
            self.populate_bom_tables()
            
            QMessageBox.information(self, '✓ Success', 
                f'BOM generated successfully!\n\nTotal items: {len(self.bom_data)}')
        
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Failed to generate BOM:\n\n{str(e)}')

    def populate_bom_tables(self):
        """Populate all BOM tables with calculated data"""
        if not self.bom_data: return
        
        markup = 1 + (self.markup_percent.value() / 100)
        discount = 1 - (self.discount_percent.value() / 100)
        
        # Recalculate totals to ensure latest markup/discount is applied
        for item in self.bom_data:
            base_total = item['quantity'] * item['unit_price']
            item['total_price'] = base_total * markup * discount
        
        # 1. Populate detailed table
        self.bom_table.setRowCount(len(self.bom_data))
        self.bom_table.setSortingEnabled(False)
        currency_symbol = self.currency.currentText().split()[1]
        
        for row, item in enumerate(self.bom_data):
            self.bom_table.setItem(row, 0, QTableWidgetItem(str(row + 1)))
            self.bom_table.setItem(row, 1, QTableWidgetItem(item['category']))
            self.bom_table.setItem(row, 2, QTableWidgetItem(item['description']))
            self.bom_table.setItem(row, 3, QTableWidgetItem(item['part_number']))
            self.bom_table.setItem(row, 4, QTableWidgetItem(item['manufacturer']))
            self.bom_table.setItem(row, 5, QTableWidgetItem(f"{item['quantity']:.2f}"))
            self.bom_table.setItem(row, 6, QTableWidgetItem(item['unit']))
            self.bom_table.setItem(row, 7, QTableWidgetItem(f"{currency_symbol}{item['unit_price']:,.2f}"))
            self.bom_table.setItem(row, 8, QTableWidgetItem(f"{currency_symbol}{item['total_price']:,.2f}"))
            self.bom_table.setItem(row, 9, QTableWidgetItem(item['lead_time']))
        
        self.bom_table.setSortingEnabled(True)

        # 2. Populate category table
        categories = {}
        for item in self.bom_data:
            cat = item['category']
            if cat not in categories: categories[cat] = {'count': 0, 'qty': 0, 'cost': 0}
            categories[cat]['count'] += 1
            categories[cat]['qty'] += item['quantity']
            categories[cat]['cost'] += item['total_price']
        
        total_cost = sum([cat['cost'] for cat in categories.values()])
        self.category_table.setRowCount(len(categories))
        self.category_table.setSortingEnabled(False)
        
        for row, (cat_name, cat_data) in enumerate(sorted(categories.items())):
            percentage = (cat_data['cost'] / total_cost * 100) if total_cost > 0 else 0
            self.category_table.setItem(row, 0, QTableWidgetItem(cat_name))
            self.category_table.setItem(row, 1, QTableWidgetItem(str(cat_data['count'])))
            self.category_table.setItem(row, 2, QTableWidgetItem(f"{cat_data['qty']:,.2f}"))
            self.category_table.setItem(row, 3, QTableWidgetItem(f"{currency_symbol}{cat_data['cost']:,.2f}"))
            self.category_table.setItem(row, 4, QTableWidgetItem(f"{percentage:.1f}%"))
        
        self.category_table.setSortingEnabled(True)

        # 3. Generate summary and analysis text
        subtotal = total_cost
        contingency = subtotal * 0.10 if self.include_contingency.isChecked() else 0
        grand_total = subtotal + contingency
        
        summary = f"""
{'=' * 70}\nBILL OF MATERIALS SUMMARY\n{'=' * 70}\n
Subtotal: {currency_symbol}{subtotal:,.2f}
Contingency (10%): {currency_symbol}{contingency:,.2f}
GRAND TOTAL: {currency_symbol}{grand_total:,.2f}
"""
        self.summary_text.setText(summary)
        self.analysis_text.setText(f"TOTAL CAPEX: {currency_symbol}{grand_total:,.2f}\n" + 
                                   "Cost breakdown charts here...")
    
    def filter_bom_table(self):
        search_text = self.search_box.text().lower()
        for row in range(self.bom_table.rowCount()):
            show_row = any(self.bom_table.item(row, col) and search_text in self.bom_table.item(row, col).text().lower()
                           for col in range(self.bom_table.columnCount()))
            self.bom_table.setRowHidden(row, not show_row)
    
    def export_excel(self):
        """Export BOM to Excel file (uses openpyxl if available, falls back to CSV)"""
        if not self.bom_data:
            QMessageBox.warning(self, 'No Data', 'Please generate BOM first.')
            return

        filepath, _ = QFileDialog.getSaveFileName(
            self, 'Export BOM', '', 'Excel Files (*.xlsx);;CSV Files (*.csv);;All Files (*)')
        if not filepath:
            return

        try:
            try:
                import openpyxl
                wb = openpyxl.Workbook()
                ws = wb.active
                ws.title = 'BOM'
                headers = ['Item #', 'Category', 'Description', 'Part Number',
                           'Manufacturer', 'Quantity', 'Unit', 'Unit Price', 'Total Price', 'Lead Time']
                ws.append(headers)
                for i, item in enumerate(self.bom_data):
                    ws.append([
                        i + 1, item['category'], item['description'], item['part_number'],
                        item['manufacturer'], item['quantity'], item['unit'],
                        item['unit_price'], item['total_price'], item['lead_time']
                    ])
                wb.save(filepath)
                QMessageBox.information(self, 'Exported', f'BOM exported to:\n{filepath}')
            except ImportError:
                # Fall back to CSV
                if not filepath.endswith('.csv'):
                    filepath += '.csv'
                self._write_csv(filepath)
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Export failed:\n{str(e)}')

    def export_csv(self):
        """Export BOM to CSV file"""
        if not self.bom_data:
            QMessageBox.warning(self, 'No Data', 'Please generate BOM first.')
            return

        filepath, _ = QFileDialog.getSaveFileName(
            self, 'Export BOM to CSV', '', 'CSV Files (*.csv);;All Files (*)')
        if not filepath:
            return

        try:
            self._write_csv(filepath)
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Export failed:\n{str(e)}')

    def _write_csv(self, filepath):
        """Write BOM data to CSV file"""
        import csv
        currency_symbol = self.currency.currentText().split()[1]
        with open(filepath, 'w', newline='', encoding='utf-8') as f:
            writer = csv.writer(f)
            writer.writerow(['Item #', 'Category', 'Description', 'Part Number',
                            'Manufacturer', 'Quantity', 'Unit', 'Unit Price', 'Total Price', 'Lead Time'])
            for i, item in enumerate(self.bom_data):
                writer.writerow([
                    i + 1, item['category'], item['description'], item['part_number'],
                    item['manufacturer'], f"{item['quantity']:.2f}", item['unit'],
                    f"{currency_symbol}{item['unit_price']:,.2f}",
                    f"{currency_symbol}{item['total_price']:,.2f}",
                    item['lead_time']
                ])
        QMessageBox.information(self, 'Exported', f'BOM exported to:\n{filepath}')

    def export_pdf(self):
        """Export BOM to PDF file"""
        if not self.bom_data:
            QMessageBox.warning(self, 'No Data', 'Please generate BOM first.')
            return

        filepath, _ = QFileDialog.getSaveFileName(
            self, 'Export BOM to PDF', '', 'PDF Files (*.pdf);;All Files (*)')
        if not filepath:
            return

        try:
            from qgis.PyQt.QtPrintSupport import QPrinter
            from qgis.PyQt.QtGui import QTextDocument

            printer = QPrinter(QPrinter.HighResolution)
            printer.setOutputFormat(QPrinter.PdfFormat)
            printer.setOutputFileName(filepath)

            currency_symbol = self.currency.currentText().split()[1]
            html = '<h1>Bill of Materials</h1>'
            html += f'<p>Generated: {datetime.now().strftime("%Y-%m-%d %H:%M")}</p>'
            html += '<table border="1" cellpadding="4" cellspacing="0" width="100%">'
            html += '<tr><th>#</th><th>Category</th><th>Description</th><th>Qty</th>'
            html += '<th>Unit Price</th><th>Total</th></tr>'
            for i, item in enumerate(self.bom_data):
                html += f'<tr><td>{i+1}</td><td>{item["category"]}</td>'
                html += f'<td>{item["description"]}</td><td>{item["quantity"]:.2f}</td>'
                html += f'<td>{currency_symbol}{item["unit_price"]:,.2f}</td>'
                html += f'<td>{currency_symbol}{item["total_price"]:,.2f}</td></tr>'
            html += '</table>'

            total = sum(item['total_price'] for item in self.bom_data)
            html += f'<h3>Total: {currency_symbol}{total:,.2f}</h3>'

            doc = QTextDocument()
            doc.setHtml(html)
            doc.print_(printer)

            QMessageBox.information(self, 'Exported', f'BOM exported to PDF:\n{filepath}')
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'PDF export failed:\n{str(e)}')

    def print_bom(self):
        """Print BOM using system print dialog"""
        if not self.bom_data:
            QMessageBox.warning(self, 'No Data', 'Please generate BOM first.')
            return

        try:
            from qgis.PyQt.QtPrintSupport import QPrinter, QPrintDialog
            from qgis.PyQt.QtGui import QTextDocument

            printer = QPrinter(QPrinter.HighResolution)
            dialog = QPrintDialog(printer, self)

            if dialog.exec_() == QPrintDialog.Accepted:
                currency_symbol = self.currency.currentText().split()[1]
                html = '<h1>Bill of Materials</h1>'
                html += '<table border="1" cellpadding="4" cellspacing="0" width="100%">'
                html += '<tr><th>#</th><th>Category</th><th>Description</th><th>Qty</th>'
                html += '<th>Unit Price</th><th>Total</th></tr>'
                for i, item in enumerate(self.bom_data):
                    html += f'<tr><td>{i+1}</td><td>{item["category"]}</td>'
                    html += f'<td>{item["description"]}</td><td>{item["quantity"]:.2f}</td>'
                    html += f'<td>{currency_symbol}{item["unit_price"]:,.2f}</td>'
                    html += f'<td>{currency_symbol}{item["total_price"]:,.2f}</td></tr>'
                html += '</table>'

                total = sum(item['total_price'] for item in self.bom_data)
                html += f'<h3>Total: {currency_symbol}{total:,.2f}</h3>'

                doc = QTextDocument()
                doc.setHtml(html)
                doc.print_(printer)
        except Exception as e:
            QMessageBox.critical(self, 'Error', f'Print failed:\n{str(e)}')