# itu_fttx_plugin/fttx_layer_styles.py
from qgis.core import (
    QgsSymbol, QgsSimpleLineSymbolLayer, QgsSimpleMarkerSymbolLayer,
    QgsSimpleFillSymbolLayer, QgsCategorizedSymbolRenderer, QgsRendererCategory,
    QgsMarkerSymbol, QgsLineSymbol, QgsFillSymbol, QgsEditorWidgetSetup,
    QgsVectorLayerSimpleLabeling, QgsPalLayerSettings, QgsTextFormat, QgsTextBufferSettings,
    QgsProperty, QgsSymbolLayer, QgsProject
)
from qgis.PyQt.QtGui import QColor, QFont
from qgis.PyQt.QtCore import Qt


# ==========================================
# CONFIGURATION - MODIFY THESE CODES AS PER CLIENT REQUIREMENTS
# ==========================================

class FTTXCodes:
    """
    Centralized configuration for all FTTX codes.
    Modify these values to match client requirements without changing the rest of the code.
    """
    
    # FIBER CABLE TYPES
    CABLE_FEEDER = 'feeder'
    CABLE_DISTRIBUTION = 'distribution'
    CABLE_DROP = 'drop'
    CABLE_BACKBONE = 'backbone'
    
    # ODN NODE TYPES
    NODE_SPLICE_CLOSURE = 'splice_closure'
    NODE_SPLITTER = 'splitter'
    NODE_POLE = 'pole'
    NODE_HANDHOLE = 'handhole'
    NODE_MANHOLE = 'manhole'
    
    # OPERATIONAL STATUS (for cables, nodes, routes, equipment)
    STATUS_PLANNED = 'planned'
    STATUS_ACTIVE = 'active'
    STATUS_INACTIVE = 'inactive'
    STATUS_MAINTENANCE = 'maintenance'
    STATUS_OPERATIONAL = 'operational'
    
    # PON PORT STATUS
    PORT_ACTIVE = 'active'
    PORT_INACTIVE = 'inactive'
    PORT_MAINTENANCE = 'maintenance'
    PORT_FAILED = 'failed'
    
    # ONU STATUS
    ONU_PLANNED = 'planned'
    ONU_ONLINE = 'online'
    ONU_OFFLINE = 'offline'
    ONU_FAILED = 'failed'
    
    # BUILDING TYPES
    BUILDING_RESIDENTIAL = 'residential'
    BUILDING_COMMERCIAL = 'commercial'
    BUILDING_MDU = 'mdu'
    BUILDING_INDUSTRIAL = 'industrial'
    BUILDING_MIXED_USE = 'mixed_use'
    
    # BUILDING SERVICE STATUS
    BUILDING_CONNECTED = 'connected'
    BUILDING_SERVICEABLE = 'serviceable'
    BUILDING_NOT_SERVICEABLE = 'not_serviceable'
    BUILDING_PLANNED = 'planned'
    
    # INFRASTRUCTURE/ROUTE TYPES
    ROUTE_AERIAL = 'aerial'
    ROUTE_UNDERGROUND = 'underground'
    ROUTE_DUCT = 'duct'
    ROUTE_DIRECT_BURIED = 'direct_buried'
    
    # CONDUIT MATERIALS
    MATERIAL_PVC = 'pvc'
    MATERIAL_HDPE = 'hdpe'
    MATERIAL_STEEL = 'steel'
    MATERIAL_CONCRETE = 'concrete'
    MATERIAL_OTHER = 'other'
    
    # SPLITTER RATIOS
    SPLITTER_1_2 = '1:2'
    SPLITTER_1_4 = '1:4'
    SPLITTER_1_8 = '1:8'
    SPLITTER_1_16 = '1:16'
    SPLITTER_1_32 = '1:32'
    SPLITTER_1_64 = '1:64'


class FTTXLayerStyles:
    """QGIS Layer Styles and Forms Configuration for FTTX Layers"""
    
    @staticmethod
    def apply_all_styles(plugin):
        """Apply styles and forms to all FTTX layers"""
        layers = QgsProject.instance().mapLayers()
        
        for layer_id, layer in layers.items():
            layer_name = layer.name()
            
            # Fiber Cables
            if 'Fiber Cables' in layer_name or 'itu_fiber_cables' in layer.source():
                FTTXLayerStyles.style_fiber_cables(layer)
                FTTXLayerStyles.configure_fiber_cables_form(layer)
            
            # ODN Nodes (Splice Closures, Poles, Handholes, etc.)
            elif 'ODN Nodes' in layer_name or 'itu_odn_nodes' in layer.source():
                FTTXLayerStyles.style_odn_nodes(layer)
                FTTXLayerStyles.configure_odn_nodes_form(layer)
            
            # ONU Equipment (Customer Premise)
            elif 'ONU' in layer_name or 'itu_onu_equipment' in layer.source():
                FTTXLayerStyles.style_onus(layer)
                FTTXLayerStyles.configure_onu_form(layer)
            
            # OLT Equipment (Central Office)
            elif 'OLT' in layer_name or 'itu_olt_equipment' in layer.source():
                FTTXLayerStyles.style_olts(layer)
                FTTXLayerStyles.configure_olt_form(layer)
            
            # PON Ports (Used for connection analysis)
            elif 'PON Port' in layer_name or 'pon_ports' in layer.source():
                FTTXLayerStyles.style_pon_ports(layer)
                FTTXLayerStyles.configure_pon_port_form(layer)
            
            # Buildings (Customer Demand Points)
            elif 'Building' in layer_name or 'buildings' in layer.source():
                FTTXLayerStyles.style_buildings(layer)
                FTTXLayerStyles.configure_building_form(layer)
            
            # Cable Routes (Infrastructure lines/conduit)
            elif 'Cable Route' in layer_name or 'cable_routes' in layer.source() or 'itu_cable_routes' in layer.source():
                FTTXLayerStyles.style_cable_routes(layer)
                FTTXLayerStyles.configure_cable_routes_form(layer)
        
        plugin.iface.messageBar().pushSuccess('FTTX Styles', 'Layer styles and forms applied successfully!')
    
    # ==========================================
    # FIBER CABLES STYLING
    # ==========================================
    
    @staticmethod
    def style_fiber_cables(layer):
        """Style fiber cables by type with different colors and widths"""
        
        categories = []
        
        # Feeder cables - thick blue
        symbol = QgsLineSymbol.createSimple({
            'color': '0,0,255',
            'width': '2',
            'capstyle': 'round'
        })
        categories.append(QgsRendererCategory(FTTXCodes.CABLE_FEEDER, symbol, 'Feeder Cable'))
        
        # Distribution cables - medium green
        symbol = QgsLineSymbol.createSimple({
            'color': '0,200,0',
            'width': '1.5',
            'capstyle': 'round'
        })
        categories.append(QgsRendererCategory(FTTXCodes.CABLE_DISTRIBUTION, symbol, 'Distribution Cable'))
        
        # Drop cables - thin orange
        symbol = QgsLineSymbol.createSimple({
            'color': '255,128,0',
            'width': '0.8',
            'capstyle': 'round'
        })
        categories.append(QgsRendererCategory(FTTXCodes.CABLE_DROP, symbol, 'Drop Cable'))
        
        # Backbone cables - thick purple
        symbol = QgsLineSymbol.createSimple({
            'color': '128,0,128',
            'width': '2.5',
            'capstyle': 'round'
        })
        categories.append(QgsRendererCategory(FTTXCodes.CABLE_BACKBONE, symbol, 'Backbone Cable'))
        
        # Default/Other
        symbol = QgsLineSymbol.createSimple({
            'color': '128,128,128',
            'width': '1',
            'capstyle': 'round'
        })
        categories.append(QgsRendererCategory('', symbol, 'Other'))
        
        # Apply categorized renderer
        renderer = QgsCategorizedSymbolRenderer('cable_type', categories)
        layer.setRenderer(renderer)
        layer.triggerRepaint()
    
    @staticmethod
    def configure_fiber_cables_form(layer):
        """Configure form for fiber cables"""
        from qgis.core import QgsEditFormConfig
        
        fields = layer.fields()
        config = layer.editFormConfig()
        config.setLayout(QgsEditFormConfig.TabLayout)
        
        # Field aliases
        field_config = {
            'cable_id': {'alias': 'Cable ID'},
            'cable_code': {'alias': 'Cable Code'},
            'cable_category': {'alias': 'Cable Category'},
            'cable_type': {'alias': 'Cable Type'},
            'fiber_type': {'alias': 'Fiber Type'},
            'fiber_count': {'alias': 'Fiber Count'},
            'operational_status': {'alias': 'Operational Status'},
            'cable_length_m': {'alias': 'Length (m)'},
            'installation_date': {'alias': 'Installation Date'},
            'installation_contractor': {'alias': 'Contractor'},
            'owner_organization': {'alias': 'Owner'}
        }
        
        for idx, field in enumerate(fields):
            field_name = field.name()
            
            if field_name in field_config:
                layer.setFieldAlias(idx, field_config[field_name]['alias'])
            
            # Auto-generated ID field - read-only
            if field_name in ['cable_id']:
                config.setReadOnly(idx, False)
            
            # Code field - NOT read-only (allows trigger to populate)
            if field_name in ['cable_code']:
                 # Ensure this is NOT set to read-only
                 pass
            
            # Configure specific widgets
            if field_name == 'cable_type':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        'Feeder': FTTXCodes.CABLE_FEEDER,
                        'Distribution': FTTXCodes.CABLE_DISTRIBUTION,
                        'Drop': FTTXCodes.CABLE_DROP,
                        'Backbone': FTTXCodes.CABLE_BACKBONE
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'operational_status':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        'Planned': FTTXCodes.STATUS_PLANNED,
                        'Active': FTTXCodes.STATUS_ACTIVE,
                        'Operational': FTTXCodes.STATUS_OPERATIONAL,
                        'Inactive': FTTXCodes.STATUS_INACTIVE,
                        'Maintenance': FTTXCodes.STATUS_MAINTENANCE
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'fiber_count':
                setup = QgsEditorWidgetSetup('Range', {
                    'Min': 0,
                    'Max': 288,
                    'Step': 1,
                    'Style': 'SpinBox'
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'installation_date':
                setup = QgsEditorWidgetSetup('DateTime', {
                    'display_format': 'yyyy-MM-dd',
                    'field_format': 'yyyy-MM-dd',
                    'calendar_popup': True
                })
                layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    # ==========================================
    # ODN NODES STYLING
    # ==========================================
    
    @staticmethod
    def style_odn_nodes(layer):
        """Style ODN nodes by type"""
        
        categories = []
        
        # Splice Closure - red circle
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'circle',
            'color': '255,0,0',
            'size': '4',
            'outline_color': 'black',
            'outline_width': '0.4'
        })
        categories.append(QgsRendererCategory(FTTXCodes.NODE_SPLICE_CLOSURE, symbol, 'Splice Closure'))
        
        # Splitter - yellow square
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'square',
            'color': '255,255,0',
            'size': '3.5',
            'outline_color': 'black',
            'outline_width': '0.4'
        })
        categories.append(QgsRendererCategory(FTTXCodes.NODE_SPLITTER, symbol, 'Splitter'))
        
        # Pole - brown triangle
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'triangle',
            'color': '139,69,19',
            'size': '4',
            'outline_color': 'black',
            'outline_width': '0.4'
        })
        categories.append(QgsRendererCategory(FTTXCodes.NODE_POLE, symbol, 'Pole'))
        
        # Handhole - gray diamond
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'diamond',
            'color': '128,128,128',
            'size': '4',
            'outline_color': 'black',
            'outline_width': '0.4'
        })
        categories.append(QgsRendererCategory(FTTXCodes.NODE_HANDHOLE, symbol, 'Handhole'))
        
        # Manhole - dark gray square
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'square',
            'color': '64,64,64',
            'size': '5',
            'outline_color': 'black',
            'outline_width': '0.4'
        })
        categories.append(QgsRendererCategory(FTTXCodes.NODE_MANHOLE, symbol, 'Manhole'))
        
        # Default
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'circle',
            'color': '200,200,200',
            'size': '3',
            'outline_color': 'black',
            'outline_width': '0.4'
        })
        categories.append(QgsRendererCategory('', symbol, 'Other'))
        
        renderer = QgsCategorizedSymbolRenderer('node_type', categories)
        layer.setRenderer(renderer)
        layer.triggerRepaint()
    
    @staticmethod
    def configure_odn_nodes_form(layer):
        """Configure form for ODN nodes"""
        from qgis.core import QgsEditFormConfig
        
        fields = layer.fields()
        config = layer.editFormConfig()
        config.setLayout(QgsEditFormConfig.TabLayout)
        
        field_config = {
            'node_id': {'alias': 'Node ID'},
            'node_code': {'alias': 'Node Code'},
            'node_type': {'alias': 'Node Type'},
            'node_function': {'alias': 'Node Function'},
            'operational_status': {'alias': 'Operational Status'},
            'splitter_type': {'alias': 'Splitter Type'},
            'split_ratio': {'alias': 'Split Ratio'},
            'output_ports': {'alias': 'Output Ports'},
            'used_output_ports': {'alias': 'Used Ports'},
            'installation_date': {'alias': 'Installation Date'}
        }
        
        for idx, field in enumerate(fields):
            field_name = field.name()
            
            if field_name in field_config:
                layer.setFieldAlias(idx, field_config[field_name]['alias'])
            
            # Auto-generated ID field - read-only
            if field_name in ['node_id']:
                config.setReadOnly(idx, False)
            
            # Code field - NOT read-only (allows trigger to populate)
            if field_name in ['node_code']:
                pass
            
            if field_name == 'node_type':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        'Splice Closure': FTTXCodes.NODE_SPLICE_CLOSURE,
                        'Splitter': FTTXCodes.NODE_SPLITTER,
                        'Pole': FTTXCodes.NODE_POLE,
                        'Handhole': FTTXCodes.NODE_HANDHOLE,
                        'Manhole': FTTXCodes.NODE_MANHOLE
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'operational_status':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        'Planned': FTTXCodes.STATUS_PLANNED,
                        'Active': FTTXCodes.STATUS_ACTIVE,
                        'Operational': FTTXCodes.STATUS_OPERATIONAL,
                        'Inactive': FTTXCodes.STATUS_INACTIVE,
                        'Maintenance': FTTXCodes.STATUS_MAINTENANCE
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'split_ratio':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        '1:2': FTTXCodes.SPLITTER_1_2,
                        '1:4': FTTXCodes.SPLITTER_1_4,
                        '1:8': FTTXCodes.SPLITTER_1_8,
                        '1:16': FTTXCodes.SPLITTER_1_16,
                        '1:32': FTTXCodes.SPLITTER_1_32,
                        '1:64': FTTXCodes.SPLITTER_1_64
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name in ['output_ports', 'used_output_ports']:
                setup = QgsEditorWidgetSetup('Range', {
                    'Min': 0,
                    'Max': 128,
                    'Step': 1,
                    'Style': 'SpinBox'
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'installation_date':
                setup = QgsEditorWidgetSetup('DateTime', {
                    'display_format': 'yyyy-MM-dd',
                    'field_format': 'yyyy-MM-dd',
                    'calendar_popup': True
                })
                layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    # ==========================================
    # ONU EQUIPMENT STYLING
    # ==========================================
    
    @staticmethod
    def style_onus(layer):
        """Style ONUs by status"""
        
        categories = []
        
        # Online - green star
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'star',
            'color': '0,255,0',
            'size': '4',
            'outline_color': 'black',
            'outline_width': '0.3'
        })
        categories.append(QgsRendererCategory(FTTXCodes.ONU_ONLINE, symbol, 'Online'))
        
        # Offline - red star
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'star',
            'color': '255,0,0',
            'size': '4',
            'outline_color': 'black',
            'outline_width': '0.3'
        })
        categories.append(QgsRendererCategory(FTTXCodes.ONU_OFFLINE, symbol, 'Offline'))
        
        # Planned - yellow star
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'star',
            'color': '255,255,0',
            'size': '4',
            'outline_color': 'black',
            'outline_width': '0.3'
        })
        categories.append(QgsRendererCategory(FTTXCodes.ONU_PLANNED, symbol, 'Planned'))
        
        # Failed - orange star
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'star',
            'color': '255,128,0',
            'size': '4',
            'outline_color': 'black',
            'outline_width': '0.3'
        })
        categories.append(QgsRendererCategory(FTTXCodes.ONU_FAILED, symbol, 'Failed'))
        
        # Default
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'star',
            'color': '128,128,128',
            'size': '3.5',
            'outline_color': 'black',
            'outline_width': '0.3'
        })
        categories.append(QgsRendererCategory('', symbol, 'Unknown'))
        
        renderer = QgsCategorizedSymbolRenderer('onu_status', categories)
        layer.setRenderer(renderer)
        layer.triggerRepaint()
    
    @staticmethod
    def configure_onu_form(layer):
        """Configure form for ONU equipment"""
        from qgis.core import QgsEditFormConfig
        
        fields = layer.fields()
        config = layer.editFormConfig()
        config.setLayout(QgsEditFormConfig.TabLayout)
        
        field_config = {
            'onu_id': {'alias': 'ONU ID'},
            'onu_serial_number': {'alias': 'Serial Number'},
            'onu_status': {'alias': 'ONU Status'},
            'connected_olt_id': {'alias': 'Connected OLT'},
            'connected_pon_port_id': {'alias': 'PON Port'},
            'customer_id': {'alias': 'Customer ID'},
            'installation_date': {'alias': 'Installation Date'},
            'measured_rx_power_dbm': {'alias': 'RX Power (dBm)'},
            'measured_tx_power_dbm': {'alias': 'TX Power (dBm)'}
        }
        
        for idx, field in enumerate(fields):
            field_name = field.name()
            
            if field_name in field_config:
                layer.setFieldAlias(idx, field_config[field_name]['alias'])
            
            # Auto-generated ID field - read-only
            if field_name in ['onu_id']:
                config.setReadOnly(idx, False)

            # Code field - NOT read-only (allows trigger to populate)
            if field_name in ['onu_serial_number']:
                pass
            
            if field_name == 'onu_status':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        'Planned': FTTXCodes.ONU_PLANNED,
                        'Online': FTTXCodes.ONU_ONLINE,
                        'Offline': FTTXCodes.ONU_OFFLINE,
                        'Failed': FTTXCodes.ONU_FAILED
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'installation_date':
                setup = QgsEditorWidgetSetup('DateTime', {
                    'display_format': 'yyyy-MM-dd',
                    'field_format': 'yyyy-MM-dd',
                    'calendar_popup': True
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name in ['measured_rx_power_dbm', 'measured_tx_power_dbm']:
                setup = QgsEditorWidgetSetup('Range', {
                    'Min': -30,
                    'Max': 5,
                    'Step': 0.1,
                    'Style': 'SpinBox',
                    'Suffix': ' dBm'
                })
                layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    # ==========================================
    # OLT EQUIPMENT STYLING
    # ==========================================
    
    @staticmethod
    def style_olts(layer):
        """Style OLTs - Central office equipment"""
        
        # Single style for all OLTs - large blue hexagon
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'hexagon',
            'color': '0,0,200',
            'size': '6',
            'outline_color': 'white',
            'outline_width': '0.5'
        })
        
        from qgis.core import QgsSingleSymbolRenderer
        renderer = QgsSingleSymbolRenderer(symbol)
        layer.setRenderer(renderer)
        layer.triggerRepaint()
    
    @staticmethod
    def configure_olt_form(layer):
        """Configure form for OLT equipment"""
        from qgis.core import QgsEditFormConfig
        
        fields = layer.fields()
        config = layer.editFormConfig()
        config.setLayout(QgsEditFormConfig.TabLayout)
        
        field_config = {
            'olt_id': {'alias': 'OLT ID'},
            'olt_code': {'alias': 'OLT Code'},
            'equipment_status': {'alias': 'Equipment Status'},
            'pon_standard': {'alias': 'PON Standard'},
            'manufacturer': {'alias': 'Manufacturer'},
            'model': {'alias': 'Model'},
            'installation_date': {'alias': 'Installation Date'},
            'max_onus_supported': {'alias': 'Max ONUs'},
            'current_onus_connected': {'alias': 'Current ONUs'}
        }
        
        for idx, field in enumerate(fields):
            field_name = field.name()
            
            if field_name in field_config:
                layer.setFieldAlias(idx, field_config[field_name]['alias'])
            
            # Auto-generated ID field - read-only
            if field_name in ['olt_id']:
                config.setReadOnly(idx, False)

            # Code field - NOT read-only (allows trigger to populate)
            if field_name in ['olt_code']:
                pass
            
            if field_name == 'equipment_status':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        'Planned': FTTXCodes.STATUS_PLANNED,
                        'Active': FTTXCodes.STATUS_ACTIVE,
                        'Operational': FTTXCodes.STATUS_OPERATIONAL,
                        'Inactive': FTTXCodes.STATUS_INACTIVE,
                        'Maintenance': FTTXCodes.STATUS_MAINTENANCE
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name in ['max_onus_supported', 'current_onus_connected']:
                setup = QgsEditorWidgetSetup('Range', {
                    'Min': 0,
                    'Max': 1024,
                    'Step': 1,
                    'Style': 'SpinBox'
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'installation_date':
                setup = QgsEditorWidgetSetup('DateTime', {
                    'display_format': 'yyyy-MM-dd',
                    'field_format': 'yyyy-MM-dd',
                    'calendar_popup': True
                })
                layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    # ==========================================
    # PON PORTS STYLING
    # ==========================================
    
    @staticmethod
    def style_pon_ports(layer):
        """Style PON ports by status"""
        
        categories = []
        
        # Active - green
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'circle',
            'color': '0,200,0',
            'size': '2.5',
            'outline_color': 'black',
            'outline_width': '0.2'
        })
        categories.append(QgsRendererCategory(FTTXCodes.PORT_ACTIVE, symbol, 'Active'))
        
        # Inactive - red
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'circle',
            'color': '200,0,0',
            'size': '2.5',
            'outline_color': 'black',
            'outline_width': '0.2'
        })
        categories.append(QgsRendererCategory(FTTXCodes.PORT_INACTIVE, symbol, 'Inactive'))
        
        # Maintenance - orange
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'circle',
            'color': '255,165,0',
            'size': '2.5',
            'outline_color': 'black',
            'outline_width': '0.2'
        })
        categories.append(QgsRendererCategory(FTTXCodes.PORT_MAINTENANCE, symbol, 'Maintenance'))
        
        # Failed - gray
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'circle',
            'color': '100,100,100',
            'size': '2.5',
            'outline_color': 'black',
            'outline_width': '0.2'
        })
        categories.append(QgsRendererCategory(FTTXCodes.PORT_FAILED, symbol, 'Failed'))
        
        # Default
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'circle',
            'color': '128,128,128',
            'size': '2',
            'outline_color': 'black',
            'outline_width': '0.2'
        })
        categories.append(QgsRendererCategory('', symbol, 'Unknown'))
        
        renderer = QgsCategorizedSymbolRenderer('port_status', categories)
        layer.setRenderer(renderer)
        layer.triggerRepaint()
    
    @staticmethod
    def configure_pon_port_form(layer):
        """Configure form for PON ports"""
        from qgis.core import QgsEditFormConfig
        
        fields = layer.fields()
        config = layer.editFormConfig()
        config.setLayout(QgsEditFormConfig.TabLayout)
        
        field_config = {
            'pon_port_id': {'alias': 'Port ID'},
            'olt_id': {'alias': 'OLT ID'},
            'port_number': {'alias': 'Port Number'},
            'port_label': {'alias': 'Port Label'},
            'port_status': {'alias': 'Port Status'},
            'max_onus': {'alias': 'Max ONUs'},
            'connected_onus': {'alias': 'Connected ONUs'},
            'optical_budget_class': {'alias': 'Budget Class'}
        }
        
        for idx, field in enumerate(fields):
            field_name = field.name()
            
            if field_name in field_config:
                layer.setFieldAlias(idx, field_config[field_name]['alias'])
            
            # Auto-generated ID field - read-only
            if field_name in ['pon_port_id']:
                config.setReadOnly(idx, False)

            # Note: PON Ports do not have a dedicated 'port_code' in the schema, only an auto-increment ID.
            
            if field_name == 'port_status':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        'Active': FTTXCodes.PORT_ACTIVE,
                        'Inactive': FTTXCodes.PORT_INACTIVE,
                        'Maintenance': FTTXCodes.PORT_MAINTENANCE,
                        'Failed': FTTXCodes.PORT_FAILED
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name in ['max_onus', 'connected_onus']:
                setup = QgsEditorWidgetSetup('Range', {
                    'Min': 0,
                    'Max': 128,
                    'Step': 1,
                    'Style': 'SpinBox'
                })
                layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    # ==========================================
    # BUILDINGS STYLING (By Service Status)
    # ==========================================
    
    @staticmethod
    def style_buildings(layer):
        """Style buildings by service status"""
        
        categories = []
        
        # Connected - dark green (top priority)
        symbol = QgsFillSymbol.createSimple({
            'color': '0,128,0,180',
            'outline_color': 'black',
            'outline_width': '0.4'
        })
        categories.append(QgsRendererCategory(FTTXCodes.BUILDING_CONNECTED, symbol, 'Connected'))
        
        # Serviceable - light green
        symbol = QgsFillSymbol.createSimple({
            'color': '144,238,144,180',
            'outline_color': 'black',
            'outline_width': '0.3'
        })
        categories.append(QgsRendererCategory(FTTXCodes.BUILDING_SERVICEABLE, symbol, 'Serviceable'))
        
        # Planned - yellow
        symbol = QgsFillSymbol.createSimple({
            'color': '255,255,0,180',
            'outline_color': 'black',
            'outline_width': '0.3'
        })
        categories.append(QgsRendererCategory(FTTXCodes.BUILDING_PLANNED, symbol, 'Planned'))
        
        # Not Serviceable - light red
        symbol = QgsFillSymbol.createSimple({
            'color': '255,182,193,180',
            'outline_color': 'black',
            'outline_width': '0.3'
        })
        categories.append(QgsRendererCategory(FTTXCodes.BUILDING_NOT_SERVICEABLE, symbol, 'Not Serviceable'))
        
        # Default
        symbol = QgsFillSymbol.createSimple({
            'color': '200,200,200,180',
            'outline_color': 'black',
            'outline_width': '0.3'
        })
        categories.append(QgsRendererCategory('', symbol, 'Unknown'))
        
        renderer = QgsCategorizedSymbolRenderer('service_status', categories)
        layer.setRenderer(renderer)
        layer.triggerRepaint()
    
    @staticmethod
    def configure_building_form(layer):
        """Configure form for buildings"""
        from qgis.core import QgsEditFormConfig, QgsEditorWidgetSetup
        
        fields = layer.fields()
        config = layer.editFormConfig()
        config.setLayout(QgsEditFormConfig.TabLayout)
        
        field_config = {
            'building_id': {'alias': 'Building ID'},
            'building_code': {'alias': 'Building Code'},
            'building_name': {'alias': 'Building Name'},
            'street_address': {'alias': 'Street Address'},
            'city': {'alias': 'City'},
            'postal_code': {'alias': 'Postal Code'},
            'building_type': {'alias': 'Building Type'},
            'service_status': {'alias': 'Service Status'},
            'floors_count': {'alias': 'Number of Floors'},
            'construction_year': {'alias': 'Construction Year'},
            'owner': {'alias': 'Owner'},
            'contact_person': {'alias': 'Contact Person'},
            'contact_phone': {'alias': 'Phone'},
            'notes': {'alias': 'Notes'}
        }
        
        for idx, field in enumerate(fields):
            field_name = field.name()
            
            if field_name in field_config:
                layer.setFieldAlias(idx, field_config[field_name]['alias'])
            
            # Auto-generated ID/Date fields - read-only
            if field_name in ['building_id', 'created_date']:
                config.setReadOnly(idx, False)

            # Code field - NOT read-only (allows trigger to populate)
            if field_name in ['building_code']:
                pass
            
            # Building Type dropdown
            if field_name == 'building_type':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        'Residential': FTTXCodes.BUILDING_RESIDENTIAL,
                        'Commercial': FTTXCodes.BUILDING_COMMERCIAL,
                        'Multi-Dwelling Unit (MDU)': FTTXCodes.BUILDING_MDU,
                        'Industrial': FTTXCodes.BUILDING_INDUSTRIAL,
                        'Mixed Use': FTTXCodes.BUILDING_MIXED_USE
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            # Service Status dropdown
            elif field_name == 'service_status':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        'Connected': FTTXCodes.BUILDING_CONNECTED,
                        'Serviceable': FTTXCodes.BUILDING_SERVICEABLE,
                        'Not Serviceable': FTTXCodes.BUILDING_NOT_SERVICEABLE,
                        'Planned': FTTXCodes.BUILDING_PLANNED
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            # Floors count - reasonable range
            elif field_name == 'floors_count':
                setup = QgsEditorWidgetSetup('Range', {
                    'Min': 1,
                    'Max': 200,
                    'Step': 1,
                    'Style': 'SpinBox'
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            # Construction Year - proper year range
            elif field_name == 'construction_year':
                setup = QgsEditorWidgetSetup('Range', {
                    'Min': 1800,
                    'Max': 2100,
                    'Step': 1,
                    'Style': 'SpinBox',
                    'AllowNull': True
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            # Multi-line text fields
            elif field_name in ['street_address', 'notes']:
                setup = QgsEditorWidgetSetup('TextEdit', {
                    'IsMultiline': True,
                    'UseHtml': False
                })
                layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    # ==========================================
    # CABLE ROUTES STYLING
    # ==========================================
    
    @staticmethod
    def style_cable_routes(layer):
        """Style cable routes/conduits"""
        
        categories = []
        
        # Aerial - blue dashed
        symbol = QgsLineSymbol.createSimple({
            'color': '0,0,255',
            'width': '1.2',
            'line_style': 'dash'
        })
        categories.append(QgsRendererCategory(FTTXCodes.ROUTE_AERIAL, symbol, 'Aerial'))
        
        # Underground - brown solid
        symbol = QgsLineSymbol.createSimple({
            'color': '139,69,19',
            'width': '1.2',
            'line_style': 'solid'
        })
        categories.append(QgsRendererCategory(FTTXCodes.ROUTE_UNDERGROUND, symbol, 'Underground'))
        
        # Duct - gray solid thick
        symbol = QgsLineSymbol.createSimple({
            'color': '100,100,100',
            'width': '1.5',
            'line_style': 'solid'
        })
        categories.append(QgsRendererCategory(FTTXCodes.ROUTE_DUCT, symbol, 'Duct'))
        
        # Direct Buried - dark brown
        symbol = QgsLineSymbol.createSimple({
            'color': '101,67,33',
            'width': '1.2',
            'line_style': 'solid'
        })
        categories.append(QgsRendererCategory(FTTXCodes.ROUTE_DIRECT_BURIED, symbol, 'Direct Buried'))
        
        # Default
        symbol = QgsLineSymbol.createSimple({
            'color': '128,128,128',
            'width': '1',
            'line_style': 'solid'
        })
        categories.append(QgsRendererCategory('', symbol, 'Other'))
        
        renderer = QgsCategorizedSymbolRenderer('infrastructure_type', categories)
        layer.setRenderer(renderer)
        layer.triggerRepaint()
    
    @staticmethod
    def configure_cable_routes_form(layer):
        """Configure form for cable routes"""
        from qgis.core import QgsEditFormConfig
        
        fields = layer.fields()
        config = layer.editFormConfig()
        config.setLayout(QgsEditFormConfig.TabLayout)
        
        field_config = {
            'route_id': {'alias': 'Route ID'},
            'route_code': {'alias': 'Route Code'},
            'route_name': {'alias': 'Route Name'},
            'infrastructure_type': {'alias': 'Infrastructure Type'},
            'route_status': {'alias': 'Route Status'},
            'duct_count': {'alias': 'Number of Ducts'},
            'duct_occupancy_percent': {'alias': 'Occupancy %'},
            'route_length_m': {'alias': 'Length (m)'},
            'construction_date': {'alias': 'Construction Date'},
            'row_owner': {'alias': 'ROW Owner'}
        }
        
        for idx, field in enumerate(fields):
            field_name = field.name()
            
            if field_name in field_config:
                layer.setFieldAlias(idx, field_config[field_name]['alias'])
            
            # Auto-generated ID field - read-only
            if field_name in ['route_id']:
                config.setReadOnly(idx, False)

            # Code field - NOT read-only (allows trigger to populate)
            if field_name in ['route_code']:
                pass
            
            if field_name == 'infrastructure_type':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        'Aerial': FTTXCodes.ROUTE_AERIAL,
                        'Underground': FTTXCodes.ROUTE_UNDERGROUND,
                        'Duct': FTTXCodes.ROUTE_DUCT,
                        'Direct Buried': FTTXCodes.ROUTE_DIRECT_BURIED
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'route_status':
                setup = QgsEditorWidgetSetup('ValueMap', {
                    'map': {
                        'Planned': FTTXCodes.STATUS_PLANNED,
                        'Active': FTTXCodes.STATUS_ACTIVE,
                        'Inactive': FTTXCodes.STATUS_INACTIVE,
                        'Maintenance': FTTXCodes.STATUS_MAINTENANCE
                    }
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'duct_count':
                setup = QgsEditorWidgetSetup('Range', {
                    'Min': 0,
                    'Max': 24,
                    'Step': 1,
                    'Style': 'SpinBox'
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'route_length_m':
                setup = QgsEditorWidgetSetup('Range', {
                    'Min': 0,
                    'Max': 100000,
                    'Step': 1,
                    'Style': 'SpinBox',
                    'Suffix': ' m'
                })
                layer.setEditorWidgetSetup(idx, setup)
            
            elif field_name == 'construction_date':
                setup = QgsEditorWidgetSetup('DateTime', {
                    'display_format': 'yyyy-MM-dd',
                    'field_format': 'yyyy-MM-dd',
                    'calendar_popup': True
                })
                layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)