# -*- coding: utf-8 -*-
"""
Custom QGIS Edit Widget for Auto-Generating FTTX Codes
"""

from qgis.PyQt.QtWidgets import QLineEdit, QPushButton, QHBoxLayout, QWidget
from qgis.PyQt.QtCore import pyqtSignal
from qgis.gui import QgsEditorWidgetWrapper
from qgis.core import NULL


class FTTXCodeGeneratorWidget(QWidget):
    """Custom widget for generating FTTX codes"""
    
    valueChanged = pyqtSignal(str)
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setup_ui()
    
    def setup_ui(self):
        layout = QHBoxLayout()
        layout.setContentsMargins(0, 0, 0, 0)
        
        self.line_edit = QLineEdit()
        layout.addWidget(self.line_edit)
        
        self.generate_btn = QPushButton('🔄')
        self.generate_btn.setToolTip('Auto-generate code')
        self.generate_btn.setMaximumWidth(30)
        self.generate_btn.clicked.connect(self.generate_code)
        layout.addWidget(self.generate_btn)
        
        self.setLayout(layout)
    
    def generate_code(self):
        """Generate new code based on a timestamp-based unique identifier"""
        import datetime
        timestamp = datetime.datetime.now().strftime('%y%m%d%H%M%S')
        code = f'FW-{timestamp}'
        self.line_edit.setText(code)
        self.valueChanged.emit(code)
    
    def value(self):
        """Get current value"""
        text = self.line_edit.text()
        return text if text else NULL
    
    def setValue(self, value):
        """Set value"""
        if value is None or value == NULL:
            self.line_edit.clear()
        else:
            self.line_edit.setText(str(value))


class FTTXCodeGeneratorWidgetWrapper(QgsEditorWidgetWrapper):
    """Wrapper for custom code generator widget"""
    
    def createWidget(self, parent):
        return FTTXCodeGeneratorWidget(parent)
    
    def initWidget(self, editor):
        editor.valueChanged.connect(self.valueChanged)
    
    def valid(self):
        return self.widget() is not None
    
    def value(self):
        return self.widget().value()
    
    def setValue(self, value):
        self.widget().setValue(value)