# fiberweave/form_config.py
"""
Form Configuration Manager for FiberWeave
Creates intuitive, auto-populating forms for data entry
"""

from qgis.core import (
    QgsProject, QgsEditorWidgetSetup, QgsMessageLog, Qgis,
    QgsDefaultValue, QgsFieldConstraints
)
from qgis.PyQt.QtCore import QDate
from datetime import datetime


class FormConfigManager:
    """Manages QGIS form configurations for FiberWeave layers"""
    
    @staticmethod
    def configure_all_forms():
        """Configure forms for all FiberWeave layers"""
        FormConfigManager.configure_olt_equipment()
        FormConfigManager.configure_pon_ports()
        FormConfigManager.configure_odn_nodes()
        FormConfigManager.configure_onu_equipment()
        FormConfigManager.configure_fiber_cables()
        FormConfigManager.configure_buildings()
        FormConfigManager.configure_splice_points()
        FormConfigManager.configure_cable_routes()
        
        QgsMessageLog.logMessage(
            '✓ Forms configured for all FiberWeave layers',
            'FiberWeave',
            Qgis.Success
        )
    
    @staticmethod
    def get_layer(layer_name):
        """Get layer by name"""
        layers = QgsProject.instance().mapLayersByName(layer_name)
        return layers[0] if layers else None
    
    @staticmethod
    def configure_olt_equipment():
        """Configure OLT Equipment form"""
        layer = FormConfigManager.get_layer('OLT Equipment')
        if not layer:
            return
        
        # Set form config
        config = layer.editFormConfig()
        config.setLayout(config.TabLayout)
        
        # Field configurations
        fields = layer.fields()
        
        # OLT Code - required, uppercase
        idx = fields.indexOf('olt_code')
        if idx >= 0:
            layer.setFieldAlias(idx, 'OLT Code *')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'OLT-' || uuid()"))
            
        # PON Standard - dropdown
        idx = fields.indexOf('pon_standard')
        if idx >= 0:
            layer.setFieldAlias(idx, 'PON Standard *')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'GPON': 'GPON',
                    'XG-PON': 'XG-PON',
                    'XGS-PON': 'XGS-PON',
                    'NG-PON2': 'NG-PON2'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'GPON'"))
        
        # PON Class - dropdown
        idx = fields.indexOf('pon_class')
        if idx >= 0:
            layer.setFieldAlias(idx, 'PON Class')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Class B+': 'B+',
                    'Class C+': 'C+',
                    'Class C++': 'C++'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'B+'"))
        
        # Equipment Status - dropdown
        idx = fields.indexOf('equipment_status')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Status')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Planned': 'planned',
                    'Active': 'active',
                    'Inactive': 'inactive',
                    'Maintenance': 'maintenance',
                    'Decommissioned': 'decommissioned'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'active'"))
        
        # Manufacturer - text with suggestions
        idx = fields.indexOf('manufacturer')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Manufacturer')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'Huawei'"))
        
        # Installation Date - date picker
        idx = fields.indexOf('installation_date')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Installation Date')
            setup = QgsEditorWidgetSetup('DateTime', {
                'display_format': 'yyyy-MM-dd',
                'field_format': 'yyyy-MM-dd',
                'calendar_popup': True
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("to_date(now())"))
        
        # Max ONUs - default value
        idx = fields.indexOf('max_onus_supported')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Max ONUs Supported')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("512"))
        
        # Current ONUs - read-only, default 0
        idx = fields.indexOf('current_onus_connected')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Current ONUs Connected')
            setup = QgsEditorWidgetSetup('Hidden', {})
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("0"))
        
        # ID field - hidden
        idx = fields.indexOf('olt_id')
        if idx >= 0:
            setup = QgsEditorWidgetSetup('Hidden', {})
            layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    @staticmethod
    def configure_pon_ports():
        """Configure PON Ports form"""
        layer = FormConfigManager.get_layer('PON Ports')
        if not layer:
            return
        
        config = layer.editFormConfig()
        config.setLayout(config.TabLayout)
        
        fields = layer.fields()
        
        # Port Number
        idx = fields.indexOf('port_number')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Port Number *')
        
        # Port Label
        idx = fields.indexOf('port_label')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Port Label')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'PON-0/' || @feature"))
        
        # Optical Budget Class
        idx = fields.indexOf('optical_budget_class')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Optical Budget Class')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Class B+': 'B+',
                    'Class C+': 'C+',
                    'Class C++': 'C++'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'B+'"))
        
        # Max Reach
        idx = fields.indexOf('max_logical_reach_km')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Max Reach (km)')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("20"))
        
        # Port Status
        idx = fields.indexOf('port_status')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Status')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Inactive': 'inactive',
                    'Active': 'active',
                    'Maintenance': 'maintenance'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'active'"))
        
        # Max ONUs
        idx = fields.indexOf('max_onus')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Max ONUs')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("64"))
        
        # Connected ONUs - hidden/auto
        idx = fields.indexOf('connected_onus')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Connected ONUs')
            setup = QgsEditorWidgetSetup('Hidden', {})
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("0"))
        
        # ID - hidden
        idx = fields.indexOf('pon_port_id')
        if idx >= 0:
            setup = QgsEditorWidgetSetup('Hidden', {})
            layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    @staticmethod
    def configure_odn_nodes():
        """Configure ODN Nodes (Splitters) form"""
        layer = FormConfigManager.get_layer('ODN Nodes (Splitters)')
        if not layer:
            return
        
        config = layer.editFormConfig()
        config.setLayout(config.TabLayout)
        
        fields = layer.fields()
        
        # Node Code
        idx = fields.indexOf('node_code')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Node Code *')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'SPL-' || uuid()"))
        
        # Node Type
        idx = fields.indexOf('node_type')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Node Type')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Splitter': 'splitter',
                    'Distribution Point': 'distribution_point',
                    'Junction': 'junction'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'splitter'"))
        
        # Node Function
        idx = fields.indexOf('node_function')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Function')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Primary Split': 'primary_split',
                    'Secondary Split': 'secondary_split',
                    'Distribution': 'distribution'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # Splitter Type
        idx = fields.indexOf('splitter_type')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Splitter Type')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'PLC (Planar Lightwave Circuit)': 'PLC',
                    'FBT (Fused Biconical Taper)': 'FBT'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'PLC'"))
        
        # Split Ratio
        idx = fields.indexOf('split_ratio')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Split Ratio *')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    '1:2': '1:2',
                    '1:4': '1:4',
                    '1:8': '1:8',
                    '1:16': '1:16',
                    '1:32': '1:32',
                    '1:64': '1:64'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'1:32'"))
        
        # Insertion Loss - auto-calculate based on split ratio
        idx = fields.indexOf('insertion_loss_typical_db')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Insertion Loss (dB)')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue(
                "CASE "
                "WHEN split_ratio = '1:2' THEN 3.5 "
                "WHEN split_ratio = '1:4' THEN 7.0 "
                "WHEN split_ratio = '1:8' THEN 10.5 "
                "WHEN split_ratio = '1:16' THEN 14.0 "
                "WHEN split_ratio = '1:32' THEN 17.5 "
                "WHEN split_ratio = '1:64' THEN 21.0 "
                "ELSE 17.5 END"
            ))
        
        # Output Ports - auto from split ratio
        idx = fields.indexOf('output_ports')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Output Ports')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue(
                "to_int(regexp_substr(split_ratio, ':(\\d+)', 1))"
            ))
        
        # Installation Type
        idx = fields.indexOf('installation_type')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Installation Type')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Cabinet': 'cabinet',
                    'Street Cabinet': 'street_cabinet',
                    'Pole Mount': 'pole_mount',
                    'Underground Vault': 'underground_vault'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # Operational Status
        idx = fields.indexOf('operational_status')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Status')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Planned': 'planned',
                    'Active': 'active',
                    'Inactive': 'inactive',
                    'Maintenance': 'maintenance'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'active'"))
        
        # Installation Date
        idx = fields.indexOf('installation_date')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Installation Date')
            setup = QgsEditorWidgetSetup('DateTime', {
                'display_format': 'yyyy-MM-dd',
                'calendar_popup': True
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("to_date(now())"))
        
        # ID - hidden
        idx = fields.indexOf('node_id')
        if idx >= 0:
            setup = QgsEditorWidgetSetup('Hidden', {})
            layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    @staticmethod
    def configure_onu_equipment():
        """Configure ONU Equipment form"""
        layer = FormConfigManager.get_layer('ONU/ONT Equipment')
        if not layer:
            return
        
        config = layer.editFormConfig()
        config.setLayout(config.TabLayout)
        
        fields = layer.fields()
        
        # Serial Number
        idx = fields.indexOf('onu_serial_number')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Serial Number *')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'HWTC' || uuid()"))
        
        # PON Standard
        idx = fields.indexOf('pon_standard')
        if idx >= 0:
            layer.setFieldAlias(idx, 'PON Standard')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'GPON': 'GPON',
                    'XG-PON': 'XG-PON',
                    'XGS-PON': 'XGS-PON'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'GPON'"))
        
        # ONU Class
        idx = fields.indexOf('onu_class')
        if idx >= 0:
            layer.setFieldAlias(idx, 'ONU Class')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {'B+': 'B+', 'C+': 'C+', 'C++': 'C++'}
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'B+'"))
        
        # Ethernet Ports
        idx = fields.indexOf('ethernet_ports')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Ethernet Ports')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("4"))
        
        # POTS Ports
        idx = fields.indexOf('pots_ports')
        if idx >= 0:
            layer.setFieldAlias(idx, 'POTS/Voice Ports')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("2"))
        
        # ONU Status
        idx = fields.indexOf('onu_status')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Status')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Planned': 'planned',
                    'Active': 'active',
                    'Offline': 'offline',
                    'Maintenance': 'maintenance'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'active'"))
        
        # Manufacturer
        idx = fields.indexOf('manufacturer')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Manufacturer')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'Huawei'"))
        
        # Installation Date
        idx = fields.indexOf('installation_date')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Installation Date')
            setup = QgsEditorWidgetSetup('DateTime', {
                'display_format': 'yyyy-MM-dd',
                'calendar_popup': True
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("to_date(now())"))
        
        # Activation Date
        idx = fields.indexOf('activation_date')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Activation Date')
            setup = QgsEditorWidgetSetup('DateTime', {
                'display_format': 'yyyy-MM-dd',
                'calendar_popup': True
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # Installation Address
        idx = fields.indexOf('installation_address')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Installation Address')
            setup = QgsEditorWidgetSetup('TextEdit', {
                'IsMultiline': True
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # ID - hidden
        idx = fields.indexOf('onu_id')
        if idx >= 0:
            setup = QgsEditorWidgetSetup('Hidden', {})
            layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    @staticmethod
    def configure_fiber_cables():
        """Configure Fiber Cables form"""
        layer = FormConfigManager.get_layer('Fiber Cables')
        if not layer:
            return
        
        config = layer.editFormConfig()
        config.setLayout(config.TabLayout)
        
        fields = layer.fields()
        
        # Cable Code
        idx = fields.indexOf('cable_code')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Cable Code *')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'FB-' || uuid()"))
        
        # Cable Category
        idx = fields.indexOf('cable_category')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Cable Category')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Feeder': 'feeder',
                    'Distribution': 'distribution',
                    'Drop': 'drop'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # Cable Type
        idx = fields.indexOf('cable_type')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Cable Type')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Loose Tube': 'loose_tube',
                    'Tight Buffer': 'tight_buffer',
                    'Ribbon': 'ribbon',
                    'ADSS': 'adss'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # Fiber Type
        idx = fields.indexOf('fiber_type')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Fiber Type')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'G.652D (Standard SMF)': 'G.652D',
                    'G.655 (NZDSF)': 'G.655',
                    'G.657 (Bend-Insensitive)': 'G.657'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'G.652D'"))
        
        # Fiber Count
        idx = fields.indexOf('fiber_count')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Fiber Count *')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    '12': 12, '24': 24, '48': 48, '72': 72,
                    '96': 96, '144': 144, '288': 288
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("48"))
        
        # Max Attenuation
        idx = fields.indexOf('max_attenuation_db_per_km')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Max Attenuation (dB/km)')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("0.35"))
        
        # Operational Status
        idx = fields.indexOf('operational_status')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Status')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Planned': 'planned',
                    'Active': 'active',
                    'Maintenance': 'maintenance',
                    'Damaged': 'damaged'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'active'"))
        
        # Installation Date
        idx = fields.indexOf('installation_date')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Installation Date')
            setup = QgsEditorWidgetSetup('DateTime', {
                'display_format': 'yyyy-MM-dd',
                'calendar_popup': True
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("to_date(now())"))
        
        # Cable Length - auto-calculate from geometry
        idx = fields.indexOf('cable_length_m')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Cable Length (m)')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("$length"))
        
        # ID - hidden
        idx = fields.indexOf('cable_id')
        if idx >= 0:
            setup = QgsEditorWidgetSetup('Hidden', {})
            layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    @staticmethod
    def configure_buildings():
        """Configure Buildings form"""
        layer = FormConfigManager.get_layer('Buildings')
        if not layer:
            return
        
        config = layer.editFormConfig()
        config.setLayout(config.TabLayout)
        
        fields = layer.fields()
        
        # Building Code
        idx = fields.indexOf('building_code')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Building Code')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'BLDG-' || uuid()"))
        
        # Building Name
        idx = fields.indexOf('building_name')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Building Name *')
        
        # Building Type
        idx = fields.indexOf('building_type')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Building Type')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Residential': 'residential',
                    'Commercial': 'commercial',
                    'Industrial': 'industrial',
                    'Mixed Use': 'mixed_use',
                    'MDU (Multi-Dwelling Unit)': 'mdu'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # Service Status
        idx = fields.indexOf('service_status')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Service Status')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Potential': 'potential',
                    'Planned': 'planned',
                    'Under Construction': 'under_construction',
                    'Active': 'active',
                    'No Service': 'no_service'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'potential'"))
        
        # Address
        idx = fields.indexOf('address')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Street Address')
            setup = QgsEditorWidgetSetup('TextEdit', {
                'IsMultiline': True
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # Floors
        idx = fields.indexOf('floors')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Number of Floors')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("1"))
        
        # Units
        idx = fields.indexOf('units')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Number of Units')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("1"))
        
        # Installation Date
        idx = fields.indexOf('installation_date')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Service Installation Date')
            setup = QgsEditorWidgetSetup('DateTime', {
                'display_format': 'yyyy-MM-dd',
                'calendar_popup': True
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # Notes
        idx = fields.indexOf('notes')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Notes')
            setup = QgsEditorWidgetSetup('TextEdit', {
                'IsMultiline': True
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # ID - hidden
        idx = fields.indexOf('building_id')
        if idx >= 0:
            setup = QgsEditorWidgetSetup('Hidden', {})
            layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    @staticmethod
    def configure_splice_points():
        """Configure Splice Points form"""
        layer = FormConfigManager.get_layer('Splice Points')
        if not layer:
            return
        
        config = layer.editFormConfig()
        config.setLayout(config.TabLayout)
        
        fields = layer.fields()
        
        # Splice Code
        idx = fields.indexOf('splice_code')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Splice Code')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'SPLICE-' || uuid()"))
        
        # Splice Type
        idx = fields.indexOf('splice_type')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Splice Type')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Distribution': 'distribution',
                    'Branching': 'branching',
                    'Repair': 'repair'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # Splice Method
        idx = fields.indexOf('splice_method')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Splice Method')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Fusion Splice': 'fusion',
                    'Mechanical Splice': 'mechanical'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'fusion'"))
        
        # Typical Loss
        idx = fields.indexOf('typical_splice_loss_db')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Typical Loss (dB)')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("0.05"))
        
        # Closure Type
        idx = fields.indexOf('closure_type')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Closure Type')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Dome Closure (96 fiber)': 'dome_closure_96',
                    'Dome Closure (288 fiber)': 'dome_closure_288',
                    'Inline Closure (96 fiber)': 'inline_closure_96',
                    'Inline Closure (144 fiber)': 'inline_closure_144'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # OTDR Tested
        idx = fields.indexOf('otdr_tested')
        if idx >= 0:
            layer.setFieldAlias(idx, 'OTDR Tested')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("false"))
        
        # Test Date
        idx = fields.indexOf('test_date')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Test Date')
            setup = QgsEditorWidgetSetup('DateTime', {
                'display_format': 'yyyy-MM-dd',
                'calendar_popup': True
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # Splice Date
        idx = fields.indexOf('splice_date')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Splice Date')
            setup = QgsEditorWidgetSetup('DateTime', {
                'display_format': 'yyyy-MM-dd',
                'calendar_popup': True
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("to_date(now())"))
        
        # ID - hidden
        idx = fields.indexOf('splice_id')
        if idx >= 0:
            setup = QgsEditorWidgetSetup('Hidden', {})
            layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)
    
    @staticmethod
    def configure_cable_routes():
        """Configure Cable Routes form"""
        layer = FormConfigManager.get_layer('Cable Routes')
        if not layer:
            return
        
        config = layer.editFormConfig()
        config.setLayout(config.TabLayout)
        
        fields = layer.fields()
        
        # Route Code
        idx = fields.indexOf('route_code')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Route Code *')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'ROUTE-' || uuid()"))
        
        # Route Name
        idx = fields.indexOf('route_name')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Route Name')
        
        # Infrastructure Type
        idx = fields.indexOf('infrastructure_type')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Infrastructure Type')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Underground': 'underground',
                    'Aerial': 'aerial',
                    'Buried Direct': 'buried_direct',
                    'Building Riser': 'building_riser'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # Route Status
        idx = fields.indexOf('route_status')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Status')
            setup = QgsEditorWidgetSetup('ValueMap', {
                'map': {
                    'Planned': 'planned',
                    'Under Construction': 'under_construction',
                    'Active': 'active',
                    'Maintenance': 'maintenance'
                }
            })
            layer.setEditorWidgetSetup(idx, setup)
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("'planned'"))
        
        # Route Length - auto-calculate
        idx = fields.indexOf('route_length_m')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Route Length (m)')
            layer.setDefaultValueDefinition(idx, QgsDefaultValue("$length"))
        
        # Construction Date
        idx = fields.indexOf('construction_date')
        if idx >= 0:
            layer.setFieldAlias(idx, 'Construction Date')
            setup = QgsEditorWidgetSetup('DateTime', {
                'display_format': 'yyyy-MM-dd',
                'calendar_popup': True
            })
            layer.setEditorWidgetSetup(idx, setup)
        
        # ID - hidden
        idx = fields.indexOf('route_id')
        if idx >= 0:
            setup = QgsEditorWidgetSetup('Hidden', {})
            layer.setEditorWidgetSetup(idx, setup)
        
        layer.setEditFormConfig(config)