# fiberweave/fiberweave.py
"""
FiberWeave Main Plugin Class
Implements the 12-button toolbar with dropdown menus
"""

import os
import psycopg2
from qgis.core import QgsApplication
from .sample_data_generator import SampleDataGenerator
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QMenu
from qgis.core import QgsMessageLog, Qgis
from .database_installer import DatabaseInstaller
from .algorithms.provider import FiberWeaveProvider
from .snapping_config import SnappingManager
from .form_config import FormConfigManager


# Import database utilities
from .db.database_utils import DatabaseConnectionDialog

# Import GUI dialogs
from .gui.dialogs import (
    AddPONPortDialog, LabelGenerator, NetworkDashboard,
    DocumentationGenerator, OTDRAnalyzer
)
from .gui.calculators_ui import (
    PowerBudgetCalculator, CableTensionCalculator, AerialSlackCalculator
)
from .gui.utils_ui import (
    SpliceDiagramViewer, ButterflyClosureViewer, FiberColorReference
)
from .gui.plant_manager import PlanningWizard, InsidePlantManager
from .gui.bom_generator import BOMGenerator
from .gui.data_import import DataImportManager

# Import layer styling
from .fttx_layer_styles import FTTXLayerStyles


class FiberWeavePlugin:
    """FiberWeave - Professional Fiber Optic Network Planning Plugin"""

    def __init__(self, iface):
        """Initialize FiberWeave Plugin
        
        Args:
            iface: QGIS interface instance
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = 'FiberWeave'
        self.toolbar = None
        self.db_connection = None
        self.connection_params = {}
        
        # Menus for dropdowns
        self.calculators_menu = None
        self.analysis_menu = None
        self.reports_menu = None
        
        # Processing provider
        self.provider = None
        
        QgsMessageLog.logMessage(
            '=' * 60 + '\nFiberWeave: Initializing v1.0.0...\n' + '=' * 60,
            'FiberWeave',
            Qgis.Info
        )

    def initGui(self):
        """Initialize GUI - Creates the 12-button toolbar"""
        
        self.toolbar = self.iface.addToolBar('FiberWeave')
        self.toolbar.setObjectName('FiberWeaveToolbar')
        self.toolbar.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
        self.add_action('🧲 Snapping', self.show_snapping_settings,
                       'Configure snapping settings')
        
        QgsMessageLog.logMessage('Creating FiberWeave toolbar...', 'FiberWeave', Qgis.Info)
        
        # ==========================================
        # GROUP 1: CONNECTION (4 buttons)
        # ==========================================
        self.add_action('🔧 Setup DB', self.show_database_installer,
                       'Install FiberWeave database')
        
        self.add_action('🔌 Connect DB', self.show_connection_dialog, 
                       'Connect to PostgreSQL/PostGIS database')
        
        self.add_action('🗺️ Load Layers', self.add_fiberweave_layers,
                       'Add FiberWeave network layers to map')
        
        self.add_action('🎨 Apply Styles', self.apply_layer_styles,
                       'Apply predefined styles to layers')
        
        self.toolbar.addSeparator()
        
        # ==========================================
        # GROUP 2: DESIGN (3 buttons)
        # ==========================================
        self.add_action('🧙 Planning Wizard', self.show_planning_wizard,
                       'Launch network planning wizard')
        
        self.add_action('➕ Add PON Port', self.show_add_pon_port_dialog,
                       'Add PON port to network')
        
        self.add_action('🏢 Inside Plant', self.show_inside_plant_manager,
                       'Manage inside plant infrastructure')
        
        self.toolbar.addSeparator()
        
        # ==========================================
        # GROUP 3: TOOLS (2 dropdown menus)
        # ==========================================
        
        # CALCULATORS DROPDOWN
        calc_action = self.add_action('⚡ Calculators ▼', None,
                                     'Network calculators and tools')
        self.calculators_menu = QMenu()
        self.calculators_menu.addAction('⚡ Power Budget Calculator', 
                                       self.show_power_budget_calculator)
        self.calculators_menu.addAction('🔧 Cable Tension Calculator', 
                                       self.show_cable_tension_calculator)
        self.calculators_menu.addAction('📏 Aerial Slack Calculator', 
                                       self.show_aerial_slack_calculator)
        calc_action.setMenu(self.calculators_menu)
        
        # ANALYSIS DROPDOWN
        analysis_action = self.add_action('📊 Analysis ▼', None,
                                         'Network analysis tools')
        self.analysis_menu = QMenu()
        self.analysis_menu.addAction('📊 Network Dashboard', 
                                    self.show_network_dashboard)
        self.analysis_menu.addAction('🔍 OTDR Analyzer', 
                                    self.show_otdr_analyzer)
        self.analysis_menu.addSeparator()
        self.analysis_menu.addAction('🔗 Splice Diagram Viewer', 
                                    self.show_splice_diagram)
        self.analysis_menu.addAction('🦋 Butterfly Closure Viewer', 
                                    self.show_butterfly_closure)
        self.analysis_menu.addAction('🎨 Fiber Color Reference', 
                                    self.show_fiber_color_reference)
        analysis_action.setMenu(self.analysis_menu)
        
        self.toolbar.addSeparator()
        
        # ==========================================
        # GROUP 4: AUTOMATION (2 buttons)
        # ==========================================
        self.add_action('🤖 AI Optimize', self.show_ai_optimizer,
                       'AI-powered route optimization (Phase 2)')
        
        self.add_action('🔧 Processing', self.open_processing_toolbox,
                       'Open Processing Toolbox for algorithms')
        
        self.toolbar.addSeparator()
        
        # ==========================================
        # GROUP 5: DOCUMENTATION (2 buttons)
        # ==========================================
        
        # REPORTS DROPDOWN
        reports_action = self.add_action('📋 Reports ▼', None,
                                        'Generate reports and documentation')
        self.reports_menu = QMenu()
        self.reports_menu.addAction('📋 BOM Generator', 
                                   self.show_bom_generator)
        self.reports_menu.addAction('📄 Documentation Generator', 
                                   self.show_documentation_generator)
        self.reports_menu.addAction('🏷️ Label Generator', 
                                   self.show_label_generator)
        self.reports_menu.addSeparator()
        self.reports_menu.addAction('📥 Import Data', 
                                   self.show_data_import)
        self.reports_menu.addAction('🎲 Generate Sample Data', 
                                   self.show_sample_data_generator)
        reports_action.setMenu(self.reports_menu)
        
        # ABOUT
        self.add_action('ℹ️ About', self.show_about,
                       'About FiberWeave')

        # Initialize Processing Provider
        self.initProcessing()

        # Show welcome message
        QMessageBox.information(
            self.iface.mainWindow(),
            'FiberWeave',
            '<h2>🌐 FiberWeave v1.0.0 Loaded!</h2>'
            '<p><b>Professional Fiber Optic Network Planning & Management</b></p>'
            '<hr>'
            '<p><b>Quick Start:</b></p>'
            '<ol>'
            '<li>Click <b>🔌 Connect DB</b> to connect to your database</li>'
            '<li>Click <b>🗺️ Load Layers</b> to add network layers</li>'
            '<li>Use <b>🧙 Planning Wizard</b> to start designing</li>'
            '</ol>'
            '<hr>'
            '<p><i>Author: Clinton Papenfus (clinton@fnfast.co.za)</i></p>'
        )

        QgsMessageLog.logMessage('✓ FiberWeave toolbar created successfully',
                                'FiberWeave', Qgis.Success)

    def show_database_installer(self):
        """Show database installation wizard"""
        installer = DatabaseInstaller(self.iface.mainWindow())
        installer.exec_()

    def add_action(self, text, callback, tooltip=''):
        """Helper to add toolbar action
        
        Args:
            text: Button text
            callback: Function to call when clicked
            tooltip: Tooltip text
            
        Returns:
            QAction: The created action
        """
        action = QAction(text, self.iface.mainWindow())
        if callback:
            action.triggered.connect(callback)
        action.setStatusTip(tooltip)
        action.setToolTip(tooltip)
        
        self.toolbar.addAction(action)
        self.iface.addPluginToMenu(self.menu, action)
        self.actions.append(action)
        
        return action

    def unload(self):
        """Remove plugin and clean up resources"""
        QgsMessageLog.logMessage('FiberWeave: Unloading...', 'FiberWeave', Qgis.Info)
        
        # Remove processing provider FIRST
        if self.provider:
            QgsApplication.processingRegistry().removeProvider(self.provider)
            self.provider = None
        
        # Remove actions
        for action in self.actions:
            self.iface.removePluginMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)
        
        # Remove toolbar
        if self.toolbar:
            self.toolbar.setVisible(False)
            self.iface.mainWindow().removeToolBar(self.toolbar)
            self.toolbar.deleteLater()
            self.toolbar = None
        
        # Close database connection
        if self.db_connection:
            try:
                self.db_connection.close()
            except:
                pass
        
        QgsMessageLog.logMessage('✓ FiberWeave unloaded', 'FiberWeave', Qgis.Info)
        
        
    def initProcessing(self):
        """Initialize Processing provider"""
        try:
            self.provider = FiberWeaveProvider()
            QgsApplication.processingRegistry().addProvider(self.provider)
            QgsMessageLog.logMessage('✓ Processing provider registered', 'FiberWeave', Qgis.Success)
        except Exception as e:
            QgsMessageLog.logMessage(
                f'✗ Error registering Processing provider: {e}',
                'FiberWeave',
                Qgis.Critical
            )    

    # ==========================================
    # DATABASE & CONNECTION METHODS
    # ==========================================

    def show_connection_dialog(self):
        """Show database connection dialog"""
        dialog = DatabaseConnectionDialog(self)
        if dialog.exec_():
            self.connection_params = dialog.get_params()
            QgsMessageLog.logMessage('✓ Database connected', 'FiberWeave', Qgis.Success)

    def connect_database(self, host, database, user, password, port=5432):
        """Connect to PostgreSQL database
        
        Args:
            host: Database host
            database: Database name
            user: Username
            password: Password
            port: Port number (default 5432)
            
        Returns:
            bool: True if successful
        """
        try:
            if self.db_connection:
                self.db_connection.close()
                
            self.db_connection = psycopg2.connect(
                host=host, database=database, user=user, 
                password=password, port=port,
                options="-c search_path=fttx,public"
            )
            
            self.connection_params = {
                'host': host, 'database': database, 'user': user,
                'password': password, 'port': port
            }
            
            self.iface.messageBar().pushSuccess(
                'FiberWeave', 
                'Database connected successfully'
            )
            return True
            
        except Exception as e:
            self.iface.messageBar().pushCritical(
                'FiberWeave', 
                f'Connection error: {str(e)}'
            )
            return False

    def add_fiberweave_layers(self):
        """Add FiberWeave layers to QGIS project"""
        if not self.connection_params:
            QMessageBox.warning(
                self.iface.mainWindow(), 
                'No Connection', 
                'Please connect to database first using 🔌 Connect DB'
            )
            return
        
        from qgis.core import QgsProject, QgsVectorLayer, QgsDataSourceUri
        from .config.constants import CORE_LAYERS_TO_ADD

        added = 0
        for table, key, name in CORE_LAYERS_TO_ADD:
            uri = QgsDataSourceUri()
            uri.setConnection(
                self.connection_params['host'], 
                str(self.connection_params['port']),
                self.connection_params['database'],
                self.connection_params['user'],
                self.connection_params['password']
            )
            
            geom_field = 'edge_geom' if table == 'network_topology' else 'geom'
            uri.setDataSource('fttx', table, geom_field, '', key)
            
            layer = QgsVectorLayer(uri.uri(), name, 'postgres')
            if layer.isValid():
                QgsProject.instance().addMapLayer(layer)
                added += 1
        
        if added > 0:
            # Auto-apply styles after adding layers
            FTTXLayerStyles.apply_all_styles(self)
            
            # Configure snapping
            SnappingManager.configure_snapping(tolerance=10, enable_topological=True)
            
            
            # Configure forms
            FormConfigManager.configure_all_forms()
            
            QMessageBox.information(
                self.iface.mainWindow(), 
                'Success', 
                f'✓ Added {added} layers with styling applied!\n'
                f'Snapping is now enabled:\n'
                f'  • Tolerance: 10 pixels\n'
                f'  • Type: Vertex and Segment\n'
                f'  • Topological editing: Enabled\n\n'
                f'Start editing any layer to use snapping.'
            )
        else:
            QMessageBox.warning(
                self.iface.mainWindow(), 
                'Warning', 
                'No layers added. Check database schema.'
            )

    def apply_layer_styles(self):
        """Apply predefined styles to FiberWeave layers"""
        FTTXLayerStyles.apply_all_styles(self)
        
    def show_snapping_settings(self):
        """Show snapping configuration dialog"""
        from qgis.PyQt.QtWidgets import QDialog, QVBoxLayout, QHBoxLayout, QLabel, QSpinBox, QCheckBox, QPushButton
        
        dialog = QDialog(self.iface.mainWindow())
        dialog.setWindowTitle('FiberWeave Snapping Settings')
        dialog.setMinimumWidth(400)
        
        layout = QVBoxLayout()
        
        # Title
        title = QLabel('<h3>🧲 Snapping Configuration</h3>')
        title.setAlignment(Qt.AlignCenter)
        layout.addWidget(title)
        
        # Current settings
        info = SnappingManager.get_snapping_info()
        current = QLabel(
            f'<b>Current Settings:</b><br>'
            f'Enabled: {"Yes" if info["enabled"] else "No"}<br>'
            f'Tolerance: {info["tolerance"]} pixels<br>'
            f'Topological: {"Yes" if info["topological"] else "No"}'
        )
        layout.addWidget(current)
        
        layout.addWidget(QLabel('<hr>'))
        
        # Tolerance setting
        tolerance_layout = QHBoxLayout()
        tolerance_layout.addWidget(QLabel('Snapping Tolerance (pixels):'))
        tolerance_spin = QSpinBox()
        tolerance_spin.setMinimum(1)
        tolerance_spin.setMaximum(50)
        tolerance_spin.setValue(int(info.get('tolerance', 10)))
        tolerance_layout.addWidget(tolerance_spin)
        layout.addLayout(tolerance_layout)
        
        # Topological editing
        topo_check = QCheckBox('Enable Topological Editing')
        topo_check.setChecked(info.get('topological', True))
        layout.addWidget(topo_check)
        
        # Intersection snapping
        intersect_check = QCheckBox('Enable Snapping on Intersection')
        intersect_check.setChecked(info.get('intersection', True))
        layout.addWidget(intersect_check)
        
        layout.addWidget(QLabel('<hr>'))
        
        # Buttons
        button_layout = QHBoxLayout()
        
        apply_btn = QPushButton('Apply Settings')
        apply_btn.clicked.connect(lambda: self.apply_snapping_settings(
            tolerance_spin.value(),
            topo_check.isChecked(),
            dialog
        ))
        
        close_btn = QPushButton('Close')
        close_btn.clicked.connect(dialog.reject)
        
        button_layout.addWidget(apply_btn)
        button_layout.addWidget(close_btn)
        
        layout.addLayout(button_layout)
        dialog.setLayout(layout)
        dialog.exec_()
    
    def apply_snapping_settings(self, tolerance, topological, dialog):
        """Apply snapping settings"""
        success = SnappingManager.configure_snapping(
            tolerance=tolerance,
            enable_topological=topological
        )
        
        if success:
            QMessageBox.information(
                dialog,
                'Success',
                f'✓ Snapping settings applied!\n\n'
                f'Tolerance: {tolerance} pixels\n'
                f'Topological editing: {"Enabled" if topological else "Disabled"}'
            )
            dialog.accept()
        
    def show_sample_data_generator(self):
        """Show sample data generator"""
        generator = SampleDataGenerator(self)
        generator.exec_()

    # ==========================================
    # DIALOG LAUNCHING METHODS
    # ==========================================

    def _check_connection_and_show(self, dialog_class):
        """Helper to check DB connection before showing dialog"""
        if not self.db_connection and not self.connection_params:
            QMessageBox.warning(
                self.iface.mainWindow(),
                'No Connection',
                'Please connect to database first using 🔌 Connect DB'
            )
            return

        # If we have connection_params but no db_connection, establish one
        if not self.db_connection and self.connection_params:
            try:
                self.db_connection = psycopg2.connect(
                    host=self.connection_params['host'],
                    database=self.connection_params['database'],
                    user=self.connection_params['user'],
                    password=self.connection_params['password'],
                    port=self.connection_params['port'],
                    options="-c search_path=fttx,public"
                )
            except Exception as e:
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    'Connection Error',
                    f'Failed to connect to database:\n\n{str(e)}'
                )
                return

        dialog = dialog_class(self)
        dialog.exec_()

    # Design Tools
    def show_planning_wizard(self):
        self._check_connection_and_show(PlanningWizard)
    
    def show_add_pon_port_dialog(self):
        self._check_connection_and_show(AddPONPortDialog)
    
    def show_inside_plant_manager(self):
        self._check_connection_and_show(InsidePlantManager)

    # Calculators
    def show_power_budget_calculator(self):
        self._check_connection_and_show(PowerBudgetCalculator)
    
    def show_cable_tension_calculator(self):
        self._check_connection_and_show(CableTensionCalculator)
    
    def show_aerial_slack_calculator(self):
        self._check_connection_and_show(AerialSlackCalculator)

    # Analysis Tools
    def show_network_dashboard(self):
        self._check_connection_and_show(NetworkDashboard)
    
    def show_otdr_analyzer(self):
        self._check_connection_and_show(OTDRAnalyzer)
    
    def show_splice_diagram(self):
        self._check_connection_and_show(SpliceDiagramViewer)
    
    def show_butterfly_closure(self):
        self._check_connection_and_show(ButterflyClosureViewer)
    
    def show_fiber_color_reference(self):
        self._check_connection_and_show(FiberColorReference)

    # Reports & Documentation
    def show_bom_generator(self):
        self._check_connection_and_show(BOMGenerator)
    
    def show_documentation_generator(self):
        self._check_connection_and_show(DocumentationGenerator)
    
    def show_label_generator(self):
        self._check_connection_and_show(LabelGenerator)
    
    def show_data_import(self):
        self._check_connection_and_show(DataImportManager)

    # ==========================================
    # AUTOMATION & PROCESSING
    # ==========================================

    def show_ai_optimizer(self):
        """AI Route Optimizer - Coming in Phase 2"""
        QMessageBox.information(
            self.iface.mainWindow(),
            'AI Route Optimizer',
            '<h3>🤖 AI-Powered Route Optimization</h3>'
            '<p><b>Status:</b> Coming in Phase 2</p>'
            '<hr>'
            '<p>This feature will provide:</p>'
            '<ul>'
            '<li>Automated optimal route calculation</li>'
            '<li>Cost optimization algorithms</li>'
            '<li>Terrain analysis integration</li>'
            '<li>Multi-objective optimization</li>'
            '</ul>'
            '<p><i>Stay tuned for the next update!</i></p>'
        )

    def open_processing_toolbox(self):
        """Open QGIS Processing Toolbox"""
        try:
            from qgis.PyQt.QtWidgets import QAction
            actions = self.iface.mainWindow().findChildren(QAction)
            
            for action in actions:
                action_text = action.text().lower()
                if 'toolbox' in action_text and 'processing' in action_text:
                    action.trigger()
                    
                    QMessageBox.information(
                        self.iface.mainWindow(),
                        'Processing Toolbox',
                        '✓ Processing Toolbox opened!\n\n'
                        'Look for <b>"FiberWeave"</b> in the providers list.\n\n'
                        'Available algorithms:\n'
                        '  • Validate Network\n'
                        '  • Power Budget Batch Analysis\n'
                        '  • Export Network Report\n\n'
                        'You can also search for "fiberweave" to find them quickly.'
                    )
                    return
            
            # Fallback message
            QMessageBox.information(
                self.iface.mainWindow(),
                'Open Processing Toolbox',
                '<b>To access FiberWeave algorithms:</b>\n\n'
                '<b>Method 1: Keyboard Shortcut</b>\n'
                '   Press <b>Ctrl+Alt+T</b>\n\n'
                '<b>Method 2: Menu</b>\n'
                '   Go to: <b>Processing → Toolbox</b>\n\n'
                '<b>Then:</b>\n'
                '   • Look for <b>FiberWeave</b> in the providers\n'
                '   • Or search: <b>fiberweave</b>\n\n'
                '<b>Available Algorithms:</b>\n'
                '   • Validate Network\n'
                '   • Power Budget Batch Analysis\n'
                '   • Export Network Report'
            )
            
        except Exception as e:
            QMessageBox.warning(
                self.iface.mainWindow(),
                'Error',
                f'Could not open Processing Toolbox: {str(e)}'
            )

    # ==========================================
    # ABOUT & INFO
    # ==========================================

    def show_about(self):
        """Show About dialog"""
        QMessageBox.about(
            self.iface.mainWindow(),
            'About FiberWeave',
            '<h2>🌐 FiberWeave v1.0.0</h2>'
            '<p><b>Professional Fiber Optic Network Planning & Management</b></p>'
            '<hr>'
            '<h3>Features:</h3>'
            '<ul>'
            '<li>✓ ITU-T compliant FTTX planning (GPON, XG-PON)</li>'
            '<li>✓ Power budget & loss calculations</li>'
            '<li>✓ AI-powered route optimization</li>'
            '<li>✓ BOM generation & documentation</li>'
            '<li>✓ Inside plant management</li>'
            '<li>✓ Network analysis & validation</li>'
            '<li>✓ Multi-standard support (Metro, Long-haul)</li>'
            '</ul>'
            '<hr>'
            '<h3>Author:</h3>'
            '<p><b>Clinton Papenfus</b><br>'
            'Email: <a href="mailto:clinton@fnfast.co.za">clinton@fnfast.co.za</a></p>'
            '<hr>'
            '<p><i>© 2025 FNFast - Professional Fiber Network Solutions</i></p>'
        )