# itu_fttx_plugin/db/database_utils.py

import psycopg2
from qgis.PyQt.QtWidgets import (
    QDialog, QMessageBox, QVBoxLayout, QHBoxLayout, 
    QLabel, QPushButton, QLineEdit, QSpinBox, QGroupBox, QFormLayout
)
from qgis.PyQt.QtCore import Qt


class DatabaseConnectionDialog(QDialog):
    """Handles the PostgreSQL connection GUI and logic."""

    def __init__(self, plugin):
        super().__init__(plugin.iface.mainWindow())
        self.plugin = plugin
        self.setWindowTitle('Database Connection')
        self.setMinimumWidth(450)
        self.init_ui()
    
    def init_ui(self):
        layout = QVBoxLayout()
        header = QLabel('🔌 PostgreSQL Database Connection')
        header.setStyleSheet('font-size: 14pt; font-weight: bold; padding: 10px;')
        layout.addWidget(header)
        
        form_group = QGroupBox('Connection Parameters')
        form_layout = QFormLayout()
        
        self.host_input = QLineEdit('localhost')
        form_layout.addRow('Host:', self.host_input)
        
        self.port_input = QSpinBox()
        self.port_input.setRange(1, 65535)
        self.port_input.setValue(5432)
        form_layout.addRow('Port:', self.port_input)
        
        self.database_input = QLineEdit('fiberweave_network')
        form_layout.addRow('Database:', self.database_input)

        self.user_input = QLineEdit('budget_user')
        form_layout.addRow('User:', self.user_input)
        
        self.password_input = QLineEdit()
        self.password_input.setEchoMode(QLineEdit.Password)
        form_layout.addRow('Password:', self.password_input)
        
        form_group.setLayout(form_layout)
        layout.addWidget(form_group)
        
        # Buttons
        button_layout = QHBoxLayout()
        test_btn = QPushButton('🔍 Test Connection')
        test_btn.clicked.connect(self.test_connection)
        button_layout.addWidget(test_btn)
        connect_btn = QPushButton('✓ Connect')
        connect_btn.clicked.connect(self.connect)
        connect_btn.setDefault(True)
        connect_btn.setStyleSheet('background-color: #4CAF50; color: white; padding: 5px;')
        button_layout.addWidget(connect_btn)
        cancel_btn = QPushButton('✗ Cancel')
        cancel_btn.clicked.connect(self.reject)
        button_layout.addWidget(cancel_btn)
        layout.addLayout(button_layout)
        
        self.setLayout(layout)

    def get_params(self):
        return {
            'host': self.host_input.text(),
            'database': self.database_input.text(),
            'user': self.user_input.text(),
            'password': self.password_input.text(),
            'port': self.port_input.value()
        }
    
    def test_connection(self):
        try:
            params = self.get_params()
            conn = psycopg2.connect(
                host=params['host'], database=params['database'],
                user=params['user'], password=params['password'], port=params['port']
            )
            cursor = conn.cursor()
            cursor.execute("SELECT version();")
            version = cursor.fetchone()[0]
            cursor.close()
            conn.close()
            QMessageBox.information(self, '✓ Success', f'Connection successful!\n\nPostgreSQL Version:\n{version[:100]}...')
        except Exception as e:
            QMessageBox.critical(self, '✗ Error', f'Connection failed:\n\n{str(e)}')
    
    def connect(self):
        params = self.get_params()
        if self.plugin.connect_database(params['host'], params['database'],
                                        params['user'], params['password'], params['port']):
            self.accept()
        else:
            QMessageBox.critical(self, '✗ Error', 'Failed to connect')