# fiberweave/database_installer.py
"""
Database Installation Helper for FiberWeave
Automates database setup from within QGIS
"""

import os
import psycopg2
from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QLabel, 
    QLineEdit, QPushButton, QTextEdit, QMessageBox,
    QGroupBox, QFormLayout, QProgressBar
)
from qgis.PyQt.QtCore import Qt
from qgis.core import QgsMessageLog, Qgis


class DatabaseInstaller(QDialog):
    """Database Setup Dialog"""
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle('FiberWeave Database Installer')
        self.setMinimumWidth(600)
        self.setMinimumHeight(500)
        
        self.connection = None
        self.plugin_dir = os.path.dirname(__file__)
        
        self.init_ui()
    
    def init_ui(self):
        """Initialize UI"""
        layout = QVBoxLayout()
        
        # Title
        title = QLabel('<h2>🗄️ FiberWeave Database Setup</h2>')
        title.setAlignment(Qt.AlignCenter)
        layout.addWidget(title)
        
        # Connection Parameters Group
        conn_group = QGroupBox('Database Connection')
        conn_layout = QFormLayout()
        
        self.host_input = QLineEdit('localhost')
        self.port_input = QLineEdit('5432')
        self.database_input = QLineEdit('postgres')
        self.user_input = QLineEdit('postgres')
        self.password_input = QLineEdit()
        self.password_input.setEchoMode(QLineEdit.Password)
        
        self.new_database_input = QLineEdit('fiberweave_network')
        
        conn_layout.addRow('Host:', self.host_input)
        conn_layout.addRow('Port:', self.port_input)
        conn_layout.addRow('Connect to Database:', self.database_input)
        conn_layout.addRow('Username:', self.user_input)
        conn_layout.addRow('Password:', self.password_input)
        conn_layout.addRow('', QLabel('<i>Use "postgres" database to create new DB</i>'))
        conn_layout.addRow('New Database Name:', self.new_database_input)
        
        conn_group.setLayout(conn_layout)
        layout.addWidget(conn_group)
        
        # Progress
        self.progress = QProgressBar()
        self.progress.setVisible(False)
        layout.addWidget(self.progress)
        
        # Log Output
        log_label = QLabel('<b>Installation Log:</b>')
        layout.addWidget(log_label)
        
        self.log_output = QTextEdit()
        self.log_output.setReadOnly(True)
        self.log_output.setMaximumHeight(150)
        layout.addWidget(self.log_output)
        
        # Buttons
        button_layout = QHBoxLayout()
        
        self.test_btn = QPushButton('🔍 Test Connection')
        self.test_btn.clicked.connect(self.test_connection)
        
        self.install_btn = QPushButton('🚀 Install Database')
        self.install_btn.clicked.connect(self.install_database)
        
        self.close_btn = QPushButton('✖ Close')
        self.close_btn.clicked.connect(self.reject)
        
        button_layout.addWidget(self.test_btn)
        button_layout.addWidget(self.install_btn)
        button_layout.addStretch()
        button_layout.addWidget(self.close_btn)
        
        layout.addLayout(button_layout)
        
        self.setLayout(layout)
    
    def log(self, message, level='info'):
        """Add message to log"""
        color = {
            'info': 'black',
            'success': 'green',
            'warning': 'orange',
            'error': 'red'
        }.get(level, 'black')
        
        self.log_output.append(f'<span style="color: {color};">{message}</span>')
        QgsMessageLog.logMessage(message, 'FiberWeave Database', Qgis.Info)
    
    def test_connection(self):
        """Test database connection"""
        self.log('Testing connection...', 'info')
        
        try:
            conn = psycopg2.connect(
                host=self.host_input.text(),
                port=self.port_input.text(),
                database=self.database_input.text(),
                user=self.user_input.text(),
                password=self.password_input.text()
            )
            
            cur = conn.cursor()
            cur.execute('SELECT version();')
            version = cur.fetchone()[0]
            
            self.log(f'✓ Connection successful!', 'success')
            self.log(f'PostgreSQL version: {version[:50]}...', 'info')
            
            # Check PostGIS
            try:
                cur.execute("SELECT PostGIS_version();")
                postgis_version = cur.fetchone()[0]
                self.log(f'✓ PostGIS version: {postgis_version}', 'success')
            except:
                self.log('⚠ PostGIS not installed', 'warning')
            
            cur.close()
            conn.close()
            
            QMessageBox.information(self, 'Success', '✓ Database connection successful!')
            
        except Exception as e:
            self.log(f'✗ Connection failed: {str(e)}', 'error')
            QMessageBox.critical(self, 'Error', f'Connection failed:\n\n{str(e)}')
    
    def install_database(self):
        """Install FiberWeave database"""
        self.log('=' * 50, 'info')
        self.log('Starting FiberWeave database installation...', 'info')
        self.log('=' * 50, 'info')
        
        try:
            # Connect to postgres database
            self.log('Connecting to PostgreSQL...', 'info')
            conn = psycopg2.connect(
                host=self.host_input.text(),
                port=self.port_input.text(),
                database=self.database_input.text(),
                user=self.user_input.text(),
                password=self.password_input.text()
            )
            conn.autocommit = True
            cur = conn.cursor()
            
            new_db_name = self.new_database_input.text().strip()

            # Validate database name to prevent SQL injection
            # PostgreSQL identifiers: letters, digits, underscores; must start with letter or underscore
            import re
            if not re.match(r'^[a-zA-Z_][a-zA-Z0-9_]*$', new_db_name):
                self.log('✗ Invalid database name. Use only letters, digits, and underscores.', 'error')
                QMessageBox.critical(self, 'Error',
                    'Invalid database name.\n\n'
                    'Database names must start with a letter or underscore\n'
                    'and contain only letters, digits, and underscores.')
                return

            # Check if database exists
            cur.execute(
                "SELECT 1 FROM pg_database WHERE datname = %s",
                (new_db_name,)
            )
            exists = cur.fetchone()

            if exists:
                self.log(f'⚠ Database "{new_db_name}" already exists', 'warning')

                reply = QMessageBox.question(
                    self,
                    'Database Exists',
                    f'Database "{new_db_name}" already exists.\n\n'
                    'Do you want to:\n'
                    '- YES: Connect and create schema only\n'
                    '- NO: Cancel installation',
                    QMessageBox.Yes | QMessageBox.No
                )

                if reply == QMessageBox.No:
                    self.log('Installation cancelled by user', 'warning')
                    return
            else:
                # Create database (using psycopg2.sql for safe identifier quoting)
                from psycopg2 import sql
                self.log(f'Creating database "{new_db_name}"...', 'info')
                cur.execute(sql.SQL('CREATE DATABASE {}').format(sql.Identifier(new_db_name)))
                self.log(f'✓ Database "{new_db_name}" created', 'success')
            
            cur.close()
            conn.close()
            
            # Connect to new database
            self.log(f'Connecting to "{new_db_name}"...', 'info')
            conn = psycopg2.connect(
                host=self.host_input.text(),
                port=self.port_input.text(),
                database=new_db_name,
                user=self.user_input.text(),
                password=self.password_input.text()
            )
            conn.autocommit = True
            cur = conn.cursor()
            
            # Enable PostGIS
            self.log('Enabling PostGIS extension...', 'info')
            cur.execute('CREATE EXTENSION IF NOT EXISTS postgis')
            self.log('✓ PostGIS enabled', 'success')
            
            # Enable PostGIS Topology
            self.log('Enabling PostGIS Topology...', 'info')
            cur.execute('CREATE EXTENSION IF NOT EXISTS postgis_topology')
            self.log('✓ PostGIS Topology enabled', 'success')
            
            # Try pgRouting
            try:
                cur.execute('CREATE EXTENSION IF NOT EXISTS pgrouting')
                self.log('✓ pgRouting enabled', 'success')
            except:
                self.log('⚠ pgRouting not available (optional)', 'warning')
            
            # Create fttx schema
            self.log('Creating fttx schema...', 'info')
            cur.execute('CREATE SCHEMA IF NOT EXISTS fttx')
            self.log('✓ Schema created', 'success')
            
            # Run all numbered SQL files in order
            self.log('Creating ITU-T compliant tables...', 'info')
            import glob
            sql_files = sorted(glob.glob(os.path.join(self.plugin_dir, '[0-9][0-9]_*.sql')))

            if sql_files:
                for sql_file in sql_files:
                    filename = os.path.basename(sql_file)
                    self.log(f'  Running {filename}...', 'info')
                    with open(sql_file, 'r', encoding='utf-8') as f:
                        sql_content = f.read()
                    try:
                        cur.execute(sql_content)
                        self.log(f'  ✓ {filename}', 'success')
                    except Exception as sql_err:
                        self.log(f'  ⚠ {filename}: {str(sql_err)}', 'warning')
            else:
                self.log('⚠ No SQL files found, creating basic structure', 'warning')
            
            cur.close()
            conn.close()
            
            self.log('=' * 50, 'success')
            self.log('✓ FiberWeave database installed successfully!', 'success')
            self.log('=' * 50, 'success')
            
            QMessageBox.information(
                self,
                'Success',
                f'✓ FiberWeave database installed successfully!\n\n'
                f'Database: {new_db_name}\n'
                f'Schema: fttx\n\n'
                f'You can now connect to this database in FiberWeave.'
            )
            
        except Exception as e:
            self.log('=' * 50, 'error')
            self.log(f'✗ Installation failed: {str(e)}', 'error')
            self.log('=' * 50, 'error')
            QMessageBox.critical(
                self,
                'Error',
                f'Installation failed:\n\n{str(e)}'
            )