# itu_fttx_plugin/core/planners.py

import math
import psycopg2
from datetime import datetime
# Corrected relative import for constants
from ..config.constants import SQL_SELECT_OLT_BOM 

# ==========================================
# PLANNING ENGINE (Core Logic for PlanningWizard)
# ==========================================

class PlanningEngine:
    """Core logic for automated network planning."""
    
    def __init__(self, db_connection):
        self.db_connection = db_connection

    def execute_plan(self, params: dict) -> tuple:
        """
        Executes the network planning logic.
        :param params: Parameters from the PlanningWizard UI.
        :return: (Success status, Report/Error message)
        """
        try:
            # 1. Calculate high-level requirements
            customers = params.get('customer_count', 0)
            onus_per_port = params.get('onus_per_port', 64)
            ports_per_olt = params.get('olt_ports_per_chassis', 16)
            
            required_ports = math.ceil(customers / onus_per_port)
            required_olts = math.ceil(required_ports / ports_per_olt)
            
            # 2. Generate required GIS features (simulated)
            # This is where actual QGIS processing/topology generation would occur.
            
            # 3. Compile report (similar to the preview text in the original UI)
            report = f"Plan Generated: {required_olts} OLTs, {required_ports} Ports. Full report generated."
            return True, report
            
        except Exception as e:
            return False, f"Planning execution failed: {str(e)}"

# ==========================================
# BOM ENGINE (Core Logic for BOMGenerator)
# ==========================================

class BOMEngine:
    """Core logic for generating Bill of Materials."""
    
    def __init__(self, db_connection):
        self.db_connection = db_connection

    def generate_bom(self, config_params: dict) -> list:
        """
        Queries the database and compiles the BOM data based on network inventory.
        :param config_params: Configuration parameters (filters, markups, labor).
        :return: List of BOM item dictionaries.
        """
        bom_data = []
        
        try:
            cursor = self.db_connection.cursor()
            
            # --- 1. OLT Equipment (Chassis and Cards) ---
            
            # Get OLT Chassis
            cursor.execute(SQL_SELECT_OLT_BOM)
            for qty, standard, mfr, model in cursor.fetchall():
                bom_data.append({
                    'category': 'OLT Equipment',
                    'description': f'{mfr} {model} OLT Chassis',
                    'part_number': f'OLT-{model}',
                    'manufacturer': mfr or 'Unknown',
                    'quantity': qty,
                    'unit': 'EA',
                    'unit_price': 15000.00, # Estimated Price
                    'lead_time': '12-16 weeks'
                })
            
            # Get PON Port Cards (assuming 16-port cards)
            cursor.execute("""
                SELECT COUNT(*) 
                FROM fttx.itu_olt_pon_ports
                WHERE port_status IN ('active', 'inactive')
            """)
            pon_port_count = cursor.fetchone()[0]
            if pon_port_count > 0:
                bom_data.append({
                    'category': 'OLT Equipment',
                    'description': 'PON Port Cards (16-port)',
                    'part_number': 'PON-CARD-16P',
                    'manufacturer': 'Various',
                    'quantity': math.ceil(pon_port_count / 16),
                    'unit': 'EA',
                    'unit_price': 8000.00,
                    'lead_time': '8-12 weeks'
                })
            
            # --- 2. ONU Equipment (Customer End) ---
            cursor.execute("""
                SELECT 
                    COUNT(*) as qty, manufacturer, model
                FROM fttx.itu_onu_equipment
                WHERE onu_status IN ('planned', 'online', 'offline')
                GROUP BY manufacturer, model
            """)
            for qty, mfr, model in cursor.fetchall():
                bom_data.append({
                    'category': 'Customer Equipment',
                    'description': f'{mfr or "Generic"} ONU {model or "Standard"}',
                    'part_number': f'ONU-{model or "STD"}',
                    'manufacturer': mfr or 'Generic',
                    'quantity': qty,
                    'unit': 'EA',
                    'unit_price': 150.00,
                    'lead_time': '4-6 weeks'
                })
            
            # --- 3. Fiber Cables (Distance) ---
            cursor.execute("""
                SELECT 
                    cable_type,
                    fiber_type,
                    SUM(cable_length_m) / 1000.0 as total_km
                FROM fttx.itu_fiber_cables
                WHERE operational_status IN ('planned', 'operational', 'under_construction')
                GROUP BY cable_type, fiber_type
            """)
            for cable_type, fiber_type, total_km in cursor.fetchall():
                if total_km and total_km > 0:
                    # Assign estimated price based on type and fiber count (simulated)
                    fiber_count, unit_price = {
                        'feeder': (96, 8.50 * 1000),
                        'distribution': (24, 6.00 * 1000),
                        'drop': (2, 2.50 * 1000)
                    }.get(cable_type, (12, 4.00 * 1000))
                    
                    bom_data.append({
                        'category': 'Fiber Cable',
                        'description': f'{fiber_count}F {fiber_type or "SMF"} {cable_type.title()} Cable',
                        'part_number': f'FO-{fiber_count}F-{cable_type.upper()}',
                        'manufacturer': 'Various',
                        'quantity': round(total_km, 2),
                        'unit': 'km',
                        'unit_price': unit_price,
                        'lead_time': '6-8 weeks'
                    })
            
            # --- 4. Splitters and ODN Nodes ---
            cursor.execute("""
                SELECT split_ratio, COUNT(*) as qty
                FROM fttx.itu_odn_nodes
                WHERE node_type LIKE '%Splitter%' OR split_ratio IS NOT NULL
                GROUP BY split_ratio
            """)
            for ratio, qty in cursor.fetchall():
                if ratio:
                    bom_data.append({
                        'category': 'Passive Equipment',
                        'description': f'PLC Splitter {ratio}',
                        'part_number': f'SPL-{ratio}',
                        'manufacturer': 'Various',
                        'quantity': qty,
                        'unit': 'EA',
                        'unit_price': 50.00 if '1:8' in ratio else 200.00,
                        'lead_time': '4-6 weeks'
                    })
            
            # --- 5. Infrastructure (Poles/Manholes) ---
            cursor.execute("SELECT COUNT(*) FROM fttx.itu_odn_nodes WHERE node_type = 'Pole'")
            pole_count = cursor.fetchone()[0]
            if pole_count > 0:
                bom_data.append({
                    'category': 'Infrastructure',
                    'description': 'Utility Pole (40ft Class 4)',
                    'part_number': 'POLE-40-C4',
                    'manufacturer': 'Various',
                    'quantity': pole_count,
                    'unit': 'EA',
                    'unit_price': 450.00,
                    'lead_time': '2-4 weeks'
                })
            
            # --- 6. Labor (Conditional) ---
            if config_params.get('include_labor'):
                total_cable_km = sum([item['quantity'] for item in bom_data if item.get('unit') == 'km'])
                # Use total splitters as a proxy for splicing effort
                splice_count = sum([item['quantity'] for item in bom_data if item['category'] == 'Passive Equipment']) 
                
                if total_cable_km > 0:
                    bom_data.append({
                        'category': 'Labor',
                        'description': 'Fiber Installation Labor',
                        'part_number': 'LAB-INST-KM',
                        'manufacturer': 'N/A',
                        'quantity': round(total_cable_km, 2),
                        'unit': 'km',
                        'unit_price': 8000.00, # Estimated labor per km
                        'lead_time': 'N/A'
                    })
                if splice_count > 0:
                     bom_data.append({
                        'category': 'Labor',
                        'description': 'Splicing Labor (per closure/splitter)',
                        'part_number': 'LAB-SPLICE',
                        'manufacturer': 'N/A',
                        'quantity': splice_count,
                        'unit': 'EA',
                        'unit_price': 350.00,
                        'lead_time': 'N/A'
                    })
            
            # Apply cost calculations (Markup/Discount)
            markup = config_params.get('markup', 1.0)
            discount = config_params.get('discount', 1.0)
            
            for item in bom_data:
                # Ensure fields exist before calculating total (Robustness check)
                item['manufacturer'] = item.get('manufacturer', 'Unknown')
                item['unit'] = item.get('unit', 'EA')
                item['lead_time'] = item.get('lead_time', 'N/A')
                item['unit_price'] = item.get('unit_price', 0.0)
                item['quantity'] = item.get('quantity', 0)
                
                base_total = item['quantity'] * item['unit_price']
                item['total_price'] = base_total * markup * discount

            cursor.close()
            return bom_data
            
        except psycopg2.Error as e:
            # Propagate the error up to the UI handler
            raise Exception(f"Database error during BOM generation: {str(e)}")

# ==========================================
# DOCUMENTATION ENGINE (Core Logic for DocumentationGenerator)
# ==========================================

class DocumentationEngine:
    """Core logic for generating documentation."""
    
    def __init__(self, db_connection):
        self.db_connection = db_connection

    def generate_network_design(self) -> str:
        """Generate network design document from current DB topology."""
        try:
            cursor = self.db_connection.cursor()

            cursor.execute("SELECT COUNT(*), pon_standard FROM fttx.itu_olt_equipment GROUP BY pon_standard")
            olt_data = cursor.fetchall()

            cursor.execute("SELECT COUNT(*) FROM fttx.itu_olt_pon_ports WHERE port_status = 'active'")
            active_ports = cursor.fetchone()[0]

            cursor.execute("SELECT COUNT(*), node_type FROM fttx.itu_odn_nodes GROUP BY node_type")
            node_data = cursor.fetchall()

            cursor.execute("""
                SELECT cable_type, COUNT(*), COALESCE(SUM(cable_length_m), 0) / 1000.0
                FROM fttx.itu_fiber_cables GROUP BY cable_type
            """)
            cable_data = cursor.fetchall()

            cursor.execute("SELECT COUNT(*) FROM fttx.itu_onu_equipment")
            total_onus = cursor.fetchone()[0]

            cursor.close()

            report = f"NETWORK DESIGN DOCUMENT\n{'=' * 60}\nGenerated: {datetime.now().strftime('%Y-%m-%d %H:%M')}\n\n"
            report += "1. OLT EQUIPMENT\n" + "-" * 40 + "\n"
            for count, standard in olt_data:
                report += f"   {standard}: {count} unit(s)\n"
            report += f"   Active PON Ports: {active_ports}\n\n"

            report += "2. ODN NODES\n" + "-" * 40 + "\n"
            for count, node_type in node_data:
                report += f"   {node_type}: {count}\n"

            report += "\n3. FIBER CABLES\n" + "-" * 40 + "\n"
            for cable_type, count, total_km in cable_data:
                report += f"   {cable_type}: {count} cable(s), {total_km:.2f} km\n"

            report += f"\n4. CUSTOMER EQUIPMENT\n" + "-" * 40 + "\n"
            report += f"   Total ONUs: {total_onus}\n"

            return report
        except Exception as e:
            return f"Error generating network design: {str(e)}"

    def generate_asbuilt(self) -> str:
        """Generate as-built documentation from installed equipment records."""
        try:
            cursor = self.db_connection.cursor()

            cursor.execute("""
                SELECT olt_code, manufacturer, model, equipment_status, installation_date
                FROM fttx.itu_olt_equipment
                WHERE equipment_status IN ('active', 'operational')
                ORDER BY olt_code
            """)
            olts = cursor.fetchall()

            cursor.execute("""
                SELECT cable_code, cable_type, fiber_count, cable_length_m, operational_status
                FROM fttx.itu_fiber_cables
                WHERE operational_status IN ('active', 'operational')
                ORDER BY cable_code
            """)
            cables = cursor.fetchall()

            cursor.execute("""
                SELECT node_code, node_type, split_ratio, operational_status
                FROM fttx.itu_odn_nodes
                WHERE operational_status IN ('active', 'operational')
                ORDER BY node_code
            """)
            nodes = cursor.fetchall()

            cursor.close()

            report = f"AS-BUILT DOCUMENTATION\n{'=' * 60}\nGenerated: {datetime.now().strftime('%Y-%m-%d %H:%M')}\n\n"
            report += "1. INSTALLED OLTs\n" + "-" * 40 + "\n"
            for olt in olts:
                report += f"   {olt[0]}: {olt[1]} {olt[2]} [{olt[3]}] Installed: {olt[4] or 'N/A'}\n"

            report += f"\n2. INSTALLED CABLES ({len(cables)} total)\n" + "-" * 40 + "\n"
            for cable in cables:
                report += f"   {cable[0]}: {cable[1]}, {cable[2]}F, {cable[3] or 0:.0f}m [{cable[4]}]\n"

            report += f"\n3. INSTALLED NODES ({len(nodes)} total)\n" + "-" * 40 + "\n"
            for node in nodes:
                ratio = f" ({node[2]})" if node[2] else ""
                report += f"   {node[0]}: {node[1]}{ratio} [{node[3]}]\n"

            return report
        except Exception as e:
            return f"Error generating as-built documentation: {str(e)}"

    def generate_test_report(self) -> str:
        """Generate test report from test_results table."""
        try:
            cursor = self.db_connection.cursor()

            cursor.execute("""
                SELECT test_code, test_type, wavelength_nm, loss_db,
                       return_loss_db, test_result, technician_name
                FROM fttx.test_results
                ORDER BY test_date DESC
            """)
            tests = cursor.fetchall()
            cursor.close()

            report = f"TEST & ACCEPTANCE REPORT\n{'=' * 60}\nGenerated: {datetime.now().strftime('%Y-%m-%d %H:%M')}\n\n"

            if not tests:
                report += "No test results found in database.\n"
                return report

            pass_count = sum(1 for t in tests if t[5] and t[5].lower() == 'pass')
            fail_count = sum(1 for t in tests if t[5] and t[5].lower() == 'fail')

            report += "SUMMARY\n" + "-" * 40 + "\n"
            report += f"   Total Tests: {len(tests)}\n"
            report += f"   Pass: {pass_count}\n"
            report += f"   Fail: {fail_count}\n"
            report += f"   Pass Rate: {pass_count / len(tests) * 100:.1f}%\n\n"

            report += "DETAILED RESULTS\n" + "-" * 40 + "\n"
            for test in tests:
                report += f"   {test[0] or 'N/A'}: {test[1]} @{test[2]}nm "
                report += f"Loss={test[3]:.3f}dB " if test[3] else ""
                report += f"RL={test[4]:.1f}dB " if test[4] else ""
                report += f"[{test[5] or 'N/A'}] Tech: {test[6] or 'N/A'}\n"

            return report
        except Exception as e:
            return f"Error generating test report: {str(e)}"

    def generate_maintenance_guide(self) -> str:
        """Generate maintenance guide from equipment and infrastructure data."""
        try:
            cursor = self.db_connection.cursor()

            cursor.execute("SELECT COUNT(*) FROM fttx.itu_olt_equipment")
            olt_count = cursor.fetchone()[0]

            cursor.execute("SELECT COUNT(*) FROM fttx.itu_odn_nodes WHERE node_type = 'splitter'")
            splitter_count = cursor.fetchone()[0]

            cursor.execute("SELECT COUNT(*) FROM fttx.itu_splice_points")
            splice_count = cursor.fetchone()[0]

            cursor.execute("SELECT COALESCE(SUM(cable_length_m), 0) / 1000.0 FROM fttx.itu_fiber_cables")
            total_cable_km = cursor.fetchone()[0]

            cursor.close()

            report = f"MAINTENANCE GUIDE\n{'=' * 60}\nGenerated: {datetime.now().strftime('%Y-%m-%d %H:%M')}\n\n"

            report += "1. EQUIPMENT INVENTORY\n" + "-" * 40 + "\n"
            report += f"   OLTs: {olt_count}\n"
            report += f"   Splitters: {splitter_count}\n"
            report += f"   Splice Points: {splice_count}\n"
            report += f"   Total Fiber Cable: {total_cable_km:.2f} km\n\n"

            report += "2. SCHEDULED MAINTENANCE\n" + "-" * 40 + "\n"
            report += "   OLT Equipment: Quarterly inspection\n"
            report += "   Splice Closures: Annual inspection\n"
            report += "   Fiber Routes: Semi-annual patrol\n"
            report += "   Power Budget Tests: Annual verification\n\n"

            report += "3. RECOMMENDED PROCEDURES\n" + "-" * 40 + "\n"
            report += "   - OTDR testing on all new installations\n"
            report += "   - Power meter verification at each splice point\n"
            report += "   - Visual inspection of aerial cable attachments\n"
            report += "   - Closure re-sealing check after extreme weather\n"
            report += "   - ONU signal level monitoring (monthly)\n"

            return report
        except Exception as e:
            return f"Error generating maintenance guide: {str(e)}"
