# itu_fttx_plugin/core/calculators.py

import math
from datetime import datetime
from ..config.constants import DEFAULT_SPLITTER_LOSS

class PowerBudgetEngine:
    """Core logic for Power Budget Calculation (Decoupled from UI)"""

    def calculate(self, params: dict) -> dict:
        """
        Performs the ITU-T G.984.2 power budget calculation.
        :param params: Dictionary of input parameters (e.g., olt_tx_power, fiber_length).
        :return: Dictionary of calculation results.
        """
        
        splitter_loss_db = DEFAULT_SPLITTER_LOSS.get(params['splitter_ratio'], 17.0)
        
        fiber_loss = params['fiber_length'] * params['fiber_attenuation']
        connector_loss_total = params['connector_count'] * params['connector_loss']
        splice_loss_total = params['splice_count'] * params['splice_loss']
        
        total_loss = fiber_loss + splitter_loss_db + connector_loss_total + splice_loss_total
        
        # Downstream
        onu_rx_power = params['olt_tx_power'] - total_loss
        downstream_margin = onu_rx_power - params['onu_rx_sensitivity']
        
        # Upstream
        olt_rx_power = params['onu_tx_power'] - total_loss
        upstream_margin = olt_rx_power - params['olt_rx_sensitivity']
        
        min_margin = 3.0
        
        return {
            'total_loss': total_loss,
            'fiber_loss': fiber_loss,
            'splitter_loss': splitter_loss_db,
            'connector_loss_total': connector_loss_total,
            'splice_loss_total': splice_loss_total,
            'onu_rx_power': onu_rx_power,
            'downstream_margin': downstream_margin,
            'olt_rx_power': olt_rx_power,
            'upstream_margin': upstream_margin,
            'min_margin': min_margin,
            'is_compliant': downstream_margin >= min_margin and upstream_margin >= min_margin
        }

class CableTensionEngine:
    """Core logic for Cable Pull Tension Calculation"""
    
    def calculate(self, params: dict) -> dict:
        """Performs the cable tension calculation."""
        
        weight_per_m = params['cable_weight'] / 1000  # kg/m
        weight_force = weight_per_m * 9.81  # N/m (Force/m)
        
        # Installation parameters
        length = params['pull_length']
        friction = params['coefficient_friction']
        bends = params['bend_count']
        bend_angle_rad = math.radians(params['bend_angle'])
        elevation = params['elevation_change']
        max_rated = params['max_tension_rated']
        
        # 1. Straight pull tension (approximate)
        straight_tension = weight_force * length * friction
        
        # 2. Bend tension (Capstan equation approximation)
        # Assuming tension at start is T_start. T_end = T_start * e^(µ*θ)
        bend_tension = 0
        if bends > 0:
            # Simplification: Assume all tension is applied to the final bend
            # In a full model, this is iterative. Using a simplified summation:
            bend_factor = math.exp(friction * bend_angle_rad) - 1
            bend_tension = straight_tension * bend_factor * bends
        
        # 3. Elevation tension
        elevation_tension = weight_force * elevation
        
        # Total tension
        total_tension = straight_tension + bend_tension + abs(elevation_tension)
        
        safety_factor = 4.0  # Industry standard
        max_allowed = max_rated / safety_factor
        utilization = (total_tension / max_allowed) * 100
        
        return {
            'straight_tension': straight_tension,
            'bend_tension': bend_tension,
            'elevation_tension': abs(elevation_tension),
            'total_tension': total_tension,
            'max_rated': max_rated,
            'max_allowed': max_allowed,
            'utilization': utilization
        }

class AerialSlackEngine:
    """Core logic for Aerial Cable Sag/Slack Calculation"""
    
    def calculate(self, params: dict) -> dict:
        """Performs the aerial cable sag/slack calculation."""
        
        span = params['span_length']
        # Wind loading is in Pa (N/m²). Convert to equivalent linear load (kg/km)
        # using a typical fiber cable diameter of 12mm (0.012m)
        cable_diameter_m = 0.012
        wind_linear_load_kg_km = (params['wind_loading'] * cable_diameter_m / 9.81) * 1000
        weight = params['cable_weight_aerial'] + params['ice_loading'] + wind_linear_load_kg_km
        weight_force = (weight / 1000) * 9.81 # N/m
        
        # Catenary sag calculation (simplified, assuming constant horizontal tension)
        breaking_strength = 600  # N (typical for fiber cable)
        horizontal_tension = breaking_strength * 0.10  # Assume 10% of breaking strength
        
        # Sag at midspan: sag = (w * L^2) / (8 * H)
        sag = (weight_force * span**2) / (8 * horizontal_tension)
        
        # Actual cable length (catenary length approximation)
        sag_ratio = sag / span
        cable_length = span * (1 + (8/3) * sag_ratio**2)
        
        # Slack calculation
        slack = cable_length - span
        slack_percent = (slack / span) * 100
        
        # Temperature effects
        temp_coeff = 1.2e-5 # Coeff of thermal expansion for fiber cable
        temp_range = params['max_temp'] - params['min_temp']
        temp_expansion = span * temp_coeff * temp_range
        
        # Recommended slack
        recommended_slack = slack + temp_expansion + (span * 0.02) # Add 2% safety margin
        recommended_slack_percent = (recommended_slack / span) * 100
        
        return {
            'sag': sag,
            'sag_ratio': sag_ratio,
            'horizontal_tension': horizontal_tension,
            'cable_length': cable_length,
            'geometric_slack': slack,
            'slack_percent': slack_percent,
            'temp_expansion': temp_expansion,
            'temp_range': temp_range,
            'recommended_slack': recommended_slack,
            'recommended_slack_percent': recommended_slack_percent,
            'installed_length': span + recommended_slack
        }