# -*- coding: utf-8 -*-
"""
QGIS Form Configuration for FTTX Solution
Automatically configure custom forms, widgets, and snapping for all layers
"""

from qgis.PyQt.QtWidgets import QMessageBox
from qgis.core import (
    QgsProject, QgsEditorWidgetSetup, QgsVectorLayer,
    QgsDefaultValue, QgsFieldConstraint, QgsEditFormConfig,
    QgsSnappingConfig, QgsTolerance, QgsMapLayerType
)
from qgis.gui import QgsExpressionBuilderDialog
import os


class FTTXFormConfigurator:
    """Configure custom forms for all FTTX layers"""
    
    def __init__(self, iface, plugin):
        self.iface = iface
        self.plugin = plugin
        self.project = QgsProject.instance()
    
    def configure_all_forms(self):
        """Configure forms for all FTTX layers"""
        try:
            # Configure each layer
            self.configure_olt_equipment_form()
            self.configure_onu_equipment_form()
            self.configure_pon_ports_form()
            self.configure_fiber_cables_form()
            self.configure_fiber_strands_form()
            self.configure_odn_nodes_form()
            self.configure_splice_points_form()
            self.configure_cable_routes_form()
            
            # Inside Plant forms
            self.configure_buildings_form()
            self.configure_rooms_form()
            self.configure_racks_form()
            self.configure_equipment_form()
            self.configure_ports_form()
            self.configure_patch_panels_form()
            
            # Configure snapping
            self.configure_snapping()
            
            QMessageBox.information(
                self.iface.mainWindow(),
                'Success',
                'All FTTX forms configured successfully!\n\n'
                'Features:\n'
                '• Auto-generate codes\n'
                '• Date/time stamps\n'
                '• Cascading dropdowns\n'
                '• Field validation\n'
                '• Snapping enabled'
            )
            return True
            
        except Exception as e:
            QMessageBox.critical(
                self.iface.mainWindow(),
                'Error',
                f'Failed to configure forms:\n{str(e)}'
            )
            return False
    
    # ==========================================
    # OLT EQUIPMENT FORM
    # ==========================================
    
    def configure_olt_equipment_form(self):
        """Configure OLT Equipment custom form"""
        layer = self.get_layer_by_name('OLT Equipment')
        if not layer:
            return
        
        # Field configurations
        field_config = {
            'olt_id': {
                'widget': 'Hidden',
                'default': None
            },
            'olt_code': {
                'widget': 'TextEdit',
                'default': "concat('OLT-', lpad(to_string(maximum(olt_id) + 1), 4, '0'))",
                'alias': 'OLT Code',
                'mandatory': True
            },
            'olt_name': {
                'widget': 'TextEdit',
                'alias': 'OLT Name',
                'placeholder': 'e.g., Main Office OLT'
            },
            'pon_standard': {
                'widget': 'ValueMap',
                'values': {
                    'GPON (G.984)': 'GPON',
                    'XG-PON (G.987)': 'XG-PON',
                    'NG-PON2 (G.989)': 'NG-PON2',
                    '10G-EPON': '10G-EPON'
                },
                'default': 'GPON',
                'alias': 'PON Standard',
                'mandatory': True
            },
            'manufacturer': {
                'widget': 'ValueMap',
                'values': {
                    'Huawei': 'Huawei',
                    'Nokia': 'Nokia',
                    'ZTE': 'ZTE',
                    'Calix': 'Calix',
                    'Adtran': 'Adtran',
                    'Other': 'Other'
                },
                'alias': 'Manufacturer'
            },
            'model': {
                'widget': 'TextEdit',
                'alias': 'Model Number'
            },
            'serial_number': {
                'widget': 'TextEdit',
                'alias': 'Serial Number'
            },
            'max_pon_ports': {
                'widget': 'Range',
                'min': 1,
                'max': 32,
                'step': 1,
                'default': 16,
                'alias': 'Max PON Ports'
            },
            'equipment_status': {
                'widget': 'ValueMap',
                'values': {
                    'Planned': 'planned',
                    'Operational': 'operational',
                    'Maintenance': 'maintenance',
                    'Decommissioned': 'decommissioned'
                },
                'default': 'planned',
                'alias': 'Status',
                'mandatory': True
            },
            'installation_date': {
                'widget': 'DateTime',
                'calendar_popup': True,
                'display_format': 'yyyy-MM-dd',
                'alias': 'Installation Date'
            },
            'ip_address': {
                'widget': 'TextEdit',
                'alias': 'Management IP',
                'placeholder': '192.168.1.1'
            },
            'created_date': {
                'widget': 'DateTime',
                'default': 'now()',
                'read_only': True,
                'alias': 'Created Date'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['OLT Information', ['olt_code', 'olt_name', 'pon_standard']],
            ['Equipment Details', ['manufacturer', 'model', 'serial_number', 'max_pon_ports']],
            ['Network Configuration', ['ip_address', 'equipment_status', 'installation_date']],
            ['System', ['created_date']]
        ])
    
    # ==========================================
    # ONU EQUIPMENT FORM
    # ==========================================
    
    def configure_onu_equipment_form(self):
        """Configure ONU Equipment custom form"""
        layer = self.get_layer_by_name('ONUs')
        if not layer:
            return
        
        field_config = {
            'onu_id': {
                'widget': 'Hidden'
            },
            'onu_code': {
                'widget': 'TextEdit',
                'default': "concat('ONU-', lpad(to_string(maximum(onu_id) + 1), 5, '0'))",
                'alias': 'ONU Code',
                'mandatory': True
            },
            'onu_serial': {
                'widget': 'TextEdit',
                'alias': 'Serial Number',
                'mandatory': True
            },
            'pon_port_id': {
                'widget': 'RelationReference',
                'relation': 'onu_to_pon_port',
                'alias': 'Connected PON Port',
                'allow_null': True
            },
            'customer_name': {
                'widget': 'TextEdit',
                'alias': 'Customer Name'
            },
            'customer_account': {
                'widget': 'TextEdit',
                'alias': 'Account Number'
            },
            'onu_type': {
                'widget': 'ValueMap',
                'values': {
                    'Residential': 'residential',
                    'Business': 'business',
                    'MDU': 'mdu'
                },
                'default': 'residential',
                'alias': 'ONU Type'
            },
            'onu_status': {
                'widget': 'ValueMap',
                'values': {
                    'Online': 'online',
                    'Offline': 'offline',
                    'Planned': 'planned'
                },
                'default': 'planned',
                'alias': 'Status'
            },
            'rx_power_dbm': {
                'widget': 'Range',
                'min': -40,
                'max': 0,
                'step': 0.1,
                'suffix': ' dBm',
                'alias': 'RX Power'
            },
            'activation_date': {
                'widget': 'DateTime',
                'calendar_popup': True,
                'alias': 'Activation Date'
            },
            'created_date': {
                'widget': 'DateTime',
                'default': 'now()',
                'read_only': True,
                'alias': 'Created'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['ONU Information', ['onu_code', 'onu_serial', 'onu_type']],
            ['Customer Details', ['customer_name', 'customer_account']],
            ['Network', ['pon_port_id', 'onu_status', 'rx_power_dbm']],
            ['Dates', ['activation_date', 'created_date']]
        ])
    
    # ==========================================
    # PON PORTS FORM
    # ==========================================
    
    def configure_pon_ports_form(self):
        """Configure PON Ports custom form"""
        layer = self.get_layer_by_name('PON Ports')
        if not layer:
            return
        
        field_config = {
            'pon_port_id': {
                'widget': 'Hidden'
            },
            'olt_id': {
                'widget': 'RelationReference',
                'relation': 'port_to_olt',
                'alias': 'OLT',
                'mandatory': True
            },
            'port_number': {
                'widget': 'Range',
                'min': 1,
                'max': 128,
                'alias': 'Port Number',
                'mandatory': True
            },
            'port_label': {
                'widget': 'TextEdit',
                'alias': 'Port Label',
                'placeholder': 'e.g., PON-1/1/1'
            },
            'optical_budget_class': {
                'widget': 'ValueMap',
                'values': {
                    'Class A (5-20 dB)': 'A',
                    'Class B (10-25 dB)': 'B',
                    'Class B+ (13-28 dB)': 'B+',
                    'Class C (15-30 dB)': 'C',
                    'Class C+ (17-32 dB)': 'C+',
                    'Class C++ (19-34 dB)': 'C++'
                },
                'default': 'B+',
                'alias': 'Optical Budget Class'
            },
            'max_logical_reach_km': {
                'widget': 'Range',
                'min': 1,
                'max': 60,
                'step': 1,
                'suffix': ' km',
                'default': 20,
                'alias': 'Max Reach'
            },
            'max_onus': {
                'widget': 'Range',
                'min': 1,
                'max': 128,
                'default': 64,
                'alias': 'Max ONUs'
            },
            'current_onus': {
                'widget': 'Range',
                'read_only': True,
                'alias': 'Current ONUs'
            },
            'port_status': {
                'widget': 'ValueMap',
                'values': {
                    'Active': 'active',
                    'Inactive': 'inactive',
                    'Full': 'full',
                    'Maintenance': 'maintenance'
                },
                'default': 'active',
                'alias': 'Status'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Port Information', ['olt_id', 'port_number', 'port_label']],
            ['Capacity', ['optical_budget_class', 'max_logical_reach_km', 'max_onus', 'current_onus']],
            ['Status', ['port_status']]
        ])
    
    # ==========================================
    # FIBER CABLES FORM
    # ==========================================
    
    def configure_fiber_cables_form(self):
        """Configure Fiber Cables custom form"""
        layer = self.get_layer_by_name('Fiber Cables')
        if not layer:
            return
        
        field_config = {
            'cable_id': {
                'widget': 'Hidden'
            },
            'cable_code': {
                'widget': 'TextEdit',
                'default': "concat('FO-', lpad(to_string(maximum(cable_id) + 1), 5, '0'))",
                'alias': 'Cable Code',
                'mandatory': True
            },
            'cable_type': {
                'widget': 'ValueMap',
                'values': {
                    'Feeder': 'feeder',
                    'Distribution': 'distribution',
                    'Drop': 'drop'
                },
                'default': 'distribution',
                'alias': 'Cable Type',
                'mandatory': True
            },
            'fiber_count': {
                'widget': 'ValueMap',
                'values': {
                    '2 Fiber': 2,
                    '6 Fiber': 6,
                    '12 Fiber': 12,
                    '24 Fiber': 24,
                    '48 Fiber': 48,
                    '72 Fiber': 72,
                    '96 Fiber': 96,
                    '144 Fiber': 144,
                    '288 Fiber': 288
                },
                'default': 12,
                'alias': 'Fiber Count',
                'mandatory': True
            },
            'fiber_type': {
                'widget': 'ValueMap',
                'values': {
                    'G.652.D (Standard SMF)': 'G.652.D',
                    'G.657.A2 (Bend Optimized)': 'G.657.A2',
                    'G.657.B3 (High Bend)': 'G.657.B3'
                },
                'default': 'G.652.D',
                'alias': 'Fiber Type'
            },
            'cable_length_m': {
                'widget': 'Range',
                'min': 0,
                'max': 100000,
                'step': 0.1,
                'suffix': ' m',
                'alias': 'Cable Length',
                'expression': '$length'
            },
            'route_type': {
                'widget': 'ValueMap',
                'values': {
                    'Aerial': 'aerial',
                    'Underground Duct': 'underground_duct',
                    'Underground Direct Burial': 'underground_direct',
                    'Indoor': 'indoor'
                },
                'default': 'underground_duct',
                'alias': 'Route Type'
            },
            'operational_status': {
                'widget': 'ValueMap',
                'values': {
                    'Planned': 'planned',
                    'Under Construction': 'under_construction',
                    'Operational': 'operational',
                    'Out of Service': 'out_of_service'
                },
                'default': 'planned',
                'alias': 'Status'
            },
            'installation_date': {
                'widget': 'DateTime',
                'calendar_popup': True,
                'alias': 'Installation Date'
            },
            'notes': {
                'widget': 'TextEdit',
                'multiline': True,
                'alias': 'Notes'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Cable Information', ['cable_code', 'cable_type', 'fiber_count', 'fiber_type']],
            ['Route Details', ['route_type', 'cable_length_m']],
            ['Status', ['operational_status', 'installation_date']],
            ['Additional', ['notes']]
        ])
    
    # ==========================================
    # ODN NODES FORM
    # ==========================================
    
    def configure_odn_nodes_form(self):
        """Configure ODN Nodes custom form"""
        layer = self.get_layer_by_name('Network Nodes')
        if not layer:
            return
        
        field_config = {
            'node_id': {
                'widget': 'Hidden'
            },
            'node_code': {
                'widget': 'TextEdit',
                'default': "concat('NODE-', lpad(to_string(maximum(node_id) + 1), 5, '0'))",
                'alias': 'Node Code',
                'mandatory': True
            },
            'node_type': {
                'widget': 'ValueMap',
                'values': {
                    'Splitter Cabinet': 'Splitter_Cabinet',
                    'FDT (Fiber Distribution Terminal)': 'FDT',
                    'FAT (Fiber Access Terminal)': 'FAT',
                    'Manhole': 'Manhole',
                    'Handhole': 'Handhole',
                    'Pole': 'Pole',
                    'Splice Closure': 'Splice_Closure',
                    'Building Entry': 'Building_Entry'
                },
                'default': 'FDT',
                'alias': 'Node Type',
                'mandatory': True
            },
            'split_ratio': {
                'widget': 'ValueMap',
                'values': {
                    'No Splitter': None,
                    '1:2': '1:2',
                    '1:4': '1:4',
                    '1:8': '1:8',
                    '1:16': '1:16',
                    '1:32': '1:32',
                    '1:64': '1:64'
                },
                'alias': 'Split Ratio',
                'allow_null': True
            },
            'manufacturer': {
                'widget': 'TextEdit',
                'alias': 'Manufacturer'
            },
            'model': {
                'widget': 'TextEdit',
                'alias': 'Model'
            },
            'port_count_in': {
                'widget': 'Range',
                'min': 0,
                'max': 288,
                'alias': 'Input Ports'
            },
            'port_count_out': {
                'widget': 'Range',
                'min': 0,
                'max': 288,
                'alias': 'Output Ports'
            },
            'operational_status': {
                'widget': 'ValueMap',
                'values': {
                    'Planned': 'planned',
                    'Operational': 'operational',
                    'Maintenance': 'maintenance',
                    'Decommissioned': 'decommissioned'
                },
                'default': 'planned',
                'alias': 'Status'
            },
            'installation_date': {
                'widget': 'DateTime',
                'calendar_popup': True,
                'alias': 'Installation Date'
            },
            'address': {
                'widget': 'TextEdit',
                'multiline': True,
                'alias': 'Location Address'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Node Information', ['node_code', 'node_type', 'address']],
            ['Equipment', ['manufacturer', 'model', 'split_ratio']],
            ['Capacity', ['port_count_in', 'port_count_out']],
            ['Status', ['operational_status', 'installation_date']]
        ])
    
    # ==========================================
    # CABLE ROUTES FORM
    # ==========================================
    
    def configure_cable_routes_form(self):
        """Configure Cable Routes custom form"""
        layer = self.get_layer_by_name('Routes')
        if not layer:
            return
        
        field_config = {
            'route_id': {
                'widget': 'Hidden'
            },
            'route_code': {
                'widget': 'TextEdit',
                'default': "concat('RT-', lpad(to_string(maximum(route_id) + 1), 5, '0'))",
                'alias': 'Route Code',
                'mandatory': True
            },
            'route_name': {
                'widget': 'TextEdit',
                'alias': 'Route Name'
            },
            'route_type': {
                'widget': 'ValueMap',
                'values': {
                    'Aerial': 'aerial',
                    'Underground': 'underground',
                    'Indoor': 'indoor',
                    'Mixed': 'mixed'
                },
                'default': 'underground',
                'alias': 'Route Type'
            },
            'route_length_m': {
                'widget': 'Range',
                'min': 0,
                'max': 100000,
                'suffix': ' m',
                'alias': 'Length',
                'expression': '$length'
            },
            'duct_type': {
                'widget': 'ValueMap',
                'values': {
                    'HDPE': 'HDPE',
                    'PVC': 'PVC',
                    'Steel': 'Steel',
                    'N/A': None
                },
                'alias': 'Duct Type',
                'allow_null': True
            },
            'duct_size_mm': {
                'widget': 'ValueMap',
                'values': {
                    '32mm': 32,
                    '40mm': 40,
                    '50mm': 50,
                    '63mm': 63,
                    '75mm': 75,
                    '110mm': 110
                },
                'alias': 'Duct Size'
            },
            'route_status': {
                'widget': 'ValueMap',
                'values': {
                    'Planned': 'planned',
                    'Under Construction': 'under_construction',
                    'Operational': 'operational'
                },
                'default': 'planned',
                'alias': 'Status'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Route Information', ['route_code', 'route_name', 'route_type']],
            ['Infrastructure', ['duct_type', 'duct_size_mm', 'route_length_m']],
            ['Status', ['route_status']]
        ])
    
    # ==========================================
    # SPLICE POINTS FORM
    # ==========================================
    
    def configure_splice_points_form(self):
        """Configure Splice Points custom form"""
        layer = self.get_layer_by_name('Splice Points')
        if not layer:
            return
        
        field_config = {
            'splice_id': {
                'widget': 'Hidden'
            },
            'splice_code': {
                'widget': 'TextEdit',
                'default': "concat('SP-', lpad(to_string(maximum(splice_id) + 1), 5, '0'))",
                'alias': 'Splice Code',
                'mandatory': True
            },
            'splice_type': {
                'widget': 'ValueMap',
                'values': {
                    'Fusion Splice': 'fusion',
                    'Mechanical Splice': 'mechanical',
                    'Connector': 'connector'
                },
                'default': 'fusion',
                'alias': 'Splice Type'
            },
            'closure_type': {
                'widget': 'ValueMap',
                'values': {
                    'Dome Closure': 'dome',
                    'Inline Closure': 'inline',
                    'Tray': 'tray',
                    'Heat Shrink': 'heat_shrink'
                },
                'alias': 'Closure Type'
            },
            'fiber_count': {
                'widget': 'Range',
                'min': 1,
                'max': 288,
                'alias': 'Fibers Spliced'
            },
            'avg_loss_db': {
                'widget': 'Range',
                'min': 0,
                'max': 1,
                'step': 0.01,
                'suffix': ' dB',
                'alias': 'Avg Loss'
            },
            'splice_date': {
                'widget': 'DateTime',
                'calendar_popup': True,
                'alias': 'Splice Date'
            },
            'technician': {
                'widget': 'TextEdit',
                'alias': 'Technician'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Splice Information', ['splice_code', 'splice_type', 'closure_type']],
            ['Details', ['fiber_count', 'avg_loss_db']],
            ['Work Info', ['splice_date', 'technician']]
        ])
    
    # ==========================================
    # FIBER STRANDS FORM
    # ==========================================
    
    def configure_fiber_strands_form(self):
        """Configure Fiber Strands custom form"""
        layer = self.get_layer_by_name('Fiber Strands')
        if not layer:
            return
        
        field_config = {
            'strand_id': {
                'widget': 'Hidden'
            },
            'cable_id': {
                'widget': 'RelationReference',
                'relation': 'strand_to_cable',
                'alias': 'Parent Cable',
                'mandatory': True
            },
            'strand_number': {
                'widget': 'Range',
                'min': 1,
                'max': 288,
                'alias': 'Strand Number',
                'mandatory': True
            },
            'strand_color': {
                'widget': 'ValueMap',
                'values': {
                    'Blue': 'Blue',
                    'Orange': 'Orange',
                    'Green': 'Green',
                    'Brown': 'Brown',
                    'Slate': 'Slate',
                    'White': 'White',
                    'Red': 'Red',
                    'Black': 'Black',
                    'Yellow': 'Yellow',
                    'Violet': 'Violet',
                    'Rose': 'Rose',
                    'Aqua': 'Aqua'
                },
                'alias': 'Fiber Color'
            },
            'strand_status': {
                'widget': 'ValueMap',
                'values': {
                    'Available': 'available',
                    'In Use': 'in_use',
                    'Reserved': 'reserved',
                    'Faulty': 'faulty'
                },
                'default': 'available',
                'alias': 'Status'
            },
            'assignment': {
                'widget': 'TextEdit',
                'alias': 'Assignment',
                'placeholder': 'e.g., ONU-12345'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Fiber Information', ['cable_id', 'strand_number', 'strand_color']],
            ['Status', ['strand_status', 'assignment']]
        ])
    
    # ==========================================
    # INSIDE PLANT FORMS
    # ==========================================
    
    def configure_buildings_form(self):
        """Configure Buildings custom form"""
        layer = self.get_layer_by_name('Buildings')
        if not layer:
            return
        
        field_config = {
            'building_id': {
                'widget': 'Hidden'
            },
            'building_code': {
                'widget': 'TextEdit',
                'default': "concat('BLD-', lpad(to_string(maximum(building_id) + 1), 4, '0'))",
                'alias': 'Building Code',
                'mandatory': True
            },
            'building_name': {
                'widget': 'TextEdit',
                'alias': 'Building Name',
                'mandatory': True
            },
            'street_address': {
                'widget': 'TextEdit',
                'alias': 'Street Address'
            },
            'city': {
                'widget': 'TextEdit',
                'alias': 'City'
            },
            'postal_code': {
                'widget': 'TextEdit',
                'alias': 'Postal Code'
            },
            'building_type': {
                'widget': 'ValueMap',
                'values': {
                    'Office': 'Office',
                    'Data Center': 'Data Center',
                    'Residential': 'Residential',
                    'Mixed Use': 'Mixed Use'
                },
                'alias': 'Building Type'
            },
            'floors_count': {
                'widget': 'Range',
                'min': 1,
                'max': 200,
                'alias': 'Number of Floors'
            },
            'owner': {
                'widget': 'TextEdit',
                'alias': 'Owner'
            },
            'contact_person': {
                'widget': 'TextEdit',
                'alias': 'Contact Person'
            },
            'contact_phone': {
                'widget': 'TextEdit',
                'alias': 'Contact Phone'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Building Information', ['building_code', 'building_name', 'building_type', 'floors_count']],
            ['Address', ['street_address', 'city', 'postal_code']],
            ['Contact', ['owner', 'contact_person', 'contact_phone']]
        ])
    
    def configure_rooms_form(self):
        """Configure Rooms form"""
        layer = self.get_layer_by_name('Rooms')
        if not layer:
            return
        
        field_config = {
            'room_id': {'widget': 'Hidden'},
            'room_code': {
                'widget': 'TextEdit',
                'default': "concat('ROOM-', lpad(to_string(maximum(room_id) + 1), 4, '0'))",
                'alias': 'Room Code',
                'mandatory': True
            },
            'room_name': {'widget': 'TextEdit', 'alias': 'Room Name'},
            'building_id': {
                'widget': 'RelationReference',
                'relation': 'room_to_building',
                'alias': 'Building',
                'mandatory': True
            },
            'floor_number': {'widget': 'Range', 'min': -5, 'max': 200, 'alias': 'Floor'},
            'room_type': {
                'widget': 'ValueMap',
                'values': {
                    'MDF': 'MDF - Main Distribution Frame',
                    'IDF': 'IDF - Intermediate Distribution Frame',
                    'Data Center': 'Data Center',
                    'Server Room': 'Server Room',
                    'Telecom Closet': 'Telecom Closet'
                },
                'alias': 'Room Type'
            },
            'room_size_sqm': {'widget': 'Range', 'min': 1, 'max': 10000, 'suffix': ' m²', 'alias': 'Size'},
            'has_hvac': {'widget': 'CheckBox', 'alias': 'Has HVAC', 'checked': True, 'unchecked': False},
            'has_ups': {'widget': 'CheckBox', 'alias': 'Has UPS'},
            'power_capacity_kw': {'widget': 'Range', 'min': 0, 'max': 1000, 'suffix': ' kW', 'alias': 'Power Capacity'},
            'room_status': {
                'widget': 'ValueMap',
                'values': {'Active': 'active', 'Inactive': 'inactive', 'Maintenance': 'maintenance'},
                'default': 'active',
                'alias': 'Status'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Room Information', ['room_code', 'room_name', 'building_id', 'floor_number', 'room_type']],
            ['Specifications', ['room_size_sqm', 'power_capacity_kw']],
            ['Facilities', ['has_hvac', 'has_ups']],
            ['Status', ['room_status']]
        ])
    
    def configure_racks_form(self):
        """Configure Racks form"""
        layer = self.get_layer_by_name('Racks')
        if not layer:
            return
        
        field_config = {
            'rack_id': {'widget': 'Hidden'},
            'rack_code': {
                'widget': 'TextEdit',
                'default': "concat('RACK-', lpad(to_string(maximum(rack_id) + 1), 4, '0'))",
                'alias': 'Rack Code',
                'mandatory': True
            },
            'rack_name': {'widget': 'TextEdit', 'alias': 'Rack Name'},
            'room_id': {
                'widget': 'RelationReference',
                'relation': 'rack_to_room',
                'alias': 'Room',
                'mandatory': True
            },
            'rack_type': {
                'widget': 'ValueMap',
                'values': {
                    'Standard 19" Rack': 'Standard 19" Rack',
                    '4-Post Rack': '4-Post Rack',
                    'Wall-Mount Rack': 'Wall-Mount Rack',
                    'Open Frame': 'Open Frame Rack'
                },
                'alias': 'Rack Type'
            },
            'height_ru': {
                'widget': 'ValueMap',
                'values': {
                    '42 RU': 42,
                    '45 RU': 45,
                    '47 RU': 47,
                    '48 RU': 48,
                    '52 RU': 52
                },
                'default': 42,
                'alias': 'Height',
                'mandatory': True
            },
            'used_ru': {'widget': 'Range', 'read_only': True, 'alias': 'Used RU'},
            'available_ru': {'widget': 'Range', 'read_only': True, 'alias': 'Available RU'},
            'has_pdu': {'widget': 'CheckBox', 'alias': 'Has PDU'},
            'rack_status': {
                'widget': 'ValueMap',
                'values': {'Active': 'active', 'Inactive': 'inactive', 'Maintenance': 'maintenance'},
                'default': 'active',
                'alias': 'Status'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Rack Information', ['rack_code', 'rack_name', 'room_id', 'rack_type']],
            ['Capacity', ['height_ru', 'used_ru', 'available_ru']],
            ['Features', ['has_pdu', 'rack_status']]
        ])
    
    def configure_equipment_form(self):
        """Configure Equipment form"""
        layer = self.get_layer_by_name('Equipment')
        if not layer:
            return
        
        field_config = {
            'equipment_id': {'widget': 'Hidden'},
            'equipment_code': {
                'widget': 'TextEdit',
                'default': "concat('EQ-', lpad(to_string(maximum(equipment_id) + 1), 5, '0'))",
                'alias': 'Equipment Code',
                'mandatory': True
            },
            'equipment_name': {'widget': 'TextEdit', 'alias': 'Name'},
            'rack_id': {
                'widget': 'RelationReference',
                'relation': 'equipment_to_rack',
                'alias': 'Rack',
                'mandatory': True
            },
            'equipment_type': {
                'widget': 'ValueMap',
                'values': {
                    'OLT': 'OLT',
                    'Switch': 'Switch',
                    'Router': 'Router',
                    'ODF': 'ODF',
                    'Patch Panel': 'Patch Panel',
                    'PDU': 'PDU',
                    'UPS': 'UPS'
                },
                'alias': 'Equipment Type'
            },
            'start_ru': {
                'widget': 'Range',
                'min': 1,
                'max': 52,
                'alias': 'Start RU',
                'mandatory': True
            },
            'height_ru': {
                'widget': 'Range',
                'min': 1,
                'max': 12,
                'alias': 'Height (RU)',
                'mandatory': True
            },
            'manufacturer': {'widget': 'TextEdit', 'alias': 'Manufacturer'},
            'model': {'widget': 'TextEdit', 'alias': 'Model'},
            'serial_number': {'widget': 'TextEdit', 'alias': 'Serial Number'},
            'power_consumption_w': {'widget': 'Range', 'min': 0, 'max': 5000, 'suffix': ' W', 'alias': 'Power'},
            'port_count': {'widget': 'Range', 'min': 0, 'max': 256, 'alias': 'Ports'},
            'ip_address': {'widget': 'TextEdit', 'alias': 'IP Address'},
            'equipment_status': {
                'widget': 'ValueMap',
                'values': {
                    'Operational': 'operational',
                    'Standby': 'standby',
                    'Maintenance': 'maintenance',
                    'Faulty': 'faulty'
                },
                'default': 'operational',
                'alias': 'Status'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Equipment Information', ['equipment_code', 'equipment_name', 'equipment_type']],
            ['Rack Position', ['rack_id', 'start_ru', 'height_ru']],
            ['Details', ['manufacturer', 'model', 'serial_number']],
            ['Specifications', ['power_consumption_w', 'port_count', 'ip_address']],
            ['Status', ['equipment_status']]
        ])
    
    def configure_ports_form(self):
        """Configure Ports form"""
        layer = self.get_layer_by_name('Equipment Ports')
        if not layer:
            return
        
        field_config = {
            'port_id': {'widget': 'Hidden'},
            'port_code': {
                'widget': 'TextEdit',
                'default': "concat('PORT-', lpad(to_string(maximum(port_id) + 1), 5, '0'))",
                'alias': 'Port Code',
                'mandatory': True
            },
            'equipment_id': {
                'widget': 'RelationReference',
                'relation': 'port_to_equipment',
                'alias': 'Equipment',
                'mandatory': True
            },
            'port_number': {'widget': 'Range', 'min': 1, 'max': 256, 'alias': 'Port Number', 'mandatory': True},
            'port_label': {'widget': 'TextEdit', 'alias': 'Label'},
            'port_type': {
                'widget': 'ValueMap',
                'values': {
                    'PON': 'PON',
                    'Ethernet': 'Ethernet',
                    'Fiber': 'Fiber',
                    'SFP': 'SFP',
                    'SFP+': 'SFP+',
                    'QSFP': 'QSFP'
                },
                'alias': 'Port Type'
            },
            'port_speed': {
                'widget': 'ValueMap',
                'values': {
                    '100M': '100M',
                    '1G': '1G',
                    '10G': '10G',
                    '25G': '25G',
                    '40G': '40G',
                    '100G': '100G'
                },
                'alias': 'Speed'
            },
            'connector_type': {
                'widget': 'ValueMap',
                'values': {'LC': 'LC', 'SC': 'SC', 'FC': 'FC', 'RJ45': 'RJ45', 'SFP': 'SFP'},
                'alias': 'Connector'
            },
            'port_status': {
                'widget': 'ValueMap',
                'values': {
                    'Available': 'available',
                    'In Use': 'in_use',
                    'Reserved': 'reserved',
                    'Faulty': 'faulty'
                },
                'default': 'available',
                'alias': 'Status'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Port Information', ['port_code', 'equipment_id', 'port_number', 'port_label']],
            ['Specifications', ['port_type', 'port_speed', 'connector_type']],
            ['Status', ['port_status']]
        ])
    
    def configure_patch_panels_form(self):
        """Configure Patch Panels form"""
        layer = self.get_layer_by_name('Patch Panels')
        if not layer:
            return
        
        field_config = {
            'panel_id': {'widget': 'Hidden'},
            'panel_code': {
                'widget': 'TextEdit',
                'default': "concat('PP-', lpad(to_string(maximum(panel_id) + 1), 4, '0'))",
                'alias': 'Panel Code',
                'mandatory': True
            },
            'panel_name': {'widget': 'TextEdit', 'alias': 'Panel Name'},
            'equipment_id': {
                'widget': 'RelationReference',
                'relation': 'panel_to_equipment',
                'alias': 'Mounted On',
                'allow_null': True
            },
            'panel_type': {
                'widget': 'ValueMap',
                'values': {
                    'Fiber Patch Panel': 'Fiber Patch Panel',
                    'Copper Patch Panel': 'Copper Patch Panel',
                    'Fiber Distribution Panel': 'Fiber Distribution Panel'
                },
                'alias': 'Panel Type'
            },
            'port_density': {
                'widget': 'ValueMap',
                'values': {
                    '12 Port': 12,
                    '24 Port': 24,
                    '48 Port': 48,
                    '96 Port': 96,
                    '144 Port': 144
                },
                'alias': 'Density'
            },
            'connector_type': {
                'widget': 'ValueMap',
                'values': {
                    'LC Duplex': 'LC Duplex',
                    'SC Duplex': 'SC Duplex',
                    'MPO/MTP 12F': 'MPO/MTP 12F',
                    'RJ45': 'RJ45'
                },
                'alias': 'Connector'
            },
            'fiber_type': {
                'widget': 'ValueMap',
                'values': {
                    'Single-mode OS2': 'Single-mode OS2',
                    'Multimode OM3': 'Multimode OM3',
                    'Multimode OM4': 'Multimode OM4'
                },
                'alias': 'Fiber Type'
            },
            'panel_status': {
                'widget': 'ValueMap',
                'values': {'Active': 'active', 'Inactive': 'inactive'},
                'default': 'active',
                'alias': 'Status'
            }
        }
        
        self.apply_field_configuration(layer, field_config)
        self.create_drag_drop_form(layer, [
            ['Panel Information', ['panel_code', 'panel_name', 'panel_type']],
            ['Mounting', ['equipment_id']],
            ['Specifications', ['port_density', 'connector_type', 'fiber_type']],
            ['Status', ['panel_status']]
        ])
    
    # ==========================================
    # HELPER METHODS
    # ==========================================
    
    def get_layer_by_name(self, layer_name):
        """Get layer by name"""
        layers = QgsProject.instance().mapLayersByName(layer_name)
        if layers:
            return layers[0]
        return None
    
    def apply_field_configuration(self, layer, field_config):
        """Apply field configuration to layer"""
        if not layer:
            return
        
        fields = layer.fields()
        
        for field_name, config in field_config.items():
            field_index = fields.indexOf(field_name)
            if field_index == -1:
                continue
            
            # Set widget type
            widget_type = config.get('widget', 'TextEdit')
            widget_config = {}
            
            if widget_type == 'Hidden':
                layer.setEditorWidgetSetup(field_index, QgsEditorWidgetSetup('Hidden', {}))
            
            elif widget_type == 'TextEdit':
                widget_config = {
                    'IsMultiline': config.get('multiline', False),
                    'UseHtml': False
                }
                if 'placeholder' in config:
                    widget_config['PlaceholderText'] = config['placeholder']
                layer.setEditorWidgetSetup(field_index, QgsEditorWidgetSetup('TextEdit', widget_config))
            
            elif widget_type == 'Range':
                widget_config = {
                    'Min': config.get('min', 0),
                    'Max': config.get('max', 999999),
                    'Step': config.get('step', 1),
                    'Style': 'SpinBox',
                    'AllowNull': config.get('allow_null', True)
                }
                if 'suffix' in config:
                    widget_config['Suffix'] = config['suffix']
                layer.setEditorWidgetSetup(field_index, QgsEditorWidgetSetup('Range', widget_config))
            
            elif widget_type == 'ValueMap':
                widget_config = {'map': config.get('values', {})}
                layer.setEditorWidgetSetup(field_index, QgsEditorWidgetSetup('ValueMap', widget_config))
            
            elif widget_type == 'DateTime':
                widget_config = {
                    'display_format': config.get('display_format', 'yyyy-MM-dd'),
                    'field_format': 'yyyy-MM-dd',
                    'calendar_popup': config.get('calendar_popup', True)
                }
                layer.setEditorWidgetSetup(field_index, QgsEditorWidgetSetup('DateTime', widget_config))
            
            elif widget_type == 'CheckBox':
                widget_config = {
                    'CheckedState': config.get('checked', True),
                    'UncheckedState': config.get('unchecked', False)
                }
                layer.setEditorWidgetSetup(field_index, QgsEditorWidgetSetup('CheckBox', widget_config))
            
            elif widget_type == 'RelationReference':
                widget_config = {
                    'Relation': config.get('relation', ''),
                    'AllowNULL': config.get('allow_null', False),
                    'ShowForm': True,
                    'ShowOpenFormButton': True
                }
                layer.setEditorWidgetSetup(field_index, QgsEditorWidgetSetup('RelationReference', widget_config))
            
            # Set alias
            if 'alias' in config:
                layer.setFieldAlias(field_index, config['alias'])
            
            # Set default value
            if 'default' in config and config['default']:
                default_value = QgsDefaultValue(config['default'])
                layer.setDefaultValueDefinition(field_index, default_value)
            
            # Set expression for calculated fields
            if 'expression' in config:
                default_value = QgsDefaultValue(config['expression'], True)
                layer.setDefaultValueDefinition(field_index, default_value)
            
            # Set mandatory constraint
            if config.get('mandatory', False):
                layer.setFieldConstraint(field_index, QgsFieldConstraint.ConstraintNotNull, 
                                        QgsFieldConstraint.ConstraintStrengthHard)
            
            # Set read-only
            if config.get('read_only', False):
                layer.setFieldConstraint(field_index, QgsFieldConstraint.ConstraintNotNull, 
                                        QgsFieldConstraint.ConstraintStrengthSoft)
                layer.editFormConfig().setReadOnly(field_index, True)
    
    def create_drag_drop_form(self, layer, tab_structure):
        """Create drag and drop form layout"""
        if not layer:
            return

        from qgis.core import QgsAttributeEditorContainer, QgsAttributeEditorField

        form_config = layer.editFormConfig()
        form_config.clearTabs()
        form_config.setLayout(QgsEditFormConfig.TabLayout)

        root = form_config.invisibleRootContainer()

        for tab_name, field_names in tab_structure:
            # Create a tab container
            tab_container = QgsAttributeEditorContainer(tab_name, root)
            tab_container.setIsGroupBox(False)

            # Add fields to the tab
            for field_name in field_names:
                field_index = layer.fields().indexOf(field_name)
                if field_index >= 0:
                    field_editor = QgsAttributeEditorField(field_name, field_index, tab_container)
                    tab_container.addChildElement(field_editor)

            root.addChildElement(tab_container)

        layer.setEditFormConfig(form_config)
    
    # ==========================================
    # SNAPPING CONFIGURATION
    # ==========================================
    
    def configure_snapping(self):
        """Configure snapping for all FTTX layers"""
        snapping_config = self.project.snappingConfig()
        
        # Enable snapping
        snapping_config.setEnabled(True)
        snapping_config.setMode(QgsSnappingConfig.AllLayers)
        snapping_config.setType(QgsSnappingConfig.VertexAndSegment)
        snapping_config.setTolerance(10)
        snapping_config.setUnits(QgsTolerance.Pixels)
        snapping_config.setIntersectionSnapping(True)
        
        # Configure individual layers
        layers_to_snap = [
            'OLT Equipment',
            'ONUs',
            'Network Nodes',
            'Fiber Cables',
            'Routes',
            'Splice Points',
            'Buildings',
            'Rooms'
        ]
        
        for layer_name in layers_to_snap:
            layer = self.get_layer_by_name(layer_name)
            if layer:
                individual_settings = QgsSnappingConfig.IndividualLayerSettings(
                    True,  # enabled
                    QgsSnappingConfig.VertexAndSegment,  # type
                    15,  # tolerance
                    QgsTolerance.Pixels  # units
                )
                snapping_config.setIndividualLayerSettings(layer, individual_settings)
        
        # Apply snapping configuration
        self.project.setSnappingConfig(snapping_config)
        
        # Enable topological editing
        self.project.setTopologicalEditing(True)
        
        # Enable avoid intersections
        self.project.setAvoidIntersectionsMode(QgsProject.AvoidIntersectionsMode.AvoidIntersectionsLayers)


