# fiberweave/algorithms/validate_network.py
"""
Network Validation Algorithm
Checks fiber network integrity and compliance
"""

from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterString,
    QgsProcessingParameterBoolean,
    QgsProcessingOutputString,
    QgsProcessingOutputNumber,
    QgsMessageLog,
    Qgis
)
from qgis.PyQt.QtCore import QCoreApplication
import psycopg2


class ValidateNetworkAlgorithm(QgsProcessingAlgorithm):
    """Validates fiber network topology and configuration"""
    
    # Parameters
    DB_HOST = 'DB_HOST'
    DB_PORT = 'DB_PORT'
    DB_NAME = 'DB_NAME'
    DB_USER = 'DB_USER'
    DB_PASSWORD = 'DB_PASSWORD'
    CHECK_TOPOLOGY = 'CHECK_TOPOLOGY'
    CHECK_POWER_BUDGET = 'CHECK_POWER_BUDGET'
    
    # Outputs
    OUTPUT_STATUS = 'OUTPUT_STATUS'
    OUTPUT_ERROR_COUNT = 'OUTPUT_ERROR_COUNT'
    OUTPUT_WARNING_COUNT = 'OUTPUT_WARNING_COUNT'
    
    def tr(self, string):
        """Translate string"""
        return QCoreApplication.translate('Processing', string)
    
    def createInstance(self):
        """Create new instance"""
        return ValidateNetworkAlgorithm()
    
    def name(self):
        """Algorithm name (internal)"""
        return 'validate_network'
    
    def displayName(self):
        """Algorithm name (display)"""
        return self.tr('Validate Network')
    
    def group(self):
        """Algorithm group"""
        return self.tr('Network Analysis')
    
    def groupId(self):
        """Group ID"""
        return 'analysis'
    
    def shortHelpString(self):
        """Help text"""
        return self.tr(
            'Validates fiber network topology and configuration.\n\n'
            'Checks include:\n'
            '- OLT connectivity\n'
            '- Splitter cascading\n'
            '- ONU assignments\n'
            '- Power budget compliance\n'
            '- Network topology integrity\n\n'
            'Returns detailed validation report.'
        )
    
    def initAlgorithm(self, config=None):
        """Define inputs and outputs"""
        
        # Database connection parameters
        self.addParameter(
            QgsProcessingParameterString(
                self.DB_HOST,
                self.tr('Database Host'),
                defaultValue='localhost'
            )
        )
        
        self.addParameter(
            QgsProcessingParameterString(
                self.DB_PORT,
                self.tr('Database Port'),
                defaultValue='5432'
            )
        )
        
        self.addParameter(
            QgsProcessingParameterString(
                self.DB_NAME,
                self.tr('Database Name'),
                defaultValue='fiberweave_network'
            )
        )
        
        self.addParameter(
            QgsProcessingParameterString(
                self.DB_USER,
                self.tr('Database User'),
                defaultValue='postgres'
            )
        )
        
        self.addParameter(
            QgsProcessingParameterString(
                self.DB_PASSWORD,
                self.tr('Database Password'),
                defaultValue=''
            )
        )
        
        # Validation options
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.CHECK_TOPOLOGY,
                self.tr('Check Network Topology'),
                defaultValue=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.CHECK_POWER_BUDGET,
                self.tr('Check Power Budget Compliance'),
                defaultValue=True
            )
        )
        
        # Outputs
        self.addOutput(
            QgsProcessingOutputString(
                self.OUTPUT_STATUS,
                self.tr('Validation Status')
            )
        )
        
        self.addOutput(
            QgsProcessingOutputNumber(
                self.OUTPUT_ERROR_COUNT,
                self.tr('Error Count')
            )
        )
        
        self.addOutput(
            QgsProcessingOutputNumber(
                self.OUTPUT_WARNING_COUNT,
                self.tr('Warning Count')
            )
        )
    
    def processAlgorithm(self, parameters, context, feedback):
        """Run the algorithm"""
        
        # Get parameters
        host = self.parameterAsString(parameters, self.DB_HOST, context)
        port = self.parameterAsString(parameters, self.DB_PORT, context)
        database = self.parameterAsString(parameters, self.DB_NAME, context)
        user = self.parameterAsString(parameters, self.DB_USER, context)
        password = self.parameterAsString(parameters, self.DB_PASSWORD, context)
        
        check_topology = self.parameterAsBoolean(parameters, self.CHECK_TOPOLOGY, context)
        check_power = self.parameterAsBoolean(parameters, self.CHECK_POWER_BUDGET, context)
        
        feedback.pushInfo('Starting network validation...')
        
        errors = []
        warnings = []
        
        try:
            # Connect to database
            feedback.pushInfo(f'Connecting to database: {database}')
            conn = psycopg2.connect(
                host=host, port=port, database=database,
                user=user, password=password
            )
            cur = conn.cursor()
            
            # Check 1: OLT equipment
            feedback.setProgress(10)
            feedback.pushInfo('Checking OLT equipment...')
            
            cur.execute("""
                SELECT COUNT(*) FROM fttx.itu_olt_equipment
                WHERE equipment_status IN ('active', 'operational')
            """)
            olt_count = cur.fetchone()[0]

            if olt_count == 0:
                errors.append('No active/operational OLT equipment found')
            else:
                feedback.pushInfo(f'✓ Found {olt_count} active OLT(s)')
            
            # Check 2: PON ports
            feedback.setProgress(20)
            feedback.pushInfo('Checking PON ports...')
            
            cur.execute("""
                SELECT COUNT(*) FROM fttx.itu_olt_pon_ports
                WHERE port_status = 'active'
            """)
            port_count = cur.fetchone()[0]
            
            if port_count == 0:
                errors.append('No active PON ports found')
            else:
                feedback.pushInfo(f'✓ Found {port_count} active PON port(s)')
            
            # Check 3: ONUs
            feedback.setProgress(30)
            feedback.pushInfo('Checking ONU equipment...')
            
            cur.execute("""
                SELECT COUNT(*) FROM fttx.itu_onu_equipment
                WHERE onu_status IN ('active', 'online')
            """)
            onu_count = cur.fetchone()[0]

            if onu_count == 0:
                warnings.append('No active/online ONUs found')
            else:
                feedback.pushInfo(f'✓ Found {onu_count} active ONU(s)')

            # Check 4: Orphaned ONUs (no OLT connection)
            feedback.setProgress(40)
            feedback.pushInfo('Checking ONU connectivity...')

            cur.execute("""
                SELECT COUNT(*) FROM fttx.itu_onu_equipment
                WHERE connected_olt_id IS NULL AND onu_status IN ('active', 'online')
            """)
            orphaned_onus = cur.fetchone()[0]
            
            if orphaned_onus > 0:
                warnings.append(f'{orphaned_onus} active ONU(s) not connected to OLT')
            
            # Check 5: Splitter nodes
            feedback.setProgress(50)
            feedback.pushInfo('Checking splitter nodes...')
            
            cur.execute("""
                SELECT COUNT(*) FROM fttx.itu_odn_nodes
                WHERE node_type = 'splitter' AND operational_status IN ('active', 'operational')
            """)
            splitter_count = cur.fetchone()[0]

            if splitter_count == 0:
                warnings.append('No active/operational splitter nodes found')
            else:
                feedback.pushInfo(f'✓ Found {splitter_count} active splitter(s)')
            
            # Check 6: Fiber cables
            feedback.setProgress(60)
            feedback.pushInfo('Checking fiber cables...')
            
            cur.execute("""
                SELECT COUNT(*) FROM fttx.itu_fiber_cables
                WHERE operational_status IN ('active', 'operational')
            """)
            cable_count = cur.fetchone()[0]

            if cable_count == 0:
                errors.append('No active/operational fiber cables found')
            else:
                feedback.pushInfo(f'✓ Found {cable_count} active cable(s)')
            
            # Check 7: Network topology (if enabled)
            if check_topology:
                feedback.setProgress(70)
                feedback.pushInfo('Checking network topology...')
                
                # Check for disconnected splitters
                cur.execute("""
                    SELECT COUNT(*) FROM fttx.itu_odn_nodes
                    WHERE node_type = 'splitter' 
                    AND parent_node_id IS NULL 
                    AND olt_id IS NULL
                    AND operational_status IN ('active', 'operational')
                """)
                disconnected_splitters = cur.fetchone()[0]
                
                if disconnected_splitters > 0:
                    warnings.append(
                        f'{disconnected_splitters} splitter(s) not connected to network'
                    )
            
            # Check 8: Power budget (if enabled)
            if check_power:
                feedback.setProgress(80)
                feedback.pushInfo('Checking power budget compliance...')
                
                # Check if power budget calculations exist
                cur.execute("""
                    SELECT COUNT(*) FROM fttx.itu_power_budget
                    WHERE meets_itu_requirements = false
                """)
                non_compliant = cur.fetchone()[0]
                
                if non_compliant > 0:
                    warnings.append(
                        f'{non_compliant} path(s) do not meet ITU-T power budget requirements'
                    )
            
            # Check 9: Data integrity
            feedback.setProgress(90)
            feedback.pushInfo('Checking data integrity...')
            
            # Check for duplicate ONU serial numbers
            cur.execute("""
                SELECT onu_serial_number, COUNT(*) 
                FROM fttx.itu_onu_equipment
                GROUP BY onu_serial_number
                HAVING COUNT(*) > 1
            """)
            duplicates = cur.fetchall()
            
            if duplicates:
                errors.append(f'{len(duplicates)} duplicate ONU serial number(s) found')
            
            cur.close()
            conn.close()
            
            feedback.setProgress(100)
            
            # Generate summary
            error_count = len(errors)
            warning_count = len(warnings)
            
            feedback.pushInfo('')
            feedback.pushInfo('=' * 50)
            feedback.pushInfo('VALIDATION SUMMARY')
            feedback.pushInfo('=' * 50)
            
            if error_count == 0 and warning_count == 0:
                status = '✓ PASSED - Network is valid'
                feedback.pushInfo(status)
            elif error_count == 0:
                status = f'⚠ PASSED WITH WARNINGS ({warning_count} warning(s))'
                feedback.pushInfo(status)
            else:
                status = f'✗ FAILED ({error_count} error(s), {warning_count} warning(s))'
                feedback.pushInfo(status)
            
            if errors:
                feedback.pushInfo('')
                feedback.pushInfo('ERRORS:')
                for error in errors:
                    feedback.pushInfo(f'  ✗ {error}')
            
            if warnings:
                feedback.pushInfo('')
                feedback.pushInfo('WARNINGS:')
                for warning in warnings:
                    feedback.pushInfo(f'  ⚠ {warning}')
            
            feedback.pushInfo('=' * 50)
            
            return {
                self.OUTPUT_STATUS: status,
                self.OUTPUT_ERROR_COUNT: error_count,
                self.OUTPUT_WARNING_COUNT: warning_count
            }
            
        except Exception as e:
            feedback.reportError(f'Validation failed: {str(e)}')
            return {
                self.OUTPUT_STATUS: f'ERROR: {str(e)}',
                self.OUTPUT_ERROR_COUNT: -1,
                self.OUTPUT_WARNING_COUNT: -1
            }