# fiberweave/algorithms/export_report.py
"""
Export Network Report Algorithm
Generates comprehensive network documentation
"""

from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterString,
    QgsProcessingParameterFileDestination,
    QgsProcessingOutputFile,
    QgsMessageLog,
    Qgis
)
from qgis.PyQt.QtCore import QCoreApplication
import psycopg2
from datetime import datetime


class ExportNetworkReportAlgorithm(QgsProcessingAlgorithm):
    """Exports comprehensive network report to text file"""
    
    # Parameters
    DB_HOST = 'DB_HOST'
    DB_PORT = 'DB_PORT'
    DB_NAME = 'DB_NAME'
    DB_USER = 'DB_USER'
    DB_PASSWORD = 'DB_PASSWORD'
    OUTPUT_FILE = 'OUTPUT_FILE'
    
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
    
    def createInstance(self):
        return ExportNetworkReportAlgorithm()
    
    def name(self):
        return 'export_network_report'
    
    def displayName(self):
        return self.tr('Export Network Report')
    
    def group(self):
        return self.tr('Documentation')
    
    def groupId(self):
        return 'documentation'
    
    def shortHelpString(self):
        return self.tr(
            'Generates a comprehensive network report including:\n\n'
            '- Network summary statistics\n'
            '- Equipment inventory\n'
            '- ONU connection details\n'
            '- Power budget analysis\n'
            '- Compliance status\n\n'
            'Output is a formatted text file suitable for documentation.'
        )
    
    def initAlgorithm(self, config=None):
        # Database parameters
        self.addParameter(
            QgsProcessingParameterString(
                self.DB_HOST,
                self.tr('Database Host'),
                defaultValue='localhost'
            )
        )
        
        self.addParameter(
            QgsProcessingParameterString(
                self.DB_PORT,
                self.tr('Database Port'),
                defaultValue='5432'
            )
        )
        
        self.addParameter(
            QgsProcessingParameterString(
                self.DB_NAME,
                self.tr('Database Name'),
                defaultValue='fiberweave_network'
            )
        )
        
        self.addParameter(
            QgsProcessingParameterString(
                self.DB_USER,
                self.tr('Database User'),
                defaultValue='postgres'
            )
        )
        
        self.addParameter(
            QgsProcessingParameterString(
                self.DB_PASSWORD,
                self.tr('Database Password'),
                defaultValue=''
            )
        )
        
        # Output file
        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT_FILE,
                self.tr('Output Report File'),
                fileFilter='Text files (*.txt)'
            )
        )
    
    def processAlgorithm(self, parameters, context, feedback):
        # Get parameters
        host = self.parameterAsString(parameters, self.DB_HOST, context)
        port = self.parameterAsString(parameters, self.DB_PORT, context)
        database = self.parameterAsString(parameters, self.DB_NAME, context)
        user = self.parameterAsString(parameters, self.DB_USER, context)
        password = self.parameterAsString(parameters, self.DB_PASSWORD, context)
        output_file = self.parameterAsFileOutput(parameters, self.OUTPUT_FILE, context)
        
        feedback.pushInfo('Generating network report...')
        
        try:
            # Connect to database
            feedback.pushInfo(f'Connecting to database: {database}')
            conn = psycopg2.connect(
                host=host, port=port, database=database,
                user=user, password=password
            )
            cur = conn.cursor()
            
            report_lines = []
            
            # Header
            report_lines.append('=' * 80)
            report_lines.append('FIBERWEAVE NETWORK REPORT')
            report_lines.append('=' * 80)
            report_lines.append(f'Generated: {datetime.now().strftime("%Y-%m-%d %H:%M:%S")}')
            report_lines.append(f'Database: {database}')
            report_lines.append('=' * 80)
            report_lines.append('')
            
            # Section 1: Network Summary
            feedback.setProgress(20)
            report_lines.append('NETWORK SUMMARY')
            report_lines.append('-' * 80)
            
            # OLTs
            cur.execute("SELECT COUNT(*) FROM fttx.itu_olt_equipment WHERE equipment_status = 'active'")
            olt_count = cur.fetchone()[0]
            report_lines.append(f'Active OLT Equipment: {olt_count}')
            
            # PON Ports
            cur.execute("SELECT COUNT(*) FROM fttx.itu_olt_pon_ports WHERE port_status = 'active'")
            port_count = cur.fetchone()[0]
            report_lines.append(f'Active PON Ports: {port_count}')
            
            # Splitters
            cur.execute("SELECT COUNT(*) FROM fttx.itu_odn_nodes WHERE operational_status = 'active'")
            splitter_count = cur.fetchone()[0]
            report_lines.append(f'Active ODN Nodes: {splitter_count}')
            
            # ONUs
            cur.execute("SELECT COUNT(*) FROM fttx.itu_onu_equipment WHERE onu_status = 'active'")
            onu_count = cur.fetchone()[0]
            report_lines.append(f'Active ONUs: {onu_count}')
            
            # Cables
            cur.execute("SELECT COUNT(*) FROM fttx.itu_fiber_cables WHERE operational_status = 'active'")
            cable_count = cur.fetchone()[0]
            report_lines.append(f'Active Fiber Cables: {cable_count}')
            
            # Total fiber length
            cur.execute("SELECT SUM(cable_length_m) FROM fttx.itu_fiber_cables WHERE operational_status = 'active'")
            total_length = cur.fetchone()[0] or 0
            report_lines.append(f'Total Fiber Length: {total_length/1000:.2f} km')
            
            report_lines.append('')
            
            # Section 2: OLT Details
            feedback.setProgress(40)
            report_lines.append('OLT EQUIPMENT DETAILS')
            report_lines.append('-' * 80)
            
            cur.execute("""
                SELECT olt_code, pon_standard, manufacturer, model, 
                       max_onus_supported, current_onus_connected
                FROM fttx.itu_olt_equipment
                WHERE equipment_status = 'active'
            """)
            
            for row in cur.fetchall():
                olt_code, pon_std, mfr, model, max_onus, current = row
                report_lines.append(f'  OLT: {olt_code}')
                report_lines.append(f'    Standard: {pon_std}')
                report_lines.append(f'    Equipment: {mfr} {model}')
                report_lines.append(f'    Capacity: {current}/{max_onus} ONUs')
                report_lines.append('')
            
            # Section 3: Power Budget Summary
            feedback.setProgress(60)
            report_lines.append('POWER BUDGET ANALYSIS')
            report_lines.append('-' * 80)
            
            cur.execute("""
                SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN meets_itu_requirements THEN 1 ELSE 0 END) as compliant,
                    AVG(downstream_margin_db) as avg_down_margin,
                    AVG(upstream_margin_db) as avg_up_margin
                FROM fttx.itu_power_budget
            """)
            
            pb_data = cur.fetchone()
            if pb_data and pb_data[0] > 0:
                total, compliant, avg_down, avg_up = pb_data
                non_compliant = total - compliant
                
                report_lines.append(f'Total Paths Analyzed: {total}')
                report_lines.append(f'Compliant Paths: {compliant}')
                report_lines.append(f'Non-Compliant Paths: {non_compliant}')
                report_lines.append(f'Average Downstream Margin: {avg_down:.2f} dB')
                report_lines.append(f'Average Upstream Margin: {avg_up:.2f} dB')
                
                if total > 0:
                    compliance_rate = (compliant / total) * 100
                    report_lines.append(f'Compliance Rate: {compliance_rate:.1f}%')
            else:
                report_lines.append('No power budget calculations available')
            
            report_lines.append('')
            
            # Section 4: Non-Compliant Paths (if any)
            feedback.setProgress(80)
            cur.execute("""
                SELECT 
                    pb.budget_id,
                    o.onu_serial_number,
                    pb.downstream_margin_db,
                    pb.upstream_margin_db,
                    pb.limiting_factor
                FROM fttx.itu_power_budget pb
                JOIN fttx.itu_onu_equipment o ON pb.onu_id = o.onu_id
                WHERE pb.meets_itu_requirements = false
                LIMIT 20
            """)
            
            non_compliant_paths = cur.fetchall()
            if non_compliant_paths:
                report_lines.append('NON-COMPLIANT PATHS (Top 20)')
                report_lines.append('-' * 80)
                
                for row in non_compliant_paths:
                    budget_id, serial, down_margin, up_margin, limiting = row
                    report_lines.append(f'  ONU: {serial}')
                    report_lines.append(f'    Downstream Margin: {down_margin:.2f} dB')
                    report_lines.append(f'    Upstream Margin: {up_margin:.2f} dB')
                    report_lines.append(f'    Limiting Factor: {limiting}')
                    report_lines.append('')
            
            # Footer
            report_lines.append('=' * 80)
            report_lines.append('END OF REPORT')
            report_lines.append('=' * 80)
            
            cur.close()
            conn.close()
            
            # Write to file
            feedback.setProgress(90)
            feedback.pushInfo(f'Writing report to: {output_file}')
            
            with open(output_file, 'w', encoding='utf-8') as f:
                f.write('\n'.join(report_lines))
            
            feedback.setProgress(100)
            feedback.pushInfo('✓ Report generated successfully')
            
            return {self.OUTPUT_FILE: output_file}
            
        except Exception as e:
            feedback.reportError(f'Report generation failed: {str(e)}')
            raise
