-- ==========================================
-- BUILDING TABLE ENHANCEMENTS
-- Adds service status and auto-generation of building codes
-- ==========================================

SET search_path TO fttx, public;

-- ==========================================
-- STEP 1: ADD SERVICE STATUS COLUMN
-- ==========================================

-- Add service_status column if it doesn't exist
DO $$
BEGIN
    IF NOT EXISTS (
        SELECT 1 FROM information_schema.columns 
        WHERE table_schema = 'fttx' 
        AND table_name = 'buildings' 
        AND column_name = 'service_status'
    ) THEN
        ALTER TABLE fttx.buildings 
        ADD COLUMN service_status VARCHAR(30) DEFAULT 'not_serviceable';
        
        RAISE NOTICE '✓ Added service_status column to buildings';
    ELSE
        RAISE NOTICE 'ℹ service_status column already exists';
    END IF;
END $$;

-- Create index for performance
CREATE INDEX IF NOT EXISTS idx_buildings_service_status 
ON fttx.buildings(service_status);

-- Add check constraint for valid values
ALTER TABLE fttx.buildings
DROP CONSTRAINT IF EXISTS chk_building_service_status;

ALTER TABLE fttx.buildings
ADD CONSTRAINT chk_building_service_status 
CHECK (service_status IN ('serviceable', 'connected', 'not_serviceable', 'planned'));

COMMENT ON COLUMN fttx.buildings.service_status IS 
'Service availability status: connected (has active service), serviceable (can be connected), not_serviceable (out of reach), planned (future coverage)';

-- Update existing records to default if NULL
UPDATE fttx.buildings 
SET service_status = 'not_serviceable'
WHERE service_status IS NULL;

-- ==========================================
-- STEP 2: AUTO-GENERATE BUILDING CODES
-- ==========================================

-- Function to generate building code automatically
CREATE OR REPLACE FUNCTION generate_building_code()
RETURNS TRIGGER AS $$
DECLARE
    v_next_num INTEGER;
    v_code VARCHAR(50);
BEGIN
    -- Only generate if building_code is NULL
    IF NEW.building_code IS NULL THEN
        -- Get next number by finding max existing number
        SELECT COALESCE(MAX(CAST(SUBSTRING(building_code FROM '[0-9]+') AS INTEGER)), 0) + 1
        INTO v_next_num
        FROM fttx.buildings
        WHERE building_code ~ '^BLD-[0-9]+$';
        
        -- Generate code: BLD-0001, BLD-0002, etc.
        v_code := 'BLD-' || LPAD(v_next_num::TEXT, 4, '0');
        NEW.building_code := v_code;
        
        RAISE NOTICE 'Auto-generated building code: %', v_code;
    END IF;
    
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Drop existing trigger if exists
DROP TRIGGER IF EXISTS trg_generate_building_code ON fttx.buildings;

-- Create trigger to auto-generate building code on INSERT
CREATE TRIGGER trg_generate_building_code
    BEFORE INSERT ON fttx.buildings
    FOR EACH ROW
    EXECUTE FUNCTION generate_building_code();

COMMENT ON FUNCTION generate_building_code() IS 
'Automatically generates building codes in format BLD-0001, BLD-0002, etc.';

-- ==========================================
-- STEP 3: UPDATE EXISTING BUILDINGS
-- ==========================================

-- Generate codes for existing buildings that have NULL building_code
DO $$
DECLARE
    v_building RECORD;
    v_counter INTEGER := 1;
    v_updated INTEGER := 0;
BEGIN
    -- Find the starting number (1 more than highest existing code)
    SELECT COALESCE(MAX(CAST(SUBSTRING(building_code FROM '[0-9]+') AS INTEGER)), 0) + 1
    INTO v_counter
    FROM fttx.buildings
    WHERE building_code ~ '^BLD-[0-9]+$';
    
    -- Update buildings with NULL codes
    FOR v_building IN 
        SELECT building_id 
        FROM fttx.buildings 
        WHERE building_code IS NULL
        ORDER BY building_id
    LOOP
        UPDATE fttx.buildings
        SET building_code = 'BLD-' || LPAD(v_counter::TEXT, 4, '0')
        WHERE building_id = v_building.building_id;
        
        v_counter := v_counter + 1;
        v_updated := v_updated + 1;
    END LOOP;
    
    IF v_updated > 0 THEN
        RAISE NOTICE '✓ Updated % existing buildings with auto-generated codes', v_updated;
    ELSE
        RAISE NOTICE 'ℹ No buildings needed code generation';
    END IF;
END $$;

-- ==========================================
-- STEP 4: ADD HELPER VIEWS
-- ==========================================

-- View: Buildings by service status
CREATE OR REPLACE VIEW fttx.v_buildings_by_service_status AS
SELECT 
    service_status,
    COUNT(*) as building_count,
    SUM(floors_count) as total_floors
FROM fttx.buildings
GROUP BY service_status
ORDER BY 
    CASE service_status
        WHEN 'connected' THEN 1
        WHEN 'serviceable' THEN 2
        WHEN 'planned' THEN 3
        WHEN 'not_serviceable' THEN 4
        ELSE 5
    END;

COMMENT ON VIEW fttx.v_buildings_by_service_status IS 
'Summary of buildings grouped by service status';

-- View: Buildings ready for connection
CREATE OR REPLACE VIEW fttx.v_buildings_ready_for_connection AS
SELECT 
    b.building_id,
    b.building_code,
    b.building_name,
    b.building_type,
    b.street_address,
    b.city,
    b.floors_count,
    b.service_status,
    b.geom,
    -- Find nearest OLT
    (
        SELECT olt_code 
        FROM fttx.itu_olt_equipment 
        WHERE equipment_status = 'operational'
        ORDER BY ST_Distance(b.geom, itu_olt_equipment.geom)
        LIMIT 1
    ) as nearest_olt,
    -- Distance to nearest OLT
    (
        SELECT ROUND(ST_Distance(b.geom, itu_olt_equipment.geom)::NUMERIC, 2)
        FROM fttx.itu_olt_equipment 
        WHERE equipment_status = 'operational'
        ORDER BY ST_Distance(b.geom, itu_olt_equipment.geom)
        LIMIT 1
    ) as distance_to_olt_m
FROM fttx.buildings b
WHERE b.service_status = 'serviceable'
ORDER BY b.building_code;

COMMENT ON VIEW fttx.v_buildings_ready_for_connection IS 
'Buildings that are serviceable and ready for fiber connection';

-- ==========================================
-- STEP 5: TESTING
-- ==========================================

-- Test the auto-generation (commented out - uncomment to test)
-- INSERT INTO fttx.buildings (building_name, building_type, service_status)
-- VALUES ('Test Building', 'residential', 'not_serviceable');
-- SELECT building_code FROM fttx.buildings WHERE building_name = 'Test Building';
-- DELETE FROM fttx.buildings WHERE building_name = 'Test Building';

-- ==========================================
-- SUMMARY
-- ==========================================

DO $$ 
DECLARE
    v_total_buildings INTEGER;
    v_connected INTEGER;
    v_serviceable INTEGER;
    v_not_serviceable INTEGER;
    v_planned INTEGER;
BEGIN
    -- Get counts
    SELECT COUNT(*) INTO v_total_buildings FROM fttx.buildings;
    SELECT COUNT(*) INTO v_connected FROM fttx.buildings WHERE service_status = 'connected';
    SELECT COUNT(*) INTO v_serviceable FROM fttx.buildings WHERE service_status = 'serviceable';
    SELECT COUNT(*) INTO v_not_serviceable FROM fttx.buildings WHERE service_status = 'not_serviceable';
    SELECT COUNT(*) INTO v_planned FROM fttx.buildings WHERE service_status = 'planned';
    
    RAISE NOTICE '';
    RAISE NOTICE '=================================================';
    RAISE NOTICE '✓ BUILDING ENHANCEMENTS COMPLETED SUCCESSFULLY!';
    RAISE NOTICE '=================================================';
    RAISE NOTICE '';
    RAISE NOTICE 'Enhancements Applied:';
    RAISE NOTICE '  ✓ Added service_status column with constraint';
    RAISE NOTICE '  ✓ Created auto-generation trigger for building_code';
    RAISE NOTICE '  ✓ Updated existing buildings with codes';
    RAISE NOTICE '  ✓ Created helper views';
    RAISE NOTICE '';
    RAISE NOTICE 'Building Statistics:';
    RAISE NOTICE '  Total Buildings: %', v_total_buildings;
    RAISE NOTICE '  Connected: %', v_connected;
    RAISE NOTICE '  Serviceable: %', v_serviceable;
    RAISE NOTICE '  Not Serviceable: %', v_not_serviceable;
    RAISE NOTICE '  Planned: %', v_planned;
    RAISE NOTICE '';
    RAISE NOTICE 'Usage:';
    RAISE NOTICE '  • New buildings will auto-generate codes (BLD-0001, etc.)';
    RAISE NOTICE '  • Service status values: connected, serviceable, not_serviceable, planned';
    RAISE NOTICE '  • Use v_buildings_ready_for_connection view to see serviceable buildings';
    RAISE NOTICE '=================================================';
END $$;

-- Grant permissions
GRANT SELECT ON fttx.v_buildings_by_service_status TO PUBLIC;
GRANT SELECT ON fttx.v_buildings_ready_for_connection TO PUBLIC;