-- ==========================================
-- ADD SERVICE STATUS TO BUILDINGS TABLE
-- ==========================================

SET search_path TO fttx, public;

-- Add service_status column
ALTER TABLE fttx.buildings 
ADD COLUMN IF NOT EXISTS service_status VARCHAR(30);

-- Add index for performance
CREATE INDEX IF NOT EXISTS idx_buildings_service_status 
ON fttx.buildings(service_status);

-- Set default value for existing records
UPDATE fttx.buildings 
SET service_status = 'not_serviceable'
WHERE service_status IS NULL;

-- Add check constraint to ensure valid values
ALTER TABLE fttx.buildings
DROP CONSTRAINT IF EXISTS chk_building_service_status;

ALTER TABLE fttx.buildings
ADD CONSTRAINT chk_building_service_status 
CHECK (service_status IN (
    'serviceable',
    'connected', 
    'not_serviceable',
    'planned',
    NULL
));

-- Create helper view for service status dropdown
CREATE OR REPLACE VIEW fttx.v_building_service_status AS
SELECT 
    status as code,
    display_name
FROM (VALUES 
    ('serviceable', 'Serviceable - Fiber Available'),
    ('connected', 'Connected - Active Service'),
    ('not_serviceable', 'Not Serviceable - Outside Coverage'),
    ('planned', 'Planned - Future Coverage')
) AS statuses(status, display_name)
ORDER BY 
    CASE status
        WHEN 'connected' THEN 1
        WHEN 'serviceable' THEN 2
        WHEN 'planned' THEN 3
        WHEN 'not_serviceable' THEN 4
    END;

-- Grant permissions
GRANT SELECT ON fttx.v_building_service_status TO PUBLIC;

-- Success message
DO $$ 
BEGIN
    RAISE NOTICE '=================================================';
    RAISE NOTICE '✓ Service Status added to buildings table!';
    RAISE NOTICE '=================================================';
    RAISE NOTICE 'Column added: service_status';
    RAISE NOTICE 'Valid values:';
    RAISE NOTICE '  - serviceable   (fiber available, not yet connected)';
    RAISE NOTICE '  - connected     (has active fiber service)';
    RAISE NOTICE '  - not_serviceable (outside coverage area)';
    RAISE NOTICE '  - planned       (future coverage planned)';
    RAISE NOTICE '';
    RAISE NOTICE 'Current records set to: not_serviceable';
    RAISE NOTICE 'Update as needed based on your network coverage.';
    RAISE NOTICE '=================================================';
END $$;

-- Optional: Update buildings within certain distance of fiber as serviceable
-- Uncomment and adjust distance as needed:

/*
-- Example: Mark buildings within 200m of active fiber as serviceable
UPDATE fttx.buildings b
SET service_status = 'serviceable'
WHERE service_status = 'not_serviceable'
AND EXISTS (
    SELECT 1 
    FROM fttx.itu_fiber_cables c
    WHERE c.operational_status = 'active'
    AND ST_DWithin(
        b.geom::geography,
        c.geom::geography,
        200  -- 200 meters
    )
);

-- Example: Mark buildings with connected ONUs as connected
UPDATE fttx.buildings b
SET service_status = 'connected'
WHERE EXISTS (
    SELECT 1
    FROM fttx.itu_onu_equipment o
    WHERE o.onu_status = 'online'
    AND ST_DWithin(
        b.geom::geography,
        o.geom::geography,
        50  -- 50 meters (same building)
    )
);
*/