-- ==========================================
-- INSIDE PLANT DATABASE TABLES (FIXED)
-- Rooms, Racks, Equipment, and Ports
-- ==========================================

SET search_path TO fttx, public;

-- ==========================================
-- BUILDINGS AND ROOMS
-- ==========================================

CREATE TABLE IF NOT EXISTS fttx.buildings (
    building_id SERIAL PRIMARY KEY,
    building_code VARCHAR(50) UNIQUE NOT NULL,
    building_name VARCHAR(200),
    street_address TEXT,
    city VARCHAR(100),
    postal_code VARCHAR(20),
    building_type VARCHAR(50),
    floors_count INTEGER,
    construction_year INTEGER,
    owner VARCHAR(100),
    contact_person VARCHAR(100),
    contact_phone VARCHAR(50),
    geom GEOMETRY(POINT, 4326),
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    notes TEXT
);

CREATE TABLE IF NOT EXISTS fttx.rooms (
    room_id SERIAL PRIMARY KEY,
    room_code VARCHAR(50) UNIQUE NOT NULL,
    room_name VARCHAR(200),
    building_id INTEGER REFERENCES fttx.buildings(building_id),
    floor_number INTEGER,
    room_type VARCHAR(50),
    room_size_sqm NUMERIC(8,2),
    ceiling_height_m NUMERIC(4,2),
    access_type VARCHAR(50),
    has_hvac BOOLEAN DEFAULT FALSE,
    has_ups BOOLEAN DEFAULT FALSE,
    has_fire_suppression BOOLEAN DEFAULT FALSE,
    temperature_min_c INTEGER,
    temperature_max_c INTEGER,
    humidity_min_percent INTEGER,
    humidity_max_percent INTEGER,
    power_capacity_kw NUMERIC(8,2),
    power_used_kw NUMERIC(8,2),
    room_status VARCHAR(20) DEFAULT 'active',
    security_level VARCHAR(20),
    access_hours VARCHAR(100),
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    notes TEXT
);

-- ==========================================
-- RACKS AND MOUNTING
-- ==========================================

CREATE TABLE IF NOT EXISTS fttx.racks (
    rack_id SERIAL PRIMARY KEY,
    rack_code VARCHAR(50) UNIQUE NOT NULL,
    rack_name VARCHAR(200),
    room_id INTEGER REFERENCES fttx.rooms(room_id),
    rack_type VARCHAR(50),
    manufacturer VARCHAR(100),
    model VARCHAR(100),
    height_ru INTEGER,
    width_inches INTEGER DEFAULT 19,
    depth_mm INTEGER,
    max_load_kg NUMERIC(8,2),
    current_load_kg NUMERIC(8,2),
    used_ru INTEGER DEFAULT 0,
    available_ru INTEGER,
    position_x NUMERIC(8,2),
    position_y NUMERIC(8,2),
    rack_status VARCHAR(20) DEFAULT 'active',
    installation_date DATE,
    has_pdu BOOLEAN DEFAULT FALSE,
    pdu_outlets INTEGER,
    has_cooling BOOLEAN DEFAULT FALSE,
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    notes TEXT,
    CONSTRAINT rack_height_check CHECK (height_ru > 0 AND height_ru <= 52),
    CONSTRAINT rack_used_check CHECK (used_ru >= 0 AND used_ru <= height_ru)
);

-- ==========================================
-- EQUIPMENT IN RACKS
-- ==========================================

CREATE TABLE IF NOT EXISTS fttx.rack_equipment (
    equipment_id SERIAL PRIMARY KEY,
    equipment_code VARCHAR(50) UNIQUE NOT NULL,
    equipment_name VARCHAR(200),
    rack_id INTEGER REFERENCES fttx.racks(rack_id),
    equipment_type VARCHAR(50),
    manufacturer VARCHAR(100),
    model VARCHAR(100),
    serial_number VARCHAR(100),
    start_ru INTEGER,
    height_ru INTEGER,
    power_consumption_w NUMERIC(8,2),
    heat_output_btu NUMERIC(8,2),
    weight_kg NUMERIC(8,2),
    equipment_status VARCHAR(20) DEFAULT 'operational',
    installation_date DATE,
    warranty_expiry DATE,
    maintenance_schedule VARCHAR(50),
    last_maintenance_date DATE,
    firmware_version VARCHAR(50),
    ip_address INET,
    management_url TEXT,
    port_count INTEGER,
    used_ports INTEGER DEFAULT 0,
    available_ports INTEGER,
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    notes TEXT,
    CONSTRAINT equipment_height_check CHECK (height_ru > 0 AND height_ru <= 12),
    CONSTRAINT equipment_start_check CHECK (start_ru > 0)
);

-- Create function to validate equipment fits in rack
CREATE OR REPLACE FUNCTION validate_equipment_fits()
RETURNS TRIGGER AS $$
DECLARE
    rack_height INTEGER;
BEGIN
    -- Get rack height
    SELECT height_ru INTO rack_height
    FROM fttx.racks
    WHERE rack_id = NEW.rack_id;
    
    -- Check if equipment fits
    IF NEW.start_ru + NEW.height_ru > rack_height THEN
        RAISE EXCEPTION 'Equipment does not fit: needs RU % to % but rack only has % RU',
            NEW.start_ru, NEW.start_ru + NEW.height_ru - 1, rack_height;
    END IF;
    
    -- Check for overlapping equipment
    IF EXISTS (
        SELECT 1 FROM fttx.rack_equipment
        WHERE rack_id = NEW.rack_id
        AND equipment_id != COALESCE(NEW.equipment_id, -1)
        AND (
            (start_ru BETWEEN NEW.start_ru AND NEW.start_ru + NEW.height_ru - 1)
            OR (start_ru + height_ru - 1 BETWEEN NEW.start_ru AND NEW.start_ru + NEW.height_ru - 1)
            OR (NEW.start_ru BETWEEN start_ru AND start_ru + height_ru - 1)
        )
    ) THEN
        RAISE EXCEPTION 'Equipment overlaps with existing equipment in rack';
    END IF;
    
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER trg_validate_equipment_fits
    BEFORE INSERT OR UPDATE ON fttx.rack_equipment
    FOR EACH ROW
    EXECUTE FUNCTION validate_equipment_fits();

-- ==========================================
-- PORTS AND CONNECTIONS
-- ==========================================

CREATE TABLE IF NOT EXISTS fttx.equipment_ports (
    port_id SERIAL PRIMARY KEY,
    port_code VARCHAR(50) UNIQUE NOT NULL,
    equipment_id INTEGER REFERENCES fttx.rack_equipment(equipment_id),
    port_number INTEGER,
    port_label VARCHAR(100),
    port_type VARCHAR(50),
    port_speed VARCHAR(20),
    connector_type VARCHAR(50),
    port_status VARCHAR(20) DEFAULT 'available',
    connected_to_port_id INTEGER REFERENCES fttx.equipment_ports(port_id),
    cable_id INTEGER REFERENCES fttx.itu_fiber_cables(cable_id),
    wavelength_nm INTEGER,
    signal_type VARCHAR(50),
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    notes TEXT,
    UNIQUE(equipment_id, port_number)
);

-- ==========================================
-- PATCH PANELS AND FIBER DISTRIBUTION
-- ==========================================

CREATE TABLE IF NOT EXISTS fttx.patch_panels (
    panel_id SERIAL PRIMARY KEY,
    panel_code VARCHAR(50) UNIQUE NOT NULL,
    panel_name VARCHAR(200),
    equipment_id INTEGER REFERENCES fttx.rack_equipment(equipment_id),
    panel_type VARCHAR(50),
    port_density INTEGER,
    connector_type VARCHAR(50),
    fiber_type VARCHAR(20),
    front_ports INTEGER,
    rear_ports INTEGER,
    used_ports INTEGER DEFAULT 0,
    panel_status VARCHAR(20) DEFAULT 'active',
    color_code VARCHAR(50),
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    notes TEXT
);

CREATE TABLE IF NOT EXISTS fttx.patch_connections (
    connection_id SERIAL PRIMARY KEY,
    connection_code VARCHAR(50) UNIQUE,
    panel_id INTEGER REFERENCES fttx.patch_panels(panel_id),
    front_port INTEGER,
    rear_port INTEGER,
    cable_id INTEGER REFERENCES fttx.itu_fiber_cables(cable_id),
    fiber_strand_id INTEGER REFERENCES fttx.itu_fiber_strands(strand_id),
    connection_type VARCHAR(50),
    connector_a_type VARCHAR(50),
    connector_b_type VARCHAR(50),
    insertion_loss_db NUMERIC(4,2),
    return_loss_db NUMERIC(4,2),
    test_date DATE,
    connection_status VARCHAR(20) DEFAULT 'active',
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    notes TEXT
);

-- ==========================================
-- CABLE MANAGEMENT
-- ==========================================

CREATE TABLE IF NOT EXISTS fttx.cable_pathways (
    pathway_id SERIAL PRIMARY KEY,
    pathway_code VARCHAR(50) UNIQUE NOT NULL,
    pathway_name VARCHAR(200),
    room_id INTEGER REFERENCES fttx.rooms(room_id),
    pathway_type VARCHAR(50),
    pathway_size VARCHAR(50),
    capacity_percent NUMERIC(5,2) DEFAULT 0,
    pathway_status VARCHAR(20) DEFAULT 'active',
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE IF NOT EXISTS fttx.cable_labels (
    label_id SERIAL PRIMARY KEY,
    cable_id INTEGER REFERENCES fttx.itu_fiber_cables(cable_id),
    label_text VARCHAR(200),
    label_position VARCHAR(50),
    label_type VARCHAR(50),
    barcode VARCHAR(100),
    qr_code TEXT,
    created_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- ==========================================
-- FIBER COLOR CODES (TIA-598-C)
-- ==========================================

CREATE TABLE IF NOT EXISTS fttx.fiber_color_codes (
    color_id SERIAL PRIMARY KEY,
    position INTEGER UNIQUE NOT NULL,
    color_name VARCHAR(50) NOT NULL,
    hex_color VARCHAR(7),
    rgb_color VARCHAR(20),
    tube_position INTEGER,
    standard VARCHAR(50) DEFAULT 'TIA-598-C',
    notes TEXT
);

-- Insert standard fiber color codes
INSERT INTO fttx.fiber_color_codes (position, color_name, hex_color, rgb_color, tube_position) VALUES
(1, 'Blue', '#0000FF', '0,0,255', 1),
(2, 'Orange', '#FF6600', '255,102,0', 1),
(3, 'Green', '#00FF00', '0,255,0', 1),
(4, 'Brown', '#8B4513', '139,69,19', 1),
(5, 'Slate', '#708090', '112,128,144', 1),
(6, 'White', '#FFFFFF', '255,255,255', 1),
(7, 'Red', '#FF0000', '255,0,0', 2),
(8, 'Black', '#000000', '0,0,0', 2),
(9, 'Yellow', '#FFFF00', '255,255,0', 2),
(10, 'Violet', '#8B00FF', '139,0,255', 2),
(11, 'Rose', '#FF007F', '255,0,127', 2),
(12, 'Aqua', '#00FFFF', '0,255,255', 2);

-- ==========================================
-- MAINTENANCE AND TESTING
-- ==========================================

CREATE TABLE IF NOT EXISTS fttx.maintenance_schedules (
    schedule_id SERIAL PRIMARY KEY,
    equipment_id INTEGER REFERENCES fttx.rack_equipment(equipment_id),
    maintenance_type VARCHAR(50),
    frequency_days INTEGER,
    last_maintenance DATE,
    next_maintenance DATE,
    responsible_person VARCHAR(100),
    status VARCHAR(20) DEFAULT 'scheduled',
    notes TEXT
);

CREATE TABLE IF NOT EXISTS fttx.test_results (
    test_id SERIAL PRIMARY KEY,
    test_code VARCHAR(50) UNIQUE,
    cable_id INTEGER REFERENCES fttx.itu_fiber_cables(cable_id),
    fiber_strand_id INTEGER REFERENCES fttx.itu_fiber_strands(strand_id),
    test_type VARCHAR(50),
    test_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    technician_name VARCHAR(100),
    wavelength_nm INTEGER,
    loss_db NUMERIC(6,3),
    return_loss_db NUMERIC(6,2),
    test_equipment VARCHAR(100),
    test_result VARCHAR(20),
    test_certificate_path TEXT,
    notes TEXT
);

-- ==========================================
-- INDEXES
-- ==========================================

CREATE INDEX idx_buildings_geom ON fttx.buildings USING GIST(geom);
CREATE INDEX idx_buildings_code ON fttx.buildings(building_code);
CREATE INDEX idx_rooms_building ON fttx.rooms(building_id);
CREATE INDEX idx_rooms_code ON fttx.rooms(room_code);
CREATE INDEX idx_racks_room ON fttx.racks(room_id);
CREATE INDEX idx_racks_code ON fttx.racks(rack_code);
CREATE INDEX idx_equipment_rack ON fttx.rack_equipment(rack_id);
CREATE INDEX idx_equipment_code ON fttx.rack_equipment(equipment_code);
CREATE INDEX idx_equipment_type ON fttx.rack_equipment(equipment_type);
CREATE INDEX idx_ports_equipment ON fttx.equipment_ports(equipment_id);
CREATE INDEX idx_ports_status ON fttx.equipment_ports(port_status);
CREATE INDEX idx_patch_equipment ON fttx.patch_panels(equipment_id);

-- ==========================================
-- VIEWS FOR QGIS
-- ==========================================

-- View: Available rack space
CREATE OR REPLACE VIEW fttx.v_rack_availability AS
SELECT 
    r.rack_id,
    r.rack_code,
    r.rack_name,
    rm.room_code,
    rm.room_name,
    b.building_code,
    r.height_ru,
    r.used_ru,
    r.available_ru,
    ROUND((r.used_ru::NUMERIC / NULLIF(r.height_ru, 0) * 100), 1) as utilization_percent,
    CASE 
        WHEN r.used_ru::NUMERIC / NULLIF(r.height_ru, 0) >= 0.9 THEN 'Critical'
        WHEN r.used_ru::NUMERIC / NULLIF(r.height_ru, 0) >= 0.7 THEN 'High'
        WHEN r.used_ru::NUMERIC / NULLIF(r.height_ru, 0) >= 0.5 THEN 'Medium'
        ELSE 'Low'
    END as utilization_status,
    r.rack_status
FROM fttx.racks r
LEFT JOIN fttx.rooms rm ON r.room_id = rm.room_id
LEFT JOIN fttx.buildings b ON rm.building_id = b.building_id
WHERE r.rack_status = 'active';

-- View: Equipment inventory
CREATE OR REPLACE VIEW fttx.v_equipment_inventory AS
SELECT 
    e.equipment_id,
    e.equipment_code,
    e.equipment_name,
    e.equipment_type,
    r.rack_code,
    rm.room_code,
    b.building_code,
    e.manufacturer,
    e.model,
    e.serial_number,
    e.start_ru,
    e.height_ru,
    e.equipment_status,
    e.port_count,
    e.used_ports,
    e.available_ports,
    ROUND((e.used_ports::NUMERIC / NULLIF(e.port_count, 0) * 100), 1) as port_utilization_percent
FROM fttx.rack_equipment e
LEFT JOIN fttx.racks r ON e.rack_id = r.rack_id
LEFT JOIN fttx.rooms rm ON r.room_id = rm.room_id
LEFT JOIN fttx.buildings b ON rm.building_id = b.building_id
ORDER BY b.building_code, rm.room_code, r.rack_code, e.start_ru;

-- View: Port availability
CREATE OR REPLACE VIEW fttx.v_port_availability AS
SELECT 
    e.equipment_id,
    e.equipment_code,
    e.equipment_name,
    e.equipment_type,
    COUNT(p.port_id) as total_ports,
    SUM(CASE WHEN p.port_status = 'available' THEN 1 ELSE 0 END) as available_ports,
    SUM(CASE WHEN p.port_status = 'in_use' THEN 1 ELSE 0 END) as used_ports,
    SUM(CASE WHEN p.port_status = 'reserved' THEN 1 ELSE 0 END) as reserved_ports,
    SUM(CASE WHEN p.port_status = 'faulty' THEN 1 ELSE 0 END) as faulty_ports,
    ROUND((SUM(CASE WHEN p.port_status = 'in_use' THEN 1 ELSE 0 END)::NUMERIC / NULLIF(COUNT(p.port_id), 0) * 100), 1) as utilization_percent
FROM fttx.rack_equipment e
LEFT JOIN fttx.equipment_ports p ON e.equipment_id = p.equipment_id
GROUP BY e.equipment_id, e.equipment_code, e.equipment_name, e.equipment_type;

-- View: Room capacity summary
CREATE OR REPLACE VIEW fttx.v_room_capacity AS
SELECT 
    rm.room_id,
    rm.room_code,
    rm.room_name,
    b.building_code,
    rm.room_type,
    rm.room_size_sqm,
    COUNT(DISTINCT r.rack_id) as rack_count,
    SUM(r.height_ru) as total_ru,
    SUM(r.used_ru) as used_ru,
    SUM(r.available_ru) as available_ru,
    ROUND((SUM(r.used_ru)::NUMERIC / NULLIF(SUM(r.height_ru), 0) * 100), 1) as room_utilization_percent,
    SUM(e.power_consumption_w) / 1000.0 as total_power_kw,
    rm.power_capacity_kw,
    ROUND(((SUM(e.power_consumption_w) / 1000.0) / NULLIF(rm.power_capacity_kw, 0) * 100), 1) as power_utilization_percent
FROM fttx.rooms rm
LEFT JOIN fttx.buildings b ON rm.building_id = b.building_id
LEFT JOIN fttx.racks r ON rm.room_id = r.room_id AND r.rack_status = 'active'
LEFT JOIN fttx.rack_equipment e ON r.rack_id = e.rack_id AND e.equipment_status = 'operational'
GROUP BY rm.room_id, rm.room_code, rm.room_name, b.building_code, 
         rm.room_type, rm.room_size_sqm, rm.power_capacity_kw;

-- ==========================================
-- TRIGGERS
-- ==========================================

-- Update rack utilization
CREATE OR REPLACE FUNCTION update_rack_utilization()
RETURNS TRIGGER AS $$
BEGIN
    IF TG_OP = 'INSERT' THEN
        UPDATE fttx.racks
        SET used_ru = COALESCE(used_ru, 0) + NEW.height_ru,
            available_ru = height_ru - (COALESCE(used_ru, 0) + NEW.height_ru)
        WHERE rack_id = NEW.rack_id;
    ELSIF TG_OP = 'DELETE' THEN
        UPDATE fttx.racks
        SET used_ru = GREATEST(COALESCE(used_ru, 0) - OLD.height_ru, 0),
            available_ru = height_ru - GREATEST(COALESCE(used_ru, 0) - OLD.height_ru, 0)
        WHERE rack_id = OLD.rack_id;
        RETURN OLD;
    ELSIF TG_OP = 'UPDATE' THEN
        -- If moved to different rack
        IF OLD.rack_id != NEW.rack_id THEN
            UPDATE fttx.racks
            SET used_ru = GREATEST(COALESCE(used_ru, 0) - OLD.height_ru, 0),
                available_ru = height_ru - GREATEST(COALESCE(used_ru, 0) - OLD.height_ru, 0)
            WHERE rack_id = OLD.rack_id;
            
            UPDATE fttx.racks
            SET used_ru = COALESCE(used_ru, 0) + NEW.height_ru,
                available_ru = height_ru - (COALESCE(used_ru, 0) + NEW.height_ru)
            WHERE rack_id = NEW.rack_id;
        -- If height changed
        ELSIF OLD.height_ru != NEW.height_ru THEN
            UPDATE fttx.racks
            SET used_ru = COALESCE(used_ru, 0) - OLD.height_ru + NEW.height_ru,
                available_ru = height_ru - (COALESCE(used_ru, 0) - OLD.height_ru + NEW.height_ru)
            WHERE rack_id = NEW.rack_id;
        END IF;
    END IF;
    
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER trg_update_rack_utilization
    AFTER INSERT OR UPDATE OR DELETE ON fttx.rack_equipment
    FOR EACH ROW
    EXECUTE FUNCTION update_rack_utilization();

-- Update equipment port count
CREATE OR REPLACE FUNCTION update_equipment_ports()
RETURNS TRIGGER AS $$
BEGIN
    IF TG_OP = 'INSERT' THEN
        IF NEW.port_status = 'in_use' THEN
            UPDATE fttx.rack_equipment
            SET used_ports = COALESCE(used_ports, 0) + 1,
                available_ports = GREATEST(COALESCE(port_count, 0) - COALESCE(used_ports, 0) - 1, 0)
            WHERE equipment_id = NEW.equipment_id;
        END IF;
    ELSIF TG_OP = 'DELETE' THEN
        IF OLD.port_status = 'in_use' THEN
            UPDATE fttx.rack_equipment
            SET used_ports = GREATEST(COALESCE(used_ports, 0) - 1, 0),
                available_ports = COALESCE(port_count, 0) - GREATEST(COALESCE(used_ports, 0) - 1, 0)
            WHERE equipment_id = OLD.equipment_id;
        END IF;
        RETURN OLD;
    ELSIF TG_OP = 'UPDATE' AND OLD.port_status != NEW.port_status THEN
        IF NEW.port_status = 'in_use' AND OLD.port_status != 'in_use' THEN
            UPDATE fttx.rack_equipment
            SET used_ports = COALESCE(used_ports, 0) + 1,
                available_ports = GREATEST(COALESCE(port_count, 0) - COALESCE(used_ports, 0) - 1, 0)
            WHERE equipment_id = NEW.equipment_id;
        ELSIF OLD.port_status = 'in_use' AND NEW.port_status != 'in_use' THEN
            UPDATE fttx.rack_equipment
            SET used_ports = GREATEST(COALESCE(used_ports, 0) - 1, 0),
                available_ports = COALESCE(port_count, 0) - GREATEST(COALESCE(used_ports, 0) - 1, 0)
            WHERE equipment_id = NEW.equipment_id;
        END IF;
    END IF;
    
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER trg_update_equipment_ports
    AFTER INSERT OR UPDATE OR DELETE ON fttx.equipment_ports
    FOR EACH ROW
    EXECUTE FUNCTION update_equipment_ports();

-- Grant permissions
GRANT ALL ON ALL TABLES IN SCHEMA fttx TO PUBLIC;
GRANT USAGE, SELECT ON ALL SEQUENCES IN SCHEMA fttx TO PUBLIC;

-- Success message
DO $$ 
BEGIN
    RAISE NOTICE '=================================================';
    RAISE NOTICE '✓ Inside Plant tables created successfully!';
    RAISE NOTICE '=================================================';
    RAISE NOTICE 'Created tables:';
    RAISE NOTICE '  - buildings (with geometry)';
    RAISE NOTICE '  - rooms';
    RAISE NOTICE '  - racks (with utilization tracking)';
    RAISE NOTICE '  - rack_equipment (with overlap validation)';
    RAISE NOTICE '  - equipment_ports';
    RAISE NOTICE '  - patch_panels';
    RAISE NOTICE '  - patch_connections';
    RAISE NOTICE '  - cable_pathways';
    RAISE NOTICE '  - cable_labels';
    RAISE NOTICE '  - fiber_color_codes (TIA-598-C)';
    RAISE NOTICE '  - maintenance_schedules';
    RAISE NOTICE '  - test_results';
    RAISE NOTICE '';
    RAISE NOTICE 'Created views:';
    RAISE NOTICE '  - v_rack_availability';
    RAISE NOTICE '  - v_equipment_inventory';
    RAISE NOTICE '  - v_port_availability';
    RAISE NOTICE '  - v_room_capacity';
    RAISE NOTICE '';
    RAISE NOTICE 'Created triggers:';
    RAISE NOTICE '  - Automatic rack utilization tracking';
    RAISE NOTICE '  - Equipment fit validation';
    RAISE NOTICE '  - Port count tracking';
    RAISE NOTICE '=================================================';
END $$;