-- ==========================================
-- PON PORTS SPATIAL LAYER
-- Makes PON ports visible and editable in QGIS
-- ==========================================

SET search_path TO fttx, public;

-- Create a spatial view of PON ports
-- They inherit the geometry from their parent OLT
CREATE OR REPLACE VIEW v_pon_ports_spatial AS
SELECT 
    p.pon_port_id,
    p.olt_id,
    p.port_number,
    p.port_label,
    p.optical_budget_class,
    p.max_logical_reach_km,
    p.max_differential_fiber_distance_km,
    p.measured_power_output_dbm,
    p.max_onus,
    p.connected_onus,
    p.port_status,
    o.olt_code,
    o.geom,
    o.olt_code || ' - Port ' || p.port_number || ' (' || 
    (p.max_onus - p.connected_onus) || ' free)' as display_label
FROM fttx.itu_olt_pon_ports p
JOIN fttx.itu_olt_equipment o ON p.olt_id = o.olt_id;

-- Grant permissions
GRANT SELECT ON fttx.v_pon_ports_spatial TO PUBLIC;

-- Create function to insert PON ports through the view
CREATE OR REPLACE FUNCTION insert_pon_port()
RETURNS TRIGGER AS $$
BEGIN
    -- Insert into actual table
    INSERT INTO fttx.itu_olt_pon_ports (
        olt_id,
        port_number,
        port_label,
        optical_budget_class,
        max_logical_reach_km,
        max_differential_fiber_distance_km,
        measured_power_output_dbm,
        max_onus,
        connected_onus,
        port_status
    ) VALUES (
        NEW.olt_id,
        NEW.port_number,
        NEW.port_label,
        NEW.optical_budget_class,
        NEW.max_logical_reach_km,
        NEW.max_differential_fiber_distance_km,
        NEW.measured_power_output_dbm,
        NEW.max_onus,
        COALESCE(NEW.connected_onus, 0),
        COALESCE(NEW.port_status, 'inactive')
    );
    
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Create trigger to handle inserts
CREATE TRIGGER trg_insert_pon_port
    INSTEAD OF INSERT ON v_pon_ports_spatial
    FOR EACH ROW
    EXECUTE FUNCTION insert_pon_port();

-- Create function to update PON ports through the view
CREATE OR REPLACE FUNCTION update_pon_port()
RETURNS TRIGGER AS $$
BEGIN
    UPDATE fttx.itu_olt_pon_ports
    SET 
        port_number = NEW.port_number,
        port_label = NEW.port_label,
        optical_budget_class = NEW.optical_budget_class,
        max_logical_reach_km = NEW.max_logical_reach_km,
        max_differential_fiber_distance_km = NEW.max_differential_fiber_distance_km,
        measured_power_output_dbm = NEW.measured_power_output_dbm,
        max_onus = NEW.max_onus,
        port_status = NEW.port_status
    WHERE pon_port_id = OLD.pon_port_id;
    
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Create trigger to handle updates
CREATE TRIGGER trg_update_pon_port
    INSTEAD OF UPDATE ON v_pon_ports_spatial
    FOR EACH ROW
    EXECUTE FUNCTION update_pon_port();

-- Create function to delete PON ports through the view
CREATE OR REPLACE FUNCTION delete_pon_port()
RETURNS TRIGGER AS $$
BEGIN
    -- Check if port has connected ONUs
    IF OLD.connected_onus > 0 THEN
        RAISE EXCEPTION 'Cannot delete PON port with % connected ONUs. Disconnect ONUs first.', OLD.connected_onus;
    END IF;
    
    DELETE FROM fttx.itu_olt_pon_ports
    WHERE pon_port_id = OLD.pon_port_id;
    
    RETURN OLD;
END;
$$ LANGUAGE plpgsql;

-- Create trigger to handle deletes
CREATE TRIGGER trg_delete_pon_port
    INSTEAD OF DELETE ON v_pon_ports_spatial
    FOR EACH ROW
    EXECUTE FUNCTION delete_pon_port();

-- Add view for optical budget classes (dropdown)
CREATE OR REPLACE VIEW v_optical_budget_classes AS
SELECT 
    class as id,
    class || ' (' || min_loss || '-' || max_loss || ' dB, ' || typical_reach_km || ' km)' as display_name,
    min_loss,
    max_loss,
    typical_reach_km
FROM (VALUES 
    ('A', 5, 20, 10),
    ('B', 10, 25, 15),
    ('B+', 13, 28, 20),
    ('C', 15, 30, 20),
    ('C+', 17, 32, 20),
    ('C++', 19, 34, 40)
) AS classes(class, min_loss, max_loss, typical_reach_km)
ORDER BY max_loss;

-- Add view for port status (dropdown)
CREATE OR REPLACE VIEW v_port_status AS
SELECT status as id, status as display_name
FROM (VALUES 
    ('active'),
    ('inactive'),
    ('maintenance'),
    ('failed')
) AS statuses(status)
ORDER BY status;

-- Success message
DO $$ 
BEGIN
    RAISE NOTICE '✓ PON Ports layer created successfully!';
    RAISE NOTICE '  You can now add PON ports directly in QGIS';
END $$;