-- ==========================================
-- WORKFLOW ENFORCEMENT RULES
-- These ensure proper order of operations
-- ==========================================

SET search_path TO fttx, public;

-- Rule 1: Cables must have routes (except short drops)
CREATE OR REPLACE FUNCTION enforce_cable_route()
RETURNS TRIGGER AS $$
BEGIN
    -- Allow drops under 200m without route
    IF NEW.cable_type = 'drop' AND (NEW.cable_length_m IS NULL OR NEW.cable_length_m <= 200) THEN
        RETURN NEW;
    END IF;
    
    -- All other cables need routes
    IF NEW.route_id IS NULL THEN
        RAISE EXCEPTION 'Cable % needs a route. Create route first!', NEW.cable_code
        USING HINT = 'Go to Routes layer and draw a route first';
    END IF;
    
    -- Check route exists
    IF NOT EXISTS (SELECT 1 FROM itu_cable_routes WHERE route_id = NEW.route_id) THEN
        RAISE EXCEPTION 'Route ID % does not exist', NEW.route_id;
    END IF;
    
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER trg_enforce_cable_route
    BEFORE INSERT OR UPDATE ON itu_fiber_cables
    FOR EACH ROW
    EXECUTE FUNCTION enforce_cable_route();

-- Rule 2: ONUs must have PON port
CREATE OR REPLACE FUNCTION enforce_onu_port()
RETURNS TRIGGER AS $$
BEGIN
    IF NEW.connected_pon_port_id IS NULL THEN
        RAISE EXCEPTION 'ONU % must be connected to a PON port', NEW.onu_serial_number
        USING HINT = 'Select a PON port from the dropdown';
    END IF;
    
    -- Check port exists and is active
    IF NOT EXISTS (
        SELECT 1 FROM itu_olt_pon_ports 
        WHERE pon_port_id = NEW.connected_pon_port_id 
        AND port_status = 'active'
    ) THEN
        RAISE EXCEPTION 'PON port is not active or does not exist';
    END IF;
    
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER trg_enforce_onu_port
    BEFORE INSERT OR UPDATE ON itu_onu_equipment
    FOR EACH ROW
    EXECUTE FUNCTION enforce_onu_port();

-- Rule 3: Update splitter capacity
CREATE OR REPLACE FUNCTION update_splitter_usage()
RETURNS TRIGGER AS $$
BEGIN
    IF TG_OP = 'INSERT' AND NEW.splitter_node_id IS NOT NULL THEN
        UPDATE itu_odn_nodes
        SET used_output_ports = used_output_ports + 1
        WHERE node_id = NEW.splitter_node_id;
    ELSIF TG_OP = 'DELETE' AND OLD.splitter_node_id IS NOT NULL THEN
        UPDATE itu_odn_nodes
        SET used_output_ports = GREATEST(used_output_ports - 1, 0)
        WHERE node_id = OLD.splitter_node_id;
        RETURN OLD;
    END IF;
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER trg_update_splitter
    AFTER INSERT OR DELETE ON itu_onu_equipment
    FOR EACH ROW
    EXECUTE FUNCTION update_splitter_usage();

-- Rule 4: Update PON port capacity
CREATE OR REPLACE FUNCTION update_port_usage()
RETURNS TRIGGER AS $$
BEGIN
    IF TG_OP = 'INSERT' AND NEW.onu_status = 'online' THEN
        UPDATE itu_olt_pon_ports
        SET connected_onus = connected_onus + 1
        WHERE pon_port_id = NEW.connected_pon_port_id;
    ELSIF TG_OP = 'DELETE' AND OLD.onu_status = 'online' THEN
        UPDATE itu_olt_pon_ports
        SET connected_onus = GREATEST(connected_onus - 1, 0)
        WHERE pon_port_id = OLD.connected_pon_port_id;
        RETURN OLD;
    END IF;
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER trg_update_port
    AFTER INSERT OR DELETE ON itu_onu_equipment
    FOR EACH ROW
    EXECUTE FUNCTION update_port_usage();

-- Success Message
DO $$ 
BEGIN
    RAISE NOTICE '✓ Workflow rules created successfully!';
    RAISE NOTICE '  The system will now enforce proper order of operations';
END $$;