-- Function to calculate ITU-T G.984.2 compliant power budget
CREATE OR REPLACE FUNCTION calculate_itu_power_budget(
    p_olt_port_id INTEGER,
    p_onu_id INTEGER
) RETURNS TABLE (
    downstream_budget_db NUMERIC,
    upstream_budget_db NUMERIC,
    meets_class_b_plus BOOLEAN,
    meets_class_c_plus BOOLEAN
) AS $$
DECLARE
    v_olt_tx_power NUMERIC;
    v_onu_rx_sensitivity NUMERIC;
    v_onu_tx_power NUMERIC;
    v_olt_rx_sensitivity NUMERIC;
    v_total_loss NUMERIC;
BEGIN
    -- Get OLT specifications
    SELECT mean_launched_power_min_dbm INTO v_olt_tx_power
    FROM itu_olt_equipment olt
    JOIN itu_olt_pon_ports port ON olt.olt_id = port.olt_id
    WHERE port.pon_port_id = p_olt_port_id;
    
    -- Get ONU specifications
    SELECT onu_rx_sensitivity_dbm, onu_tx_power_min_dbm,
           onu_rx_overload_dbm
    INTO v_onu_rx_sensitivity, v_onu_tx_power
    FROM itu_onu_equipment
    WHERE onu_id = p_onu_id;
    
    -- Calculate total path loss from network path
    SELECT total_path_loss_db INTO v_total_loss
    FROM itu_network_paths
    WHERE olt_port_id = p_olt_port_id 
    AND onu_id = p_onu_id;
    
    -- Calculate budgets
    downstream_budget_db := v_olt_tx_power - v_onu_rx_sensitivity - v_total_loss;
    upstream_budget_db := v_onu_tx_power - v_olt_rx_sensitivity - v_total_loss;
    
    -- ITU-T G.984.2 Class B+ requires 13-28 dB budget
    meets_class_b_plus := (downstream_budget_db >= 3 AND upstream_budget_db >= 3);
    
    -- ITU-T G.984.2 Class C+ requires 17-32 dB budget
    meets_class_c_plus := (downstream_budget_db >= 5 AND upstream_budget_db >= 5);
    
    RETURN NEXT;
END;
$$ LANGUAGE plpgsql;

-- Function to validate fiber type compatibility per ITU-T
CREATE OR REPLACE FUNCTION validate_fiber_compatibility(
    p_strand_id INTEGER
) RETURNS BOOLEAN AS $$
DECLARE
    v_fiber_category VARCHAR;
    v_pon_standard VARCHAR;
    v_wavelength_1310 BOOLEAN;
    v_wavelength_1490 BOOLEAN;
    v_wavelength_1550 BOOLEAN;
BEGIN
    SELECT fiber_category INTO v_fiber_category
    FROM itu_fiber_strands
    WHERE strand_id = p_strand_id;
    
    -- ITU-T G.652.D is recommended for GPON (1310/1490/1550nm)
    -- ITU-T G.657.A2 is suitable for bend-insensitive applications
    -- ITU-T G.657.B3 is for indoor/building applications
    
    IF v_fiber_category IN ('G.652.D', 'G.657.A2', 'G.657.B3') THEN
        RETURN TRUE;
    ELSE
        RETURN FALSE;
    END IF;
END;
$$ LANGUAGE plpgsql;

-- Function to trace complete path from OLT to ONU
CREATE OR REPLACE FUNCTION trace_itu_network_path(
    p_olt_port_id INTEGER,
    p_onu_id INTEGER
) RETURNS TABLE (
    segment_order INTEGER,
    segment_type VARCHAR,
    segment_id INTEGER,
    segment_name VARCHAR,
    segment_loss_db NUMERIC,
    cumulative_loss_db NUMERIC
) AS $$
-- Implementation using pgRouting for network topology traversal
-- Returns ordered list of network elements from OLT to ONU
BEGIN
    -- This would use pgRouting to traverse the network topology
    -- Following the physical cable paths, splices, and splitters
    RETURN QUERY
    WITH RECURSIVE path AS (
        -- Start from OLT port
        SELECT 1 as depth, pon_port_id, 0.0 as loss
        FROM itu_olt_pon_ports
        WHERE pon_port_id = p_olt_port_id
        
        UNION ALL
        
        -- Traverse through cables, nodes, splices to ONU
        -- (Simplified - full implementation would be more complex)
        SELECT depth + 1, next_element, loss + segment_loss
        FROM path p
        -- JOIN logic to follow network topology
    )
    SELECT * FROM path;
END;
$$ LANGUAGE plpgsql;