-- ==========================================
-- ITU-T COMPLIANT FTTX DATABASE SCHEMA
-- PostgreSQL 15+ with PostGIS 3.3+
-- Version: 1.0.0
-- Date: 2025-01-17
-- ==========================================

-- Create schema
CREATE SCHEMA IF NOT EXISTS fttx;

-- Set search path
SET search_path TO fttx, public;

-- Enable PostGIS extension
CREATE EXTENSION IF NOT EXISTS postgis;

-- Try to enable pgRouting (optional)
DO $$ 
BEGIN
    CREATE EXTENSION IF NOT EXISTS pgrouting;
    RAISE NOTICE 'pgRouting extension enabled successfully';
EXCEPTION 
    WHEN OTHERS THEN
        RAISE NOTICE 'pgRouting not available - network routing functions will be limited';
END $$;

-- ==========================================
-- CORE TABLES - ITU-T COMPLIANT
-- ==========================================

-- ITU-T L.40 Cable Route Infrastructure
CREATE TABLE IF NOT EXISTS fttx.itu_cable_routes (
    route_id SERIAL PRIMARY KEY,
    route_code VARCHAR(50) UNIQUE NOT NULL,
    route_name VARCHAR(200),
    infrastructure_type VARCHAR(30),
    duct_specification VARCHAR(50),
    duct_count INTEGER,
    duct_occupancy_percent NUMERIC(5,2) DEFAULT 0,
    route_length_m NUMERIC(10,2),
    access_difficulty VARCHAR(20),
    access_points TEXT[],
    row_owner VARCHAR(100),
    row_type VARCHAR(30),
    permit_required BOOLEAN,
    permit_number VARCHAR(50),
    route_status VARCHAR(20) DEFAULT 'planned',
    construction_date DATE,
    geom GEOMETRY(LINESTRING, 4326)
);

-- ITU-T Compliant Fiber Cable Infrastructure
CREATE TABLE IF NOT EXISTS fttx.itu_fiber_cables (
    cable_id SERIAL PRIMARY KEY,
    cable_code VARCHAR(50) UNIQUE NOT NULL,
    cable_category VARCHAR(20),
    cable_type VARCHAR(30),
    fiber_type VARCHAR(10),
    fiber_count INTEGER,
    cable_construction VARCHAR(30),
    sheath_material VARCHAR(20),
    armoring VARCHAR(20),
    cable_diameter_mm NUMERIC(6,2),
    cable_weight_kg_per_km NUMERIC(8,2),
    min_bend_radius_mm NUMERIC(6,1),
    max_tensile_load_n NUMERIC(8,1),
    temperature_range_min_c INTEGER,
    temperature_range_max_c INTEGER,
    max_attenuation_db_per_km NUMERIC(4,3),
    chromatic_dispersion_ps_nm_km NUMERIC(6,2),
    polarization_mode_dispersion_ps NUMERIC(5,2),
    installation_method VARCHAR(30),
    installation_date DATE,
    installation_contractor VARCHAR(100),
    operational_status VARCHAR(20) DEFAULT 'planned',
    capacity_used INTEGER DEFAULT 0,
    owner_organization VARCHAR(100),
    cable_length_m NUMERIC(10,2),
    geom GEOMETRY(LINESTRING, 4326),
    parent_cable_id INTEGER REFERENCES fttx.itu_fiber_cables(cable_id),
    route_id INTEGER REFERENCES fttx.itu_cable_routes(route_id),
    CONSTRAINT fiber_capacity_check CHECK (capacity_used <= fiber_count)
);

-- ITU-T Individual Fiber Strands
CREATE TABLE IF NOT EXISTS fttx.itu_fiber_strands (
    strand_id SERIAL PRIMARY KEY,
    cable_id INTEGER REFERENCES fttx.itu_fiber_cables(cable_id) ON DELETE CASCADE,
    fiber_position INTEGER,
    fiber_color_code VARCHAR(20),
    tube_color_code VARCHAR(20),
    fiber_category VARCHAR(10),
    attenuation_1310nm_db_per_km NUMERIC(4,3),
    attenuation_1550nm_db_per_km NUMERIC(4,3),
    attenuation_1625nm_db_per_km NUMERIC(4,3),
    chromatic_dispersion_ps_nm_km NUMERIC(6,2),
    pmd_coefficient_ps_per_sqrt_km NUMERIC(5,3),
    zero_dispersion_wavelength_nm INTEGER,
    zero_dispersion_slope NUMERIC(6,4),
    macrobend_loss_1550nm_db NUMERIC(4,3),
    macrobend_loss_1625nm_db NUMERIC(4,3),
    strand_status VARCHAR(20) DEFAULT 'available',
    wavelength_assignment VARCHAR(50),
    service_assignment_id INTEGER,
    a_end_port_id INTEGER,
    z_end_port_id INTEGER,
    total_path_loss_db NUMERIC(6,2),
    UNIQUE(cable_id, fiber_position)
);

-- ITU-T G.984/G.987/G.989 Compliant OLT
CREATE TABLE IF NOT EXISTS fttx.itu_olt_equipment (
    olt_id SERIAL PRIMARY KEY,
    olt_code VARCHAR(50) UNIQUE NOT NULL,
    pon_standard VARCHAR(20),
    pon_class VARCHAR(10),
    downstream_wavelength_nm INTEGER,
    upstream_wavelength_nm INTEGER,
    downstream_rate_mbps INTEGER,
    upstream_rate_mbps INTEGER,
    mean_launched_power_min_dbm NUMERIC(5,2),
    mean_launched_power_max_dbm NUMERIC(5,2),
    extinction_ratio_db NUMERIC(4,1),
    receiver_sensitivity_dbm NUMERIC(5,2),
    receiver_overload_dbm NUMERIC(5,2),
    max_logical_links INTEGER,
    max_onus_supported INTEGER,
    current_onus_connected INTEGER DEFAULT 0,
    site_id INTEGER,
    rack_position VARCHAR(20),
    installation_date DATE,
    equipment_status VARCHAR(20) DEFAULT 'planned',
    geom GEOMETRY(POINT, 4326),
    manufacturer VARCHAR(100),
    model VARCHAR(100),
    firmware_version VARCHAR(50)
);

-- ITU-T PON Ports on OLT
CREATE TABLE IF NOT EXISTS fttx.itu_olt_pon_ports (
    pon_port_id SERIAL PRIMARY KEY,
    olt_id INTEGER REFERENCES fttx.itu_olt_equipment(olt_id) ON DELETE CASCADE,
    port_number INTEGER,
    port_label VARCHAR(50),
    optical_budget_class VARCHAR(5),
    max_logical_reach_km NUMERIC(5,2),
    max_differential_fiber_distance_km NUMERIC(4,1),
    measured_power_output_dbm NUMERIC(5,2),
    max_onus INTEGER DEFAULT 64,
    connected_onus INTEGER DEFAULT 0,
    port_status VARCHAR(20) DEFAULT 'inactive',
    UNIQUE(olt_id, port_number)
);

-- ITU-T G.984/G.987 Compliant ODN Nodes
CREATE TABLE IF NOT EXISTS fttx.itu_odn_nodes (
    node_id SERIAL PRIMARY KEY,
    node_code VARCHAR(50) UNIQUE NOT NULL,
    node_type VARCHAR(30),
    node_function VARCHAR(30),
    splitter_type VARCHAR(20),
    split_ratio VARCHAR(10),
    split_configuration VARCHAR(20),
    insertion_loss_typical_db NUMERIC(4,2),
    insertion_loss_max_db NUMERIC(4,2),
    uniformity_db NUMERIC(3,2),
    directivity_db NUMERIC(4,1),
    polarization_dependent_loss_db NUMERIC(3,2),
    wavelength_range_min_nm INTEGER,
    wavelength_range_max_nm INTEGER,
    input_ports INTEGER DEFAULT 1,
    output_ports INTEGER,
    used_output_ports INTEGER DEFAULT 0,
    installation_type VARCHAR(30),
    cabinet_type VARCHAR(50),
    ingress_protection_rating VARCHAR(10),
    installation_date DATE,
    operating_temp_min_c INTEGER,
    operating_temp_max_c INTEGER,
    address TEXT,
    elevation_m NUMERIC(8,2),
    geom GEOMETRY(POINT, 4326),
    parent_node_id INTEGER REFERENCES fttx.itu_odn_nodes(node_id),
    olt_id INTEGER REFERENCES fttx.itu_olt_equipment(olt_id),
    operational_status VARCHAR(20) DEFAULT 'planned',
    owner_organization VARCHAR(100)
);

-- ITU-T G.984.2 Compliant ONU/ONT
CREATE TABLE IF NOT EXISTS fttx.itu_onu_equipment (
    onu_id SERIAL PRIMARY KEY,
    onu_serial_number VARCHAR(50) UNIQUE NOT NULL,
    pon_standard VARCHAR(20),
    onu_class VARCHAR(10),
    upstream_wavelength_nm INTEGER,
    downstream_wavelength_nm INTEGER,
    onu_tx_power_min_dbm NUMERIC(5,2),
    onu_tx_power_max_dbm NUMERIC(5,2),
    onu_extinction_ratio_db NUMERIC(4,1),
    onu_rx_sensitivity_dbm NUMERIC(5,2),
    onu_rx_overload_dbm NUMERIC(5,2),
    gem_ports INTEGER,
    ethernet_ports INTEGER,
    pots_ports INTEGER,
    rf_video_support BOOLEAN,
    connected_olt_id INTEGER REFERENCES fttx.itu_olt_equipment(olt_id),
    connected_pon_port_id INTEGER REFERENCES fttx.itu_olt_pon_ports(pon_port_id),
    splitter_node_id INTEGER REFERENCES fttx.itu_odn_nodes(node_id),
    drop_fiber_id INTEGER REFERENCES fttx.itu_fiber_strands(strand_id),
    measured_rx_power_dbm NUMERIC(5,2),
    measured_tx_power_dbm NUMERIC(5,2),
    customer_id INTEGER,
    service_id INTEGER,
    installation_address TEXT,
    installation_date DATE,
    activation_date DATE,
    onu_status VARCHAR(20) DEFAULT 'planned',
    geom GEOMETRY(POINT, 4326),
    manufacturer VARCHAR(100),
    model VARCHAR(100),
    firmware_version VARCHAR(50)
);

-- ITU-T G.671 Compliant Splice Points
CREATE TABLE IF NOT EXISTS fttx.itu_splice_points (
    splice_id SERIAL PRIMARY KEY,
    splice_code VARCHAR(50) UNIQUE,
    node_id INTEGER REFERENCES fttx.itu_odn_nodes(node_id),
    geom GEOMETRY(POINT, 4326),
    splice_type VARCHAR(30),
    splice_method VARCHAR(30),
    typical_splice_loss_db NUMERIC(4,3),
    max_splice_loss_db NUMERIC(4,3),
    return_loss_db NUMERIC(4,1),
    splice_date DATE,
    technician_name VARCHAR(100),
    splice_machine_model VARCHAR(100),
    closure_type VARCHAR(50),
    sealing_method VARCHAR(30),
    otdr_tested BOOLEAN DEFAULT FALSE,
    test_wavelength_nm INTEGER,
    measured_loss_db NUMERIC(4,3),
    test_date DATE,
    test_result VARCHAR(20)
);

-- ITU-T Splice Connections
CREATE TABLE IF NOT EXISTS fttx.itu_splice_connections (
    connection_id SERIAL PRIMARY KEY,
    splice_id INTEGER REFERENCES fttx.itu_splice_points(splice_id) ON DELETE CASCADE,
    input_strand_id INTEGER REFERENCES fttx.itu_fiber_strands(strand_id),
    output_strand_id INTEGER REFERENCES fttx.itu_fiber_strands(strand_id),
    insertion_loss_1310nm_db NUMERIC(4,3),
    insertion_loss_1550nm_db NUMERIC(4,3),
    insertion_loss_1625nm_db NUMERIC(4,3),
    return_loss_db NUMERIC(4,1),
    connection_quality VARCHAR(20),
    UNIQUE(splice_id, input_strand_id, output_strand_id)
);

-- ITU-T G.984.2 Optical Power Budget Calculations
CREATE TABLE IF NOT EXISTS fttx.itu_power_budget (
    budget_id SERIAL PRIMARY KEY,
    olt_port_id INTEGER REFERENCES fttx.itu_olt_pon_ports(pon_port_id),
    onu_id INTEGER REFERENCES fttx.itu_onu_equipment(onu_id),
    power_budget_class VARCHAR(5),
    olt_tx_power_dbm NUMERIC(5,2),
    downstream_fiber_loss_db NUMERIC(5,2),
    downstream_splice_loss_db NUMERIC(5,2),
    downstream_connector_loss_db NUMERIC(5,2),
    downstream_splitter_loss_db NUMERIC(5,2),
    onu_rx_power_dbm NUMERIC(5,2),
    downstream_margin_db NUMERIC(5,2),
    onu_tx_power_dbm NUMERIC(5,2),
    upstream_fiber_loss_db NUMERIC(5,2),
    upstream_splice_loss_db NUMERIC(5,2),
    upstream_connector_loss_db NUMERIC(5,2),
    upstream_splitter_loss_db NUMERIC(5,2),
    olt_rx_power_dbm NUMERIC(5,2),
    upstream_margin_db NUMERIC(5,2),
    meets_itu_requirements BOOLEAN,
    limiting_factor VARCHAR(50),
    calculated_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    calculation_method VARCHAR(50)
);

-- ITU-T Network Topology and Routing
CREATE TABLE IF NOT EXISTS fttx.itu_network_paths (
    path_id SERIAL PRIMARY KEY,
    path_code VARCHAR(50) UNIQUE,
    olt_port_id INTEGER REFERENCES fttx.itu_olt_pon_ports(pon_port_id),
    onu_id INTEGER REFERENCES fttx.itu_onu_equipment(onu_id),
    cable_sequence INTEGER[],
    node_sequence INTEGER[],
    splice_sequence INTEGER[],
    strand_sequence INTEGER[],
    total_path_length_km NUMERIC(8,3),
    total_fiber_loss_1310nm_db NUMERIC(6,2),
    total_fiber_loss_1550nm_db NUMERIC(6,2),
    total_splice_loss_db NUMERIC(5,2),
    total_connector_loss_db NUMERIC(5,2),
    total_splitter_loss_db NUMERIC(5,2),
    total_path_loss_db NUMERIC(6,2),
    path_status VARCHAR(20),
    validation_date DATE,
    geom GEOMETRY(LINESTRING, 4326)
);

-- Network Topology
CREATE TABLE IF NOT EXISTS fttx.network_topology (
    edge_id SERIAL PRIMARY KEY,
    source_node_id INTEGER,
    target_node_id INTEGER,
    edge_geom GEOMETRY(LINESTRING, 4326),
    cost NUMERIC(10,2),
    reverse_cost NUMERIC(10,2),
    edge_type VARCHAR(30)
);

-- ==========================================
-- INDEXES FOR PERFORMANCE
-- ==========================================

CREATE INDEX idx_fiber_cables_geom ON fttx.itu_fiber_cables USING GIST(geom);
CREATE INDEX idx_fiber_cables_status ON fttx.itu_fiber_cables(operational_status);
CREATE INDEX idx_fiber_cables_type ON fttx.itu_fiber_cables(cable_type, fiber_type);
CREATE INDEX idx_fiber_cables_route ON fttx.itu_fiber_cables(route_id);

CREATE INDEX idx_fiber_strands_cable ON fttx.itu_fiber_strands(cable_id);
CREATE INDEX idx_fiber_strands_status ON fttx.itu_fiber_strands(strand_status);
CREATE INDEX idx_fiber_strands_service ON fttx.itu_fiber_strands(service_assignment_id);

CREATE INDEX idx_odn_nodes_geom ON fttx.itu_odn_nodes USING GIST(geom);
CREATE INDEX idx_odn_nodes_type ON fttx.itu_odn_nodes(node_type, node_function);
CREATE INDEX idx_odn_nodes_parent ON fttx.itu_odn_nodes(parent_node_id);

CREATE INDEX idx_olt_equipment_geom ON fttx.itu_olt_equipment USING GIST(geom);
CREATE INDEX idx_olt_equipment_standard ON fttx.itu_olt_equipment(pon_standard, pon_class);

CREATE INDEX idx_onu_equipment_geom ON fttx.itu_onu_equipment USING GIST(geom);
CREATE INDEX idx_onu_equipment_olt ON fttx.itu_onu_equipment(connected_olt_id);
CREATE INDEX idx_onu_equipment_status ON fttx.itu_onu_equipment(onu_status);

CREATE INDEX idx_network_paths_olt ON fttx.itu_network_paths(olt_port_id);
CREATE INDEX idx_network_paths_onu ON fttx.itu_network_paths(onu_id);
CREATE INDEX idx_network_paths_geom ON fttx.itu_network_paths USING GIST(geom);

CREATE INDEX idx_cable_routes_geom ON fttx.itu_cable_routes USING GIST(geom);
CREATE INDEX idx_cable_routes_status ON fttx.itu_cable_routes(route_status);

CREATE INDEX idx_splice_points_geom ON fttx.itu_splice_points USING GIST(geom);

CREATE INDEX idx_network_topology_source ON fttx.network_topology(source_node_id);
CREATE INDEX idx_network_topology_target ON fttx.network_topology(target_node_id);
CREATE INDEX idx_network_topology_geom ON fttx.network_topology USING GIST(edge_geom);

-- ==========================================
-- COMMENTS FOR DOCUMENTATION
-- ==========================================

COMMENT ON SCHEMA fttx IS 'ITU-T compliant FTTX network planning database';
COMMENT ON TABLE fttx.itu_fiber_cables IS 'Fiber optic cables compliant with ITU-T L.40 specifications';
COMMENT ON TABLE fttx.itu_olt_equipment IS 'Optical Line Terminals compliant with ITU-T G.984/G.987/G.989';
COMMENT ON TABLE fttx.itu_onu_equipment IS 'Optical Network Units compliant with ITU-T GPON/XG-PON/NG-PON2 standards';
COMMENT ON TABLE fttx.itu_power_budget IS 'Power budget calculations per ITU-T G.984.2';

-- Grant permissions
GRANT USAGE ON SCHEMA fttx TO PUBLIC;
GRANT SELECT, INSERT, UPDATE, DELETE ON ALL TABLES IN SCHEMA fttx TO PUBLIC;
GRANT USAGE, SELECT ON ALL SEQUENCES IN SCHEMA fttx TO PUBLIC;

-- Success message
DO $$ 
BEGIN
    RAISE NOTICE '================================================';
    RAISE NOTICE 'ITU-T FTTX Database Schema Created Successfully';
    RAISE NOTICE '================================================';
    RAISE NOTICE 'Schema: fttx';
    RAISE NOTICE 'Tables created: %', (SELECT count(*) FROM information_schema.tables WHERE table_schema = 'fttx');
    RAISE NOTICE '';
    RAISE NOTICE 'Next steps:';
    RAISE NOTICE '1. Run 02_views_and_constraints.sql';
    RAISE NOTICE '2. Run 03_workflow_enforcement.sql';
    RAISE NOTICE '3. Install QGIS plugin';
    RAISE NOTICE '================================================';
END $$;